package com.liquidnet.service.account.funding.service.processor;

import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.account.common.ErrorConstants;
import com.liquidnet.service.account.common.FinConstants;
import com.liquidnet.service.account.funding.dto.FundingRegisterResult;
import com.liquidnet.service.account.funding.dto.base.FundingContextParam;
import com.liquidnet.service.account.funding.entity.FinAccount;
import com.liquidnet.service.account.funding.service.FinAccountService;
import com.liquidnet.service.account.funding.service.FundingProcessorService;
import com.liquidnet.service.account.util.FinUtil;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * FundingRegisterProcessService.class
 *
 * @author zhanggb
 * Created by IntelliJ IDEA at 2020/10/20
 */
@Service
public class FundingRegisterProcessor extends FundingProcessorService {
    private static final Logger log = LoggerFactory.getLogger(FundingRegisterProcessor.class);
    @Autowired
    private FinAccountService accountService;

    @Override
    public FundingRegisterResult checkInputParams(FundingContextParam context) {
        if (StringUtils.isEmpty(context.getFinId())) {
            return new FundingRegisterResult(ErrorConstants.PARAMS_VALUE_ERROR_CODE, "Invalid finId");
        }
        if (StringUtils.isEmpty(context.getAccType())) {
            return new FundingRegisterResult(ErrorConstants.PARAMS_VALUE_ERROR_CODE, "Invalid accType");
        }
        if (StringUtils.isEmpty(context.getAccGroupId())) {
            return new FundingRegisterResult(ErrorConstants.PARAMS_VALUE_ERROR_CODE, "Invalid accGroupId");
        }
        if (StringUtils.isEmpty(context.getCurrency())) {
            return new FundingRegisterResult(ErrorConstants.PARAMS_VALUE_ERROR_CODE, "Invalid tracingNo");
        }
        if (StringUtils.isNotEmpty(context.getAccAlias()) && context.getAccAlias().length() > 100) {
            return new FundingRegisterResult(ErrorConstants.PARAMS_VALUE_ERROR_CODE, "Toolong accAlias");
        }
        return new FundingRegisterResult();
    }

    @Override
    public FundingRegisterResult checkBussinessLogic(FundingContextParam context) {
        return new FundingRegisterResult();
    }

    @Override
    public FundingRegisterResult doBussiness(FundingContextParam context) {
        String accNo = FinUtil.getAccNo(context.getAccType(), context.getAccGroupId());

        FinAccount existedAccount = accountService.queryByAccNo(accNo);
        if (null != existedAccount) {
            log.warn("The account has been opened, please verify:[{},{},{}]", accNo, context.getAccGroupId(), context.getFinId());
            return new FundingRegisterResult(new String[]{existedAccount.getAccNo(), existedAccount.getAccGroupId()});
        }

        FinAccount addInfo = new FinAccount();
        addInfo.setAccNo(accNo);
        addInfo.setAccCurrency(context.getCurrency());
        addInfo.setAccAlias(context.getAccAlias());
        addInfo.setAccGroupId(context.getAccGroupId());
        addInfo.setAccType(context.getAccType());
        addInfo.setFinId(context.getFinId());
        addInfo.setAccStatus(FinConstants.AccStatus.NORMAL.getVal());
        addInfo.setCreateTime(context.getCreateTime());
        if (accountService.addAccount(addInfo) <= 0) {
            log.warn("Add data failed[Funding.account]:[{}]", JsonUtils.toJson(addInfo));
            return new FundingRegisterResult(ErrorConstants.ACCOUNT_CREATE_ERROR_CODE, ErrorConstants.ACCOUNT_CREATE_ERROR_DESC);
        }
        return new FundingRegisterResult(new String[]{addInfo.getAccNo(), addInfo.getAccGroupId()});
    }
}
