package com.liquidnet.service.goblin.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.liquidnet.common.exception.LiquidnetServiceException;
import com.liquidnet.commons.lang.constant.LnsRegex;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.IDCardUtil;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.goblin.service.IGoblinNftUserService;
import com.liquidnet.service.goblin.util.GoblinRedisUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.Pattern;

@Api(tags = "NFT:账号相关")
@Slf4j
@RestController
@RequestMapping("user")
public class GoblinNFTUserController {

    @Autowired
    IGoblinNftUserService goblinNftUserService;
    @Autowired
    GoblinRedisUtils goblinRedisUtils;

    @ApiOperationSupport(order = 11)
    @ApiOperation(value = "业务账号开通", notes = "返回`data`值标识：业务账号状态[1-已开通]")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "bizCode", value = "业务码[NFT_ZX]", allowableValues = "NFT_ZX"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "name", value = "姓名"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "idCard", value = "身份证号"),
    })
    @PostMapping(value = "open/account")
    public ResponseDto<Integer> openAccount(@Pattern(regexp = "\\b(NFT_ZX)\\b", message = "业务码无效")
                                            @RequestParam String bizCode,
                                            @RequestParam(required = true) String name,
                                            @RequestParam(required = true) String idCard) {
        if (!java.util.regex.Pattern.matches(LnsRegex.Valid.CN_HANZI, name)) {
            return ResponseDto.failure(ErrorMapping.get("140013"));
        }
        if (!java.util.regex.Pattern.matches(LnsRegex.Valid.CN_ID_CARD_REF, idCard)) {
            return ResponseDto.failure(ErrorMapping.get("140014"));
        }
        if (IDCardUtil.getAge(idCard) < 18) {
            return ResponseDto.failure(ErrorMapping.get("140010"));
        }

        String currentUid = CurrentUtil.getCurrentUid();
        String mobile = (String) CurrentUtil.getTokenClaims().get(CurrentUtil.TOKEN_MOBILE);
        try {
            return goblinNftUserService.openNftAccount(currentUid, bizCode, name, idCard, mobile);
        } catch (Exception e) {
            if (e instanceof LiquidnetServiceException) {
                LiquidnetServiceException lsEx = (LiquidnetServiceException) e;
                return ResponseDto.failure(lsEx.getCode(), lsEx.getMessage());
            }
            return ResponseDto.failure(ErrorMapping.get("140015"));
        }
    }

    @ApiOperationSupport(order = 11)
    @ApiOperation(value = "业务账号是否开通")
    @GetMapping(value = "open/account")
    public ResponseDto<Boolean> openAccount() {
        return ResponseDto.success(goblinRedisUtils.getNftNumAccount(CurrentUtil.getCurrentUid()));
    }
}
