package com.liquidnet.service.goblin.service.impl;

import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.DateUtil;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.goblin.dto.vo.*;
import com.liquidnet.service.goblin.service.manage.IGoblinMixAppService;
import com.liquidnet.service.goblin.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;

import static com.liquidnet.commons.lang.util.DateUtil.DTF_YMD_HMS;

/**
 * <p>
 * 商城-名单表 服务实现类
 * </p>
 *
 * @author liquidnet
 * @since 2022-04-29
 */
@Service
@Slf4j
public class GoblinMixAppServiceImpl implements IGoblinMixAppService {

    @Autowired
    GoblinRedisUtils redisUtils;

    @Override
    public ResponseDto<List<GoblinMixAppListVo>> mixList(Integer showPosition) {
        List<GoblinMixAppListVo> voList = ObjectUtil.goblinMixAppListVo();
        List<String> startList;
        List<String> stockList;
        if (showPosition == 1) {//商城
            startList = redisUtils.getGoblinMixSkuStartList();
            stockList = redisUtils.getGoblinMixSkuStockList();
        } else if (showPosition == 2) {//nft
            startList = redisUtils.getGoblinMixNftStartList();
            stockList = redisUtils.getGoblinMixNftStockList();
        } else {
            return ResponseDto.failure();
        }
        String nowTime = DateUtil.getNowTime();
        for (String mixId : startList) {
            GoblinMixDetailsVo vo = redisUtils.getMixDetails(mixId);
            GoblinMixAppListVo itemVo = GoblinMixAppListVo.getNew().copy(vo,nowTime);
            itemVo.setStatus(6);
            voList.add(itemVo);
        }

        for (String mixId : stockList) {
            GoblinMixDetailsVo vo = redisUtils.getMixDetails(mixId);
            GoblinMixAppListVo itemVo = GoblinMixAppListVo.getNew().copy(vo,nowTime);
            itemVo.setStatus(8);
            voList.add(itemVo);
        }

        return ResponseDto.success(voList);
    }

    @Override
    public ResponseDto<GoblinMixAppDetailsVo> mixDetails(String mixId) {
        GoblinMixDetailsVo baseVo = redisUtils.getMixDetails(mixId);
        if(baseVo==null){
            return ResponseDto.failure("参数异常");
        }
        GoblinStoreInfoVo storeInfoVo = redisUtils.getStoreInfoVo(baseVo.getStoreId());
        GoblinMixAppDetailsVo vo = GoblinMixAppDetailsVo.getNew().copy(baseVo, storeInfoVo.getStoreName());
        int stock = 0;
        for (GoblinMixDetailsItemVo item : vo.getItem()) {
            GoblinGoodsSkuInfoVo skuInfoVo = redisUtils.getGoodsSkuInfoVo(item.getSkuId());
            GoblinGoodsInfoVo spuInfoVo = redisUtils.getGoodsInfoVo(skuInfoVo.getSpuId());
            item.setSkuPic(skuInfoVo.getSkuPic());
            item.setStock(0);
            item.setSkuName(skuInfoVo.getName());
            item.setSpuName(spuInfoVo.getName());
            if (skuInfoVo.getUnbox().equals("1")) {
                List<String> relationIds = redisUtils.getGoblinMixRelationBox(item.getSkuId(), vo.getMixId());
                for (String skuId : relationIds) {
                    stock += redisUtils.getSkuStock(vo.getMixId(), skuId);
                }
            } else {
                stock += redisUtils.getSkuStock(vo.getMixId(), item.getSkuId());
            }
        }
        if (stock == 0) {
            vo.setStatus(8);
        } else {
            LocalDateTime nt = LocalDateTime.now();
            LocalDateTime st = LocalDateTime.parse(vo.getTimeStart(), DTF_YMD_HMS);
            LocalDateTime et = LocalDateTime.parse(vo.getTimeEnd(), DTF_YMD_HMS);
            if (nt.isBefore(st)) {
                vo.setStatus(9);
            } else if (nt.isAfter(et)) {
                vo.setStatus(10);
            } else {
                vo.setStatus(6);
            }
        }
        return ResponseDto.success(vo);
    }

    @Override
    public ResponseDto<List<GoblinMixDetailsItemVo>> getItem(String mixId) {
        GoblinMixDetailsVo baseVo = redisUtils.getMixDetails(mixId);
        for (GoblinMixDetailsItemVo item : baseVo.getItem()) {
            GoblinGoodsSkuInfoVo skuInfoVo = redisUtils.getGoodsSkuInfoVo(item.getSkuId());
            GoblinGoodsInfoVo spuInfoVo = redisUtils.getGoodsInfoVo(skuInfoVo.getSpuId());
            item.setStock(0);
            item.setSkuPic(skuInfoVo.getSkuPic());
            item.setSkuName(skuInfoVo.getName());
            item.setSpuName(spuInfoVo.getName());
        }
        return ResponseDto.success(baseVo.getItem());
    }

    @Override
    public ResponseDto<GoblinMixOrderListVo> getMixList(String masterCode) {
        HashMap<String, String[]> map = redisUtils.getMixMasterCode(masterCode);
        String[] nft = map.get("nft");
        String[] goblin = map.get("goblin");

        GoblinMixOrderListVo vo = GoblinMixOrderListVo.getNew();
        List<GoblinAppOrderListVo> orderListVos = ObjectUtil.goblinAppOrderListVoArrayList();
        List<GoblinNftOrderDetailsVo> nftOrderListVos = ObjectUtil.goblinNftOrderDetailsVoArrayList();

        for (String orderId : nft) {
            if ("".equals(orderId)) {
                break;
            }
            // 订单信息
            GoblinNftOrderVo orderVo = redisUtils.getGoblinNftOrder(orderId);
            GoblinNftOrderDetailsVo orderDetails = GoblinNftOrderDetailsVo.getNew().copy(orderVo);
            // 商品信息
            GoblinGoodsSkuInfoVo goodsSkuInfoVo = redisUtils.getGoodsSkuInfoVo(orderVo.getSkuId());
            GoblinNftOrderGoodsSkuInfoVo nftGoodsSkuInfoVo = GoblinNftOrderGoodsSkuInfoVo.getNew().copy(goodsSkuInfoVo);
            orderDetails.setGoodsSkuInfoVo(nftGoodsSkuInfoVo);
            nftOrderListVos.add(orderDetails);
        }
        for (String orderId : goblin) {
            if ("".equals(orderId)) {
                break;
            }
            GoblinStoreOrderVo orderVo = redisUtils.getGoblinOrder(orderId);
            GoblinAppOrderListVo vos = GoblinAppOrderListVo.getNew();
            BeanUtils.copyProperties(orderVo, vos);
            vos.setRestTime(0L);
            List<GoblinOrderSkuVo> skuVos = ObjectUtil.getGoblinOrderSkuVoArrayList();
            for (String orderSkuId : orderVo.getOrderSkuVoIds()) {
                GoblinOrderSkuVo orderSkuVo = redisUtils.getGoblinOrderSkuVo(orderSkuId);
                skuVos.add(orderSkuVo);
            }
            vos.setOrderSkuVos(skuVos);
            orderListVos.add(vos);
        }
        vo.setGoblinOrder(orderListVos);
        vo.setNftOrder(nftOrderListVos);
        return ResponseDto.success(vo);
    }

    @Override
    public ResponseDto<Boolean> canBy(String mixId, String uid) {
        String mobile = StringUtils.defaultString(((String) CurrentUtil.getTokenClaims().get(CurrentUtil.TOKEN_MOBILE)), "");
        GoblinMixDetailsVo baseVo = redisUtils.getMixDetails(mixId);
        int whiteType = baseVo.getWhiteType();
        if (whiteType != 2) {
            return ResponseDto.success(redisUtils.getListCanBuy(mixId, mixId, mobile, uid, whiteType));
        } else {
            return ResponseDto.success(true);
        }
    }
}
