package com.liquidnet.service.sweet.service.impl;

import cn.binarywang.wx.miniapp.api.WxMaService;
import cn.binarywang.wx.miniapp.bean.WxMaSubscribeMessage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.DateUtil;
import com.liquidnet.commons.lang.util.IDGenerator;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.base.SqlMapping;
import com.liquidnet.service.base.constant.MQConst;
import com.liquidnet.service.feign.kylin.api.FeignKylinPerformancesClient;
import com.liquidnet.service.kylin.dto.vo.mongo.KylinPerformanceVo;
import com.liquidnet.service.sweet.config.WechatMaConfigure;
import com.liquidnet.service.sweet.dto.SweetAppletSubMsgOfTypeDto;
import com.liquidnet.service.sweet.entity.SweetAppletSubMsg;
import com.liquidnet.service.sweet.mapper.SweetAppletSubMsgMapper;
import com.liquidnet.service.sweet.service.ISweetAppletSubMsgService;
import com.liquidnet.service.sweet.utils.QueueUtils;
import com.liquidnet.service.sweet.utils.RedisActivityUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

/**
 * <p>
 * 小程序订阅消息记录表 服务实现类
 * </p>
 *
 * @author jiangxiulong
 * @since 2021-11-22
 */
@Service
@Slf4j
public class SweetAppletSubMsgServiceImpl extends ServiceImpl<SweetAppletSubMsgMapper, SweetAppletSubMsg> implements ISweetAppletSubMsgService {

    @Autowired
    private SweetAppletSubMsgMapper subMsgMapper;

    @Autowired
    private FeignKylinPerformancesClient feignKylinPerformancesClient;

    @Autowired
    WechatMaConfigure wechatMaConfigure;

    @Autowired
    private QueueUtils queueUtils;

    @Autowired
    private RedisActivityUtils redisActivityUtils;

    @Override
    public ResponseDto sendMsg(Integer type) {
        List<SweetAppletSubMsgOfTypeDto> targetIds = subMsgMapper.selectListOfType(type);

        for (SweetAppletSubMsgOfTypeDto targetIdInfo : targetIds) {
            String targetId = targetIdInfo.getTargetId();
            ResponseDto<KylinPerformanceVo> kylinPerformanceVo = feignKylinPerformancesClient.detail(targetId, 0, 0, "");
            KylinPerformanceVo performanceVoData = kylinPerformanceVo.getData();
            if (null == performanceVoData || ObjectUtils.isEmpty(performanceVoData)) {
                log.info("无演出数据：[performancesId={}]", targetId);
                continue;
            }
            Integer isPush = isPush(performanceVoData, targetId);
            if (isPush != 3) {
                log.info("当前演出还不能推送：[状态={}]", isPush);
                continue;
            }

            List<SweetAppletSubMsg> msgList = subMsgMapper.selectList(
                    Wrappers.lambdaQuery(SweetAppletSubMsg.class)
                            .eq(SweetAppletSubMsg::getIsPush, 1)
                            .eq(SweetAppletSubMsg::getActivityType, type)
                            .eq(SweetAppletSubMsg::getTargetId, targetId)
            );

            String title = performanceVoData.getTitle();
            String timeStart = performanceVoData.getTimeStart();
            ArrayList<String> msgIdList = CollectionUtil.arrayListString();
            if (!CollectionUtils.isEmpty(msgList)) {
                for (SweetAppletSubMsg info : msgList) {
                    try {
                        String msgId = info.getMsgId();
                        // 发送订阅消息接口
                        boolean subMessage = sendSubMessage(info, targetId, title, timeStart);
                        if (subMessage) {
                            msgIdList.add(msgId);
                        }
                    } catch (Exception e) {
                        log.error("小程序演出订阅提醒消息处理异常", e);
                    }
                }
                if (!CollectionUtils.isEmpty(msgIdList)) {
                    SweetAppletSubMsg update = new SweetAppletSubMsg();
                    update.setIsPush(2);
                    update.setUpdatedAt(LocalDateTime.now());
                    subMsgMapper.update(
                            update,
                            Wrappers.lambdaUpdate(SweetAppletSubMsg.class)
                                    .in(SweetAppletSubMsg::getMsgId, msgIdList)
                    );
                }
            }
        }
        return ResponseDto.success();
    }

    @Override
    public ResponseDto sendOfActivityType(Integer activityType, String targetId, Integer timeType) {
        List<SweetAppletSubMsg> msgList = subMsgMapper.selectList(
                Wrappers.lambdaQuery(SweetAppletSubMsg.class)
                        .eq(SweetAppletSubMsg::getIsPush, 1)
                        .eq(SweetAppletSubMsg::getActivityType, activityType)
                        .eq(SweetAppletSubMsg::getTargetId, targetId)
        );

        KylinPerformanceVo performanceVoData = null;
        if (!targetId.isEmpty()) {
            ResponseDto<KylinPerformanceVo> kylinPerformanceVo = feignKylinPerformancesClient.detail(targetId, 0, 0, "");
            performanceVoData = kylinPerformanceVo.getData();
            if (null == performanceVoData || ObjectUtils.isEmpty(performanceVoData)) {
                log.info("无演出数据：[performancesId={}]", targetId);
                return ResponseDto.failure("无演出数据");
            }
            if (timeType == 1) {
                Integer isPush = isPush(performanceVoData, targetId);
                if (isPush != 3) {
                    return ResponseDto.failure("当前演出还不能推送，状态".concat(String.valueOf(isPush)));
                }
            }
        }

        String title = performanceVoData.getTitle();
        String timeStart = performanceVoData.getTimeStart();
        ArrayList<String> msgIdList = CollectionUtil.arrayListString();
        if (!CollectionUtils.isEmpty(msgList)) {
            for (SweetAppletSubMsg info : msgList) {
                try {
                    String msgId = info.getMsgId();
                    // 发送订阅消息接口
                    boolean subMessage = sendSubMessage(info, targetId, title, timeStart);
                    if (subMessage) {
                        redisActivityUtils.delSubscribe(info.getOpenId(), targetId);
                        msgIdList.add(msgId);
                    }
                } catch (Exception e) {
                    log.error("小程序演出订阅提醒消息处理异常", e);
                }
            }
            if (!CollectionUtils.isEmpty(msgIdList)) {
                SweetAppletSubMsg update = new SweetAppletSubMsg();
                update.setIsPush(2);
                update.setTargetId(targetId);
                update.setUpdatedAt(LocalDateTime.now());
                subMsgMapper.update(
                        update,
                        Wrappers.lambdaUpdate(SweetAppletSubMsg.class)
                                .in(SweetAppletSubMsg::getMsgId, msgIdList)
                );
            }
        }
        return ResponseDto.success();
    }

    @Override
    public ResponseDto sendOfMid(String midList, String targetId, Integer timeType) {
        String[] midListArray = midList.split(",");
        List<SweetAppletSubMsg> msgList = subMsgMapper.selectList(
                Wrappers.lambdaQuery(SweetAppletSubMsg.class)
                        .eq(SweetAppletSubMsg::getIsPush, 1)
                        .in(SweetAppletSubMsg::getMid, midListArray)
        );

        KylinPerformanceVo performanceVoData = null;
        if (!targetId.isEmpty()) {
            ResponseDto<KylinPerformanceVo> kylinPerformanceVo = feignKylinPerformancesClient.detail(targetId, 0, 0, "");
            performanceVoData = kylinPerformanceVo.getData();
            if (null == performanceVoData || ObjectUtils.isEmpty(performanceVoData)) {
                log.info("无演出数据：[performancesId={}]", targetId);
                return ResponseDto.failure("无演出数据");
            }
            if (timeType == 1) {
                Integer isPush = isPush(performanceVoData, targetId);
                if (isPush != 3) {
                    return ResponseDto.failure("当前演出还不能推送，状态".concat(String.valueOf(isPush)));
                }
            }
        }

        String title = performanceVoData.getTitle();
        String timeStart = performanceVoData.getTimeStart();
        ArrayList<String> msgIdList = CollectionUtil.arrayListString();
        if (!CollectionUtils.isEmpty(msgList)) {
            for (SweetAppletSubMsg info : msgList) {
                try {
                    String msgId = info.getMsgId();
                    // 发送订阅消息接口
                    boolean subMessage = sendSubMessage(info, targetId, title, timeStart);
                    if (subMessage) {
                        redisActivityUtils.delSubscribe(info.getOpenId(), targetId);
                        msgIdList.add(msgId);
                    }
                } catch (Exception e) {
                    log.error("小程序演出订阅提醒消息处理异常", e);
                }
            }
            if (!CollectionUtils.isEmpty(msgIdList)) {
                SweetAppletSubMsg update = new SweetAppletSubMsg();
                update.setIsPush(2);
                update.setTargetId(targetId);
                update.setUpdatedAt(LocalDateTime.now());
                subMsgMapper.update(
                        update,
                        Wrappers.lambdaUpdate(SweetAppletSubMsg.class)
                                .in(SweetAppletSubMsg::getMsgId, msgIdList)
                );
            }
        }
        return ResponseDto.success();
    }

    @Override
    public ResponseDto<Boolean> create(String openId, String unionId, String templateId, String targetId, Integer appletType, Integer activityType) {
        String[] templateIdArray = templateId.split(",");

        LinkedList<String> sqls = CollectionUtil.linkedListString();
        LinkedList<Object[]> sqlsDataA = CollectionUtil.linkedListObjectArr();
        sqls.add(SqlMapping.get("sweet_applet_sub_msg.insert"));
        for (String id : templateIdArray) {
            sqlsDataA.add(new Object[]{
                    IDGenerator.nextSnowId(), openId, unionId, id, targetId, appletType, activityType
            });
        }
        queueUtils.sendMsgByRedis(MQConst.SweetQueue.SWEET_REMIND_INSERT.getKey(),
                SqlMapping.gets(sqls, sqlsDataA));

        redisActivityUtils.setSubscribe(openId, targetId);

        return ResponseDto.success();
    }

    @Override
    public ResponseDto<Integer> isSubPerform(String openId, String targetId) {
        Integer subscribe = redisActivityUtils.getSubscribe(openId, targetId);
        return ResponseDto.success(subscribe);
    }

    /**
     * 微信小程序推送订阅消息
     */
    private boolean sendSubMessage(SweetAppletSubMsg info, String performancesId, String title, String timeStart) {

        WxMaSubscribeMessage subscribeMessage = new WxMaSubscribeMessage();

        subscribeMessage.setPage("pages/webview?query=showdetails&id=".concat(performancesId));
        subscribeMessage.setTemplateId(info.getTemplateId());
        subscribeMessage.setToUser(info.getOpenId());

        ArrayList<WxMaSubscribeMessage.MsgData> wxMaSubscribeData = new ArrayList<>();
        WxMaSubscribeMessage.MsgData wxMaSubscribeData1 = new WxMaSubscribeMessage.MsgData();
        wxMaSubscribeData1.setName("thing1");
        wxMaSubscribeData1.setValue(title);
        //每个参数 存放到大集合中
        wxMaSubscribeData.add(wxMaSubscribeData1);

        // 第二个内容：用户昵称
        WxMaSubscribeMessage.MsgData wxMaSubscribeData2 = new WxMaSubscribeMessage.MsgData();
        wxMaSubscribeData2.setName("time2");
        wxMaSubscribeData2.setValue(timeStart);
        wxMaSubscribeData.add(wxMaSubscribeData2);

        // 第三个内容：领取方式
        WxMaSubscribeMessage.MsgData wxMaSubscribeData3 = new WxMaSubscribeMessage.MsgData();
        wxMaSubscribeData3.setName("thing3");
        wxMaSubscribeData3.setValue("您关注的演出活动门票即将开售，请准备购票");
        wxMaSubscribeData.add(wxMaSubscribeData3);

        subscribeMessage.setData(wxMaSubscribeData);
        try {
            //获取微信小程序配置：
            WxMaService wxService = wechatMaConfigure.getWxMaService(info.getAppletType());
            //进行推送
            wxService.getMsgService().sendSubscribeMsg(subscribeMessage);
            return true;
        } catch (Exception e) {
            log.error("sendSubMessageException e{}", e);
        }
        return false;
    }

    /**
     * 判断演出时间 是否可以推送
     */
    private Integer isPush(KylinPerformanceVo performanceVoData, String targetId) {
        LocalDateTime nowTime = LocalDateTime.now();
        String nowTimeStr = DateUtil.Formatter.yyyyMMddHHmmss.format(nowTime);
        String sellTime = performanceVoData.getSellTime();
        LocalDateTime sellTimeLocal = LocalDateTime.parse(sellTime, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        LocalDateTime sellTimeLocalNew = sellTimeLocal.minusMinutes(10);
        String sellTimeLocalNewStr = DateUtil.Formatter.yyyyMMddHHmmss.format(sellTimeLocalNew);

        String stopSellTime = performanceVoData.getStopSellTime();
        if (1 == DateUtil.compareStrDay(nowTimeStr, stopSellTime)) { // 超过售卖期 不推 删redis
            log.info("超过售卖期：[nowTimeStr={}, stopSellTime={}, performancesId={}]", nowTimeStr, stopSellTime, targetId);
            return 1;
        }
        if (-1 == DateUtil.compareStrDay(nowTimeStr, sellTimeLocalNewStr)) { // 还没到售卖期
            log.info("还没到售卖期：[nowTimeStr={}, sellTimeLocalNewStr={}, performancesId={}]", nowTimeStr, sellTimeLocalNewStr, targetId);
            return 2;
        }
        return 3;
    }
}
