package com.liquidnet.service.adam.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.service.adam.dto.vo.AdamMemberSimpleVo;
import com.liquidnet.service.adam.dto.vo.AdamMemberVo;
import com.liquidnet.service.adam.dto.vo.AdamUserInfoVo;
import com.liquidnet.service.adam.dto.vo.AdamUserMemberVo;
import com.liquidnet.service.adam.service.AdamRdmService;
import com.liquidnet.service.adam.service.IAdamUserMemberService;
import com.liquidnet.service.base.ResponseDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotBlank;
import java.util.ArrayList;
import java.util.List;

@ApiSupport(order = 10030)
@Api(tags = "会员信息")
@Slf4j
@Validated
@RestController
@RequestMapping("member")
public class AdamMemberController {
    @Autowired
    AdamRdmService adamRdmService;
    @Autowired
    IAdamUserMemberService adamUserMemberService;

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "会员信息")
    @GetMapping("info")
    public ResponseDto<AdamMemberVo> query() {
        AdamMemberSimpleVo memberSimpleVo = adamRdmService.getMemberSimpleVo();
        AdamMemberVo memberVo = null;
        if (null != memberSimpleVo && null != (memberVo = adamRdmService.getMemberVoByMemberId(memberSimpleVo.getMemberId()))) {
            memberVo.setPriceVoList(adamRdmService.getMemberPriceVoByMemberId(memberVo.getMemberId()));

            memberVo.setRightsVoList(adamRdmService.getMemberRightsVoByMemberId(memberVo.getMemberId()));

            String currentUid = CurrentUtil.getCurrentUid();
            if (!StringUtils.isEmpty(currentUid)) {
                memberVo.setUserMemberVo(adamRdmService.getUserMemberVoByUid(currentUid));
            }
        }
        return ResponseDto.success(memberVo);
    }

    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "会员协议")
    @GetMapping("agreement")
    public ResponseDto<String> agreement() {
        return ResponseDto.success(adamRdmService.getMemberAgreement());
    }

    @ApiOperationSupport(order = 10)
    @ApiOperation(value = "会员黑名单UID")
    @PostMapping("blk/lis")
    public ResponseDto<List<String>> listBlack() {
        return ResponseDto.success(adamRdmService.getBlacklistForMember());
    }

    @ApiOperationSupport(order = 11)
    @ApiOperation(value = "添加会员黑名单")
    @PostMapping("blk/add")
    public ResponseDto<List<String>> addBlack(@NotBlank @RequestParam String opts) {
        String[] uidArr = opts.split(",");
        List<String> invalidList = new ArrayList<>(), existList = new ArrayList<>();
        for (String uid : uidArr) {
            AdamUserInfoVo vo = adamRdmService.getUserInfoVoByUid(uid);
            if (null != vo) {
                existList.add(uid);
            } else {
                invalidList.add(uid);
            }
        }
        List<String> blacklist = adamRdmService.getBlacklistForMember();
        if (CollectionUtils.isEmpty(blacklist)) {
            blacklist = existList;
        } else {
            blacklist.addAll(existList);
        }
        if (!CollectionUtils.isEmpty(blacklist)) {
            adamRdmService.setBlacklistForMember(blacklist);
        } else {
            log.warn("###blacklist.add:failed,invalidList:{},existList:{},opts:{}", invalidList.size(), existList.size(), opts);
        }
        return ResponseDto.success(invalidList);
    }

    @ApiOperationSupport(order = 12)
    @ApiOperation(value = "删除会员黑名单")
    @PostMapping("blk/del")
    public ResponseDto<Object> delBlack(@NotBlank @RequestParam String opts) {
        String[] uidArr = opts.split(",");
        List<String> blacklist = adamRdmService.getBlacklistForMember();
        if (!CollectionUtils.isEmpty(blacklist)) {
            for (String uid : uidArr) {
                blacklist.remove(uid);
            }
            adamRdmService.setBlacklistForMember(blacklist);
        }
        return ResponseDto.success();
    }

    /* ------------------------------------------------------------｜API */

    @ApiOperationSupport(order = 99)
    @ApiOperation(value = "@API:是否会员", notes = "根据UID判断用户是否会员，响应参数[`data`:true-是,false-否]")
    @GetMapping("check/depth")
    public ResponseDto<Boolean> isMember(@NotBlank @RequestParam String uid) {
        AdamUserMemberVo userMemberInfo = adamRdmService.getUserMemberVoByUid(uid);
        return ResponseDto.success(null != userMemberInfo && userMemberInfo.isActive());
    }
}
