package com.liquidnet.service.adam.controller.web.account;
import com.liquidnet.common.exception.constant.ErrorCode;
import com.liquidnet.common.swagger.config.SwaggerApiVersion;
import com.liquidnet.service.ResponseDto;
import com.liquidnet.service.account.wallet.dto.WalletQueryParam;
import com.liquidnet.service.account.wallet.vo.WalletAccountInfoVo;
import com.liquidnet.service.adam.config.swagger.AdamSwaggerApiVersionConstant;
import com.liquidnet.service.adam.dto.base.AdamCurrentUser;
import com.liquidnet.service.adam.dto.base.AdamResultDto;
import com.liquidnet.service.adam.entity.AdamWalletAccess;
import com.liquidnet.service.adam.interceptor.annotation.RequiresPermissions;
import com.liquidnet.service.adam.service.IAdamWalletAccessService;
import com.liquidnet.service.adam.service.feign.fin.IAdamFeignAccountService;
import com.liquidnet.service.adam.util.CurrentUserUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 * 账户钱包 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2020-10-19
 */
@Api(tags = "web balance")
@Slf4j
@RestController
@RequestMapping("/adam-account-balance")
public class AdamAccountBalanceController {

    @Autowired
    IAdamFeignAccountService adamFeignAccountService;
    @Autowired
    IAdamWalletAccessService adamWalletAccessService;

    @RequiresPermissions("4F15121D")
    @ApiOperation(value = "list balances")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_1)
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "query", dataType = "String", name = "currency", value = "币种,条件模糊搜索")
    })
    @GetMapping("/list")
    public ResponseDto<Map<String, BigDecimal>> list(HttpServletRequest request,
                                                     @RequestParam(required = false) String currency) {
        try {
            AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);

            // Filter user allowed access wallets
            List<AdamWalletAccess> allowedWalletList = adamWalletAccessService.queryByUserId(currentUser.getId());

            WalletQueryParam param = new WalletQueryParam();
            param.setUserExtId(currentUser.getComId());
            param.setWalletNoList(allowedWalletList.stream().map(AdamWalletAccess::getFinWalletNo).collect(Collectors.toList()));
            param.setFuzzySearch(currency);
            param.setFuzzyField(1);

            AdamResultDto<List<WalletAccountInfoVo>> resultDto = adamFeignAccountService.list(param);

            if (!resultDto.isSuccess()) {
                return ResponseDto.failure(resultDto.getCode(), resultDto.getMessage());
            }
            List<WalletAccountInfoVo> list = resultDto.getData();
            Map<String, BigDecimal> collect = list.stream().collect(Collectors.groupingBy(WalletAccountInfoVo::getCurrency,
                    Collectors.reducing(BigDecimal.ZERO,
                            WalletAccountInfoVo::getBalanceAvailable,
                            BigDecimal::add)));
            return ResponseDto.success(collect);
        } catch (Exception e) {
            log.error("list balances error", e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

}
