package com.liquidnet.service.adam.controller.web.payer;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.google.common.collect.Lists;
import com.liquidnet.common.exception.constant.ErrorCode;
import com.liquidnet.common.swagger.config.SwaggerApiVersion;
import com.liquidnet.service.ResponseDto;
import com.liquidnet.service.adam.config.swagger.AdamSwaggerApiVersionConstant;
import com.liquidnet.service.adam.constant.AdamPayerConstants;
import com.liquidnet.service.adam.dto.base.AdamCurrentUser;
import com.liquidnet.service.adam.interceptor.annotation.RequiresPermissions;
import com.liquidnet.service.adam.vo.AdamPayerEnumVo;
import com.liquidnet.service.adam.dto.query.AdamPayerQuery;
import com.liquidnet.service.adam.dto.base.AdamResultDto;
import com.liquidnet.service.adam.entity.AdamPayer;
import com.liquidnet.service.adam.interceptor.annotation.NoAuth;
import com.liquidnet.service.adam.service.IAdamPayerService;
import com.liquidnet.service.adam.util.CurrentUserUtil;
import com.liquidnet.service.adam.util.EnumUtil;
import com.liquidnet.service.adam.vo.AdamPayerVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.List;

/**
 * <p>
 * 付款人信息 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2020-11-02
 */
@Api(tags = "web payer")
@Slf4j
@RestController
@RequestMapping("/adam-payer")
public class AdamPayerController {

    @Autowired
    private IAdamPayerService adamPayerService;

    @NoAuth
    @ApiOperation(value = "enum list")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_1)
    @GetMapping("/enum_list")
    public ResponseDto<AdamPayerEnumVo> enumList() {
        AdamPayerEnumVo vo = new AdamPayerEnumVo();
        // 公司类型
        vo.setIndividualIdTypeList(EnumUtil.getList(AdamPayerConstants.IndividualIdTypeEnum.values()));
        return ResponseDto.success(vo);
    }

    @ApiOperation(value = "get")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_1)
    @GetMapping("/get")
    public ResponseDto get(HttpServletRequest request,
                           @RequestParam String id) {
        try {
            AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);
            AdamResultDto<AdamPayer> adamResultDto = adamPayerService.select(id, currentUser.getComId());
            ResponseDto responseDto = AdamResultDto.getResponseDto(adamResultDto);
            return responseDto;
        } catch (Exception e) {
            log.error("get payer.error", e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    @ApiOperation(value = "list")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_1)
    @GetMapping("/list")
    public ResponseDto list(HttpServletRequest request) {
        try {
            AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);
            AdamPayerQuery selectParam = new AdamPayerQuery();
            selectParam.setComId(currentUser.getComId());
            selectParam.setStatus(AdamPayerConstants.StatusEnum.NORMAL.getCode());
            List<AdamPayer> list = adamPayerService.selectList(selectParam);
            return ResponseDto.success(list);
        } catch (Exception e) {
            log.error("list payer.error", e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    @RequiresPermissions("A1059625")
    @ApiOperation(value = "page")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_1)
    @GetMapping("/page")
    public ResponseDto pageList(HttpServletRequest request,
                                @RequestParam(required = false) String search,
                                @RequestParam(required = false, value = "pageNum", defaultValue = "1") Integer pageNum,
                                @RequestParam(required = false, value = "pageSize", defaultValue = "10") Integer pageSize) {
        try {
            AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);
            String comId = currentUser.getComId();
            AdamPayerQuery selectParam = new AdamPayerQuery();
            selectParam.setComId(comId);
            selectParam.setStatus(AdamPayerConstants.StatusEnum.NORMAL.getCode());
            selectParam.setSearch(search);
            Page<AdamPayer> adamPayerPage = adamPayerService.selectPage(pageNum, pageSize, selectParam);

            List<AdamPayerVo> recordsDisplay = Lists.newArrayList();
            adamPayerPage.getRecords().forEach(r -> {
                AdamPayerVo vo = new AdamPayerVo();
                vo.conversion(r);
                recordsDisplay.add(vo);
            });

            Page<AdamPayerVo> page = new Page<>();
            BeanUtils.copyProperties(adamPayerPage, page);
            page.setRecords(recordsDisplay);
            return ResponseDto.success(page);

        } catch (Exception e) {
            log.error("page payer.error", e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    @RequiresPermissions("A040E8E8")
    @ApiOperation(value = "create")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_1)
    @PostMapping("/create")
    public ResponseDto create(HttpServletRequest request,
                              @ModelAttribute AdamPayer param) {
        String logFx = "create payer.";
        try {
            AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);

            log.info(logFx + "comId=[{}],AdamPayer=[{}]", currentUser.getComId(), JSON.toJSONString(param, SerializerFeature.WriteMapNullValue));
            param.setComId(currentUser.getComId());
            AdamResultDto adamResultDto = adamPayerService.create(param);
            ResponseDto responseDto = AdamResultDto.getResponseDto(adamResultDto);
            return responseDto;
        } catch (Exception e) {
            log.error(logFx + "error", e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    @RequiresPermissions("A040E8E8")
    @ApiOperation(value = "edit")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_1)
    @PostMapping("/edit")
    public ResponseDto edit(HttpServletRequest request,
                            @ModelAttribute AdamPayer param) {
        String logFx = "edit payer.";

        try {
            AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);
            log.info(logFx + "comId=[{}],AdamPayer=[{}]", currentUser.getComId(), JSON.toJSONString(param, SerializerFeature.WriteMapNullValue));
            param.setComId(currentUser.getComId());
            AdamResultDto adamResultDto = adamPayerService.update(param);
            ResponseDto responseDto = AdamResultDto.getResponseDto(adamResultDto);
            return responseDto;
        } catch (Exception e) {
            log.error(logFx + "error", e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    @RequiresPermissions("A040E8E8")
    @ApiOperation(value = "delete")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_1)
    @PostMapping("/delete")
    public ResponseDto delete(HttpServletRequest request,
                              @RequestParam String id) {
        String logFx = "delete payer.";

        try {
            AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);

            log.info(logFx + "comId=[{}],id=[{}]", currentUser.getComId(), id);
            AdamResultDto adamResultDto = adamPayerService.delete(id, currentUser.getComId());
            ResponseDto responseDto = AdamResultDto.getResponseDto(adamResultDto);
            return responseDto;
        } catch (Exception e) {
            log.error(logFx + "error", e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

}
