package com.liquidnet.service.adam.controller.web.user;

import com.liquidnet.common.exception.constant.ErrorCode;
import com.liquidnet.common.swagger.config.SwaggerApiVersion;
import com.liquidnet.service.ResponseDto;
import com.liquidnet.service.adam.config.swagger.AdamSwaggerApiVersionConstant;
import com.liquidnet.service.adam.dto.base.AdamCurrentUser;
import com.liquidnet.service.adam.dto.base.AdamResultDto;
import com.liquidnet.service.adam.interceptor.annotation.NoAuth;
import com.liquidnet.service.adam.interceptor.annotation.RequiresPermissions;
import com.liquidnet.service.adam.service.IAdamUserEmailService;
import com.liquidnet.service.adam.util.CurrentUserUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.time.LocalDateTime;

@Slf4j
@Api(tags = "web user email")
@RestController
@RequestMapping("/adam-user-email")
public class AdamUserEmailController {

    @Autowired
    private IAdamUserEmailService adamUserEmailService;

    /**
     * 邮箱重新发送邮件
     */
    @NoAuth
    @ApiOperation(value = "resend email")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_0)
    @ApiImplicitParams({
            @ApiImplicitParam(type = "query", dataType = "String", name = "email", value = "email")
    })
    @GetMapping("/resend")
    public ResponseDto resend(String email) {
        log.info("[RESEND EMAIL]; current time : {}", LocalDateTime.now());
        try {
            AdamResultDto<Object> dto = adamUserEmailService.resendEmail(email);
            return AdamResultDto.getResponseDto(dto);
        } catch (Exception e) {
            log.error("[RESEND EMAIL]; catch an exception, {} current time : {}", e.getMessage(), LocalDateTime.now(), e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    /**
     * 用户注册后,邮箱中的URL链接点击确认邮箱认证
     */
    @NoAuth
    @ApiOperation(value = "user confirm email")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_0)
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "confirmKey", value = "tag for confirm email")
    })
    @PostMapping("/confirm")
    public ResponseDto confirm(@RequestParam String confirmKey) {
        String logFx = "user confirm email,confirmKey=[" + confirmKey + "].";
        log.info(logFx);
        try {
            AdamResultDto<Object> dto = adamUserEmailService.confirmEmail(confirmKey);
            return AdamResultDto.getResponseDto(dto);
        } catch (Exception e) {
            log.error(logFx + "execption=[{}]", e.getMessage(), e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    /**
     * 邮箱更新,发送验证码
     */
    @RequiresPermissions("FE3FC5E2")
    @ApiOperation(value = "email update verify code send")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_2)
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "newEmail", value = "new email address"),
            @ApiImplicitParam(type = "form", dataType = "String", name = "pw", value = "password")
    })
    @PostMapping("/update_verify_code_send")
    public ResponseDto updateVerifyCodeSend(HttpServletRequest request, @RequestParam String newEmail, @RequestParam String pw) {
        AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);
        String logFx = "email update verify code send,userEmail=[" + currentUser.getEmail() + "].";
        log.info(logFx + "newEmail=[{}]", newEmail);
        try {
            AdamResultDto adamResultDto = adamUserEmailService.updateVerifyCodeSend(currentUser.getId(), newEmail, pw);
            return AdamResultDto.getResponseDto(adamResultDto);
        } catch (Exception e) {
            log.error(logFx + "execption=[{}]", e.getMessage(), e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    /**
     * 邮箱更新,确认验证码
     */
    @RequiresPermissions("FE3FC5E2")
    @ApiOperation(value = "email update verify code confirm")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_2)
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "verifyCode", value = "verify code")
    })
    @PostMapping("/update_verify_code_confirm")
    public ResponseDto updateVerifyCodeConfirm(HttpServletRequest request, @RequestParam String verifyCode) {
        AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);
        String logFx = "email update verify code confirm,userEmail=[" + currentUser.getEmail() + "].";
        log.info(logFx + "verifyCode=[{}]", verifyCode);
        try {
            AdamResultDto adamResultDto = adamUserEmailService.updateVerifyCodeConfirm(currentUser, verifyCode);
            return AdamResultDto.getResponseDto(adamResultDto);
        } catch (Exception e) {
            log.error(logFx + "execption=[{}]", e.getMessage(), e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

}
