package com.liquidnet.service.adam.controller.web.user;

import com.liquidnet.common.exception.constant.ErrorCode;
import com.liquidnet.common.swagger.config.SwaggerApiVersion;
import com.liquidnet.service.ResponseDto;
import com.liquidnet.service.adam.common.AdamErrorCode;
import com.liquidnet.service.adam.config.swagger.AdamSwaggerApiVersionConstant;
import com.liquidnet.service.adam.dto.base.AdamCurrentUser;
import com.liquidnet.service.adam.dto.base.AdamResultDto;
import com.liquidnet.service.adam.interceptor.annotation.NoAuth;
import com.liquidnet.service.adam.service.IAdamUserEmailService;
import com.liquidnet.service.adam.service.IAdamUserPasswordService;
import com.liquidnet.service.adam.util.CurrentUserUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;

@Slf4j
@Api(tags = "web user password")
@RestController
@RequestMapping("/adam-user-pw")
public class AdamUserPasswordController {

    @Autowired
    private IAdamUserPasswordService adamUserPasswordService;
    @Autowired
    private IAdamUserEmailService adamUserEmailService;

    /**
     * 忘记密码,发送验证码
     *
     * @param email        邮箱账号
     * @param verifyMethod 验证码接收方式
     */
    @NoAuth
    @ApiOperation(value = "forget verify code send")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_2)
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "email", value = "email account"),
            @ApiImplicitParam(type = "form", dataType = "String", name = "verifyMethod", value = "verify method", allowableValues = "email,phone"),
    })
    @PostMapping("/forget_verify_code_send")
    public ResponseDto forgetVerifyCodeSend(@RequestParam String email, @RequestParam String verifyMethod) {
        String logFx = "forget verify code send.";
        log.info(logFx + "email=[{}],verifyMethod=[{}]", email, verifyMethod);
        try {
            AdamResultDto adamResultDto = adamUserPasswordService.forgetVerifyCodeSend(email, verifyMethod);
            return AdamResultDto.getResponseDto(adamResultDto);
        } catch (Exception e) {
            log.error(logFx + "execption=[{}]", e.getMessage(), e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    /**
     * 忘记密码,确认验证码
     *
     * @param email      邮箱账号
     * @param verifyCode 验证码
     * @param newPw      新密码,为空只校验验证码
     */
    @NoAuth
    @ApiOperation(value = "forget verify code confirm")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_2)
    @PostMapping("/forget_verify_code_confirm")
    public ResponseDto forgetVerifyCodeConfirm(@RequestParam String email,
                                               @RequestParam String verifyCode,
                                               @RequestParam(required = false) String newPw) {
        String logFx = "forget verify code confirm.";
        log.info(logFx + "email=[{}],verifyCode=[{}]", email, verifyCode);
        try {
            AdamResultDto adamResultDto = adamUserPasswordService.forgetVerifyCodeConfirm(email, verifyCode, newPw);
            return AdamResultDto.getResponseDto(adamResultDto);
        } catch (Exception e) {
            log.error(logFx + "execption=[{}]", e.getMessage(), e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

    @NoAuth
    @ApiOperation(value = "set password for invitation link")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_2)
    @PostMapping("/set_by_invite")
    public ResponseDto<?> setByInvite(@RequestParam String key, @RequestParam String pw) {
        try {
            log.info("set password for invitation link. key:{}, pw:{}", key, pw);
            return adamUserEmailService.confirmEmailByInvite(key, pw);
        } catch (Exception e) {
            log.error("err:adam-user-pw/set_by_invite:key:{}:pw:{}", key, pw, e);
            return ResponseDto.failure(ErrorCode.RESPONSE_ERROE_BIZ);
        }
    }

    /**
     * 密码更新
     *
     * @param oldPw 旧密码
     * @param newPw 新密码
     */
    @ApiOperation(value = "update")
    @SwaggerApiVersion(group = AdamSwaggerApiVersionConstant.WEB_1_2)
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "old_pw", value = "old password"),
            @ApiImplicitParam(type = "form", dataType = "String", name = "new_pw", value = "new password"),
    })
    @PostMapping("/update")
    public ResponseDto update(HttpServletRequest request, @RequestParam(value = "old_pw") String oldPw, @RequestParam(value = "new_pw") String newPw) {
        AdamCurrentUser currentUser = CurrentUserUtil.getCurrentUser(request);
        String logFx = "update password,userId=[" + currentUser.getId() + "].";
        log.info(logFx + "oldPw=[{}],newPw=[{}]", oldPw, newPw);
        try {
            AdamResultDto adamResultDto = adamUserPasswordService.update(currentUser.getId(), oldPw, newPw);
            return AdamResultDto.getResponseDto(adamResultDto);
        } catch (Exception e) {
            log.error(logFx + "execption=[{}]", e.getMessage(), e);
            return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
        }
    }

}
