package com.liquidnet.service.adam.interceptor;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.liquidnet.common.exception.constant.ErrorCode;
import com.liquidnet.service.ResponseDto;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.ConversionNotSupportedException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * @author AnJiabin <jiabin.an@lightnet.io>
 * @version V1.0
 * @Description: 全局异常处理类
 * @class: GlobalExceptionHandler
 * @Package com.liquidnet.stellar.filter
 * @Copyright: LightNet @ Copyright (c) 2020
 * @date 2020/8/31 13:10
 */
@Slf4j
@ControllerAdvice
@ResponseBody
public class GlobalExceptionHandler {
    private static final String logExceptionFormat = "Capture Exception By GlobalExceptionHandler: Detail: %s";

    //运行时异常
    @ExceptionHandler({
            RuntimeException.class,
            NullPointerException.class,
            ClassCastException.class,
            IOException.class,
            IndexOutOfBoundsException.class
    })
    public ResponseDto<Exception> runtimeExceptionHandler(Exception ex) {
        log.error(String.format(logExceptionFormat, ex.getMessage()));
        log.error(ex.getMessage(), ex);
        return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
    }

    //未知方法异常
    @ExceptionHandler(NoSuchMethodException.class)
    public ResponseDto<NoSuchMethodException> noSuchMethodExceptionHandler(NoSuchMethodException ex) {
        log.error(String.format(logExceptionFormat, ex.getMessage()));
        log.error(ex.getMessage(), ex);
        return ResponseDto.failure(ErrorCode.HTTP_RESOURCE_NOT_FOUND);
    }

    //500错误
    @ExceptionHandler({ConversionNotSupportedException.class, HttpMessageNotWritableException.class})
    public ResponseDto<RuntimeException> server500(RuntimeException ex) {
        log.error(String.format(logExceptionFormat, ex.getMessage()));
        log.error(ex.getMessage(), ex);
        return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
    }

    //除数不能为0
    @ExceptionHandler({ArithmeticException.class})
    public ResponseDto<ArithmeticException> arithmeticException(ArithmeticException ex) {
        log.error(String.format(logExceptionFormat, ex.getMessage()));
        log.error(ex.getMessage(), ex);
        return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
    }

    //其他错误
    @ExceptionHandler({Exception.class})
    public ResponseDto<Exception> exception(Exception ex) {
        log.error(String.format(logExceptionFormat, ex.getMessage()));
        log.error(ex.getMessage(), ex);
        return ResponseDto.failure(ErrorCode.HTTP_SYSTEM_ERROR);
    }
}
