package com.liquidnet.service.adam.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.liquidnet.service.adam.common.AdamErrorCode;
import com.liquidnet.service.adam.constant.AdamBeneficiaryConstants;
import com.liquidnet.service.adam.constant.AdamPayerConstants;
import com.liquidnet.service.adam.dto.AdamBeneficiaryParam;
import com.liquidnet.service.adam.dto.base.AdamResultDto;
import com.liquidnet.service.adam.dto.query.AdamBeneficiaryQuery;
import com.liquidnet.service.adam.entity.AdamBeneficiary;
import com.liquidnet.service.adam.entity.AdamPayer;
import com.liquidnet.service.adam.incrementer.CustomIdGenerator;
import com.liquidnet.service.adam.mapper.AdamBeneficiaryMapper;
import com.liquidnet.service.adam.service.IAdamBeneficiaryService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>
 * 收款人信息 服务实现类
 * </p>
 *
 * @author liquidnet
 * @since 2020-11-02
 */
@Service
public class AdamBeneficiaryServiceImpl implements IAdamBeneficiaryService {

    @Autowired
    AdamBeneficiaryMapper adamBeneficiaryMapper;

    @Autowired
    CustomIdGenerator customIdGenerator;

    @Override
    public AdamResultDto<AdamBeneficiary> create(AdamBeneficiary beneficiary) {
        beneficiary.setId(customIdGenerator.nextUUID(beneficiary));
        beneficiary.setStatus(AdamBeneficiaryConstants.StatusEnum.NORMAL.getCode());
        beneficiary.setCreateTime(LocalDateTime.now());
        this.setName(beneficiary);
        int i = adamBeneficiaryMapper.insert(beneficiary);
        return i > 0 ? AdamResultDto.success(beneficiary) : AdamResultDto.failure(AdamErrorCode.ADAM001_OPERATION_FAILED);
    }

    @Override
    public AdamResultDto delete(String id, String comId) {
        AdamBeneficiary del = new AdamBeneficiary();
        del.setId(id);
        del.setComId(comId);
        del.setStatus(AdamBeneficiaryConstants.StatusEnum.DELETE.getCode());
        del.setDeleteTime(LocalDateTime.now());
        int i = adamBeneficiaryMapper.updateById(del);
        return i > 0 ? AdamResultDto.success() : AdamResultDto.failure(AdamErrorCode.ADAM001_OPERATION_FAILED);
    }

    @Override
    public List<AdamBeneficiary> selectList(AdamBeneficiaryParam param) {
        QueryWrapper<AdamBeneficiary> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .eq(AdamBeneficiary::getComId, param.getComId())
                .eq(AdamBeneficiary::getStatus, AdamPayerConstants.StatusEnum.NORMAL.getCode())
        ;
        return adamBeneficiaryMapper.selectList(queryWrapper);
    }

    @Override
    public Page<AdamBeneficiary> selectPage(Integer pageNum, Integer pageSize, AdamBeneficiaryQuery param) {
        //拼装查询条件
        LambdaQueryWrapper<AdamBeneficiary> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(!StringUtils.isEmpty(param.getComId()), AdamBeneficiary::getComId, param.getComId())
                .eq(!StringUtils.isEmpty(param.getStatus()), AdamBeneficiary::getStatus, param.getStatus());
        String search = param.getSearch();
        if (!StringUtils.isEmpty(search)) {
            queryWrapper.and(
                    wrapper -> wrapper.like(AdamBeneficiary::getNickName, search)
                            .or().like(AdamBeneficiary::getFullName, search)
                            .or().like(AdamBeneficiary::getCurrency, search)
                            .or().like(AdamBeneficiary::getBankCountry, search)
            );
        }
        Page<AdamBeneficiary> dataPage = adamBeneficiaryMapper.selectPage(new Page<>(pageNum, pageSize), queryWrapper);

        return dataPage;
    }

    @Override
    public AdamBeneficiary selectById(String id) {
        return adamBeneficiaryMapper.selectById(id);
    }

    @Override
    public AdamResultDto<AdamBeneficiary> select(String id, String comId) {
        QueryWrapper<AdamBeneficiary> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .eq(AdamBeneficiary::getId, id)
                .eq(AdamBeneficiary::getComId, comId)
                .eq(AdamBeneficiary::getStatus, AdamBeneficiaryConstants.StatusEnum.NORMAL.getCode())
        ;
        AdamBeneficiary adamBeneficiary = adamBeneficiaryMapper.selectOne(queryWrapper);
        return adamBeneficiary == null ? AdamResultDto.failure(AdamErrorCode.ADAM001_ILLEGAL_PARAM) : AdamResultDto.success(adamBeneficiary);
    }

    @Override
    public AdamResultDto<AdamBeneficiary> update(AdamBeneficiary beneficiary) {
        UpdateWrapper<AdamBeneficiary> updateWrapper = new UpdateWrapper<>();
        updateWrapper.lambda()
                .eq(AdamBeneficiary::getId, beneficiary.getId())
                .eq(AdamBeneficiary::getComId, beneficiary.getComId())
                .eq(AdamBeneficiary::getStatus, AdamPayerConstants.StatusEnum.NORMAL.getCode());
        this.setName(beneficiary);
        beneficiary.setUpdateTime(LocalDateTime.now());
        int i = adamBeneficiaryMapper.update(beneficiary, updateWrapper);
        return i > 0 ? AdamResultDto.success() : AdamResultDto.failure(AdamErrorCode.ADAM001_OPERATION_FAILED);
    }

    private void setName(AdamBeneficiary beneficiary) {
        if (AdamBeneficiaryConstants.LegalEntityTypeEnum.INDIVIDUAL.getCode().equals(beneficiary.getLegalEntityType())) {
            beneficiary.setFullName(beneficiary.getFirstName().concat(beneficiary.getLastName()));
        } else if (AdamBeneficiaryConstants.LegalEntityTypeEnum.COMPANY.getCode().equals(beneficiary.getLegalEntityType())) {
            beneficiary.setFirstName("");
            beneficiary.setLastName("");
        }

    }

}
