package com.liquidnet.service.adam.service.impl;

import com.liquidnet.service.adam.constant.AdamComConstants;
import com.liquidnet.service.adam.common.AdamErrorCode;
import com.liquidnet.service.adam.dto.*;
import com.liquidnet.service.adam.dto.base.AdamResultDto;
import com.liquidnet.service.adam.entity.AdamComInfo;
import com.liquidnet.service.adam.entity.AdamComInfoExtend;
import com.liquidnet.service.adam.entity.AdamComShareholder;
import com.liquidnet.service.adam.service.*;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * <p>
 * 公司信息 服务实现类
 * </p>
 *
 * @author liquidnet
 * @since 2020-09-16
 */
@Service
public class AdamComActivateServiceImpl implements IAdamComActivateService {

    @Autowired
    private IAdamComInfoService adamComInfoService;

    @Autowired
    private IAdamComInfoExtendService adamComInfoExtendService;

    @Autowired
    private IAdamComLegalPersonService adamComLegalPersonService;

    @Autowired
    private IAdamComShareholderService adamComShareholderService;

    @Override
    @Transactional
    public AdamResultDto<AdamComInfoDto> activeCompanyInformation(String temp, AdamComInfoParam param) {
        String comId = param.getId();
        AdamResultDto<AdamComInfo> checkComStateDto = checkComState(comId, AdamComConstants.ComStateEnum.S1);
        if (!checkComStateDto.isSuccess()) {
            return AdamResultDto.failure(checkComStateDto.getCode(), checkComStateDto.getMessage());
        }

        // review 状态不更新公司状态
        if (StringUtils.isBlank(temp) && !this.checkComStateS6(checkComStateDto.getData()).isSuccess()) {
            param.setState(AdamComConstants.ComStateEnum.S2.getCode());
        }

        AdamComInfoExtendParam extendParam = new AdamComInfoExtendParam();
        extendParam.setId(comId);
        extendParam.setFiles(param.getFiles());
        int i = this.extendInsertOrUpdate(extendParam);
        int i1 = adamComInfoService.updateById(param);
        return i > 0 && i1 > 0 ? AdamResultDto.success() : AdamResultDto.failure(AdamErrorCode.ADAM001_SYSTEM_ERROR);
    }

    @Override
    @Transactional
    public AdamResultDto<AdamComInfoExtendDto> activebusinessQuestion(String temp, AdamComInfoExtendParam param) {
        String comId = param.getId();
        AdamResultDto<AdamComInfo> checkComStateDto = checkComState(comId, AdamComConstants.ComStateEnum.S2);
        if (!checkComStateDto.isSuccess()) {
            return AdamResultDto.failure(checkComStateDto.getCode(), checkComStateDto.getMessage());
        }

        param.setFinancialQuestion(null);
        int i = this.extendInsertOrUpdate(param);
        int i1 = 0;
        // review 状态不更新公司状态
        if (StringUtils.isBlank(temp) && !this.checkComStateS6(checkComStateDto.getData()).isSuccess()) {
            AdamComInfoParam comInfoParam = new AdamComInfoParam();
            comInfoParam.setId(comId);
            comInfoParam.setState(AdamComConstants.ComStateEnum.S3.getCode());
            i1 = adamComInfoService.updateById(comInfoParam);
        } else {
            i1 = 1;
        }

        return i > 0 && i1 > 0 ? AdamResultDto.success() : AdamResultDto.failure(AdamErrorCode.ADAM001_SYSTEM_ERROR);
    }

    @Override
    @Transactional
    public AdamResultDto<AdamComLegalPersonDto> activeLegalPerson(String temp, AdamComLegalPersonParam param) {
        String comId = param.getComId();
        AdamResultDto<AdamComInfo> checkComStateDto = checkComState(comId, AdamComConstants.ComStateEnum.S3);
        if (!checkComStateDto.isSuccess()) {
            return AdamResultDto.failure(checkComStateDto.getCode(), checkComStateDto.getMessage());
        }
        AdamComLegalPersonDto legalPersonDto = adamComLegalPersonService.selectByComId(comId);
        int i = 0;
        if (legalPersonDto == null) {
            i = adamComLegalPersonService.insert(param);
        } else {
            i = adamComLegalPersonService.updateByComId(param);
        }
        int i1 = 0;
        // review 状态不更新公司状态
        if (StringUtils.isBlank(temp) && !this.checkComStateS6(checkComStateDto.getData()).isSuccess()) {
            AdamComInfoParam comInfoParam = new AdamComInfoParam();
            comInfoParam.setId(comId);
            comInfoParam.setState(AdamComConstants.ComStateEnum.S4.getCode());
            i1 = adamComInfoService.updateById(comInfoParam);
        } else {
            i1 = 1;
        }

        return i > 0 && i1 > 0 ? AdamResultDto.success() : AdamResultDto.failure(AdamErrorCode.ADAM001_SYSTEM_ERROR);
    }

    @Override
    @Transactional
    public AdamResultDto<AdamComShareholderDto> activeShareholder(String temp, String comId, List<AdamComShareholderParam> list) {

        if (StringUtils.isEmpty(comId) || CollectionUtils.isEmpty(list)) {
            return AdamResultDto.failure(AdamErrorCode.ADAM001_PARAM_ERROR);
        }

        AdamResultDto<AdamComInfo> checkComStateDto = checkComState(comId, AdamComConstants.ComStateEnum.S4);
        if (!checkComStateDto.isSuccess()) {
            return AdamResultDto.failure(checkComStateDto.getCode(), checkComStateDto.getMessage());
        }
        int i = 0;
        for (AdamComShareholderParam param : list) {
            AdamComShareholder adamComShareholder = adamComShareholderService.selectOne(comId, param.getPosition());
            param.setComId(comId);
            if (adamComShareholder == null) {
                i = adamComShareholderService.insert(param);
            } else {
                i = adamComShareholderService.updateByComId(param);
            }
            if (i == 0) {
                TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
                return AdamResultDto.failure(AdamErrorCode.ADAM001_SYSTEM_ERROR);
            }
        }

        AdamComInfoParam comInfoParam = new AdamComInfoParam();
        comInfoParam.setId(comId);
        comInfoParam.setStakeNumber(list.get(0).getStakeNumber());
        if (StringUtils.isBlank(temp) && !this.checkComStateS6(checkComStateDto.getData()).isSuccess()) {
            comInfoParam.setState(AdamComConstants.ComStateEnum.S5.getCode());
        }
        int i1 = adamComInfoService.updateById(comInfoParam);
        if (i1 == 0) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            return AdamResultDto.failure(AdamErrorCode.ADAM001_SYSTEM_ERROR);
        }

        return AdamResultDto.success();
    }

    @Override
    @Transactional
    public AdamResultDto<AdamComInfoExtendDto> activeFinancialQuestion(String temp, AdamComInfoExtendParam param) {
        String comId = param.getId();
        AdamResultDto<AdamComInfo> checkComStateDto = checkComState(comId, AdamComConstants.ComStateEnum.S5);
        if (!checkComStateDto.isSuccess()) {
            return AdamResultDto.failure(checkComStateDto.getCode(), checkComStateDto.getMessage());
        }

        param.setBusinessQuestion(null);

        int i = this.extendInsertOrUpdate(param);
        int i1 = 0;
        // review 状态不更新公司状态
        if (StringUtils.isBlank(temp) && !this.checkComStateS6(checkComStateDto.getData()).isSuccess()) {
            AdamComInfoParam comInfoParam = new AdamComInfoParam();
            comInfoParam.setId(comId);
            comInfoParam.setState(AdamComConstants.ComStateEnum.S6.getCode());
            i1 = adamComInfoService.updateById(comInfoParam);
        } else {
            i1 = 1;
        }

        return i > 0 && i1 > 0 ? AdamResultDto.success() : AdamResultDto.failure(AdamErrorCode.ADAM001_SYSTEM_ERROR);
    }

    /**
     * 检查公司状态是否正常
     *
     * @param comId        公司id
     * @param comStateEnum 状态
     */
    private AdamResultDto<AdamComInfo> checkComState(String comId, AdamComConstants.ComStateEnum comStateEnum) {
        AdamComInfo adamComInfo = adamComInfoService.selectById(comId);
        if (adamComInfo == null) {
            return AdamResultDto.failure(AdamErrorCode.ADAM001_PARAM_ERROR);
        }
        // review状态时,只更新信息,不验证状态
        if (this.checkComStateS6(adamComInfo).isSuccess()) {
            return AdamResultDto.success(adamComInfo);
        }
        if (!comStateEnum.getCode().equals(adamComInfo.getState())) {
            return AdamResultDto.failure(AdamErrorCode.ADAM001_ILLEGAL_OPERATION);
        }
        return AdamResultDto.success(adamComInfo);
    }

    /**
     * 验证公司当前状态是否是 review状态
     * review状态下,只编辑公司信息,不验证和更新公司状态
     *
     * @return success是review/failure不是review
     */
    private AdamResultDto<AdamComInfo> checkComStateS6(AdamComInfo adamComInfo) {
        if (AdamComConstants.ComStateEnum.S6.getCode().equals(adamComInfo.getState())) {
            return AdamResultDto.success(adamComInfo);
        }
        return AdamResultDto.failure(AdamErrorCode.ADAM_COM_0030005);
    }

    /**
     * 公司扩展信息,插入或更新
     *
     * @param param 公司扩展信息
     */
    private int extendInsertOrUpdate(AdamComInfoExtendParam param) {
        int i = 0;
        AdamComInfoExtend adamComInfoExtend = adamComInfoExtendService.selectById(param.getId());
        if (adamComInfoExtend == null) {
            i = adamComInfoExtendService.insert(param);
        } else {
            i = adamComInfoExtendService.updateById(param);
        }
        return i;
    }

}
