package com.liquidnet.service.adam.service.impl;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.liquidnet.service.adam.common.AdamErrorCode;
import com.liquidnet.service.adam.constant.AdamPayerConstants;
import com.liquidnet.service.adam.dto.query.AdamPayerQuery;
import com.liquidnet.service.adam.dto.base.AdamResultDto;
import com.liquidnet.service.adam.entity.AdamPayer;
import com.liquidnet.service.adam.incrementer.CustomIdGenerator;
import com.liquidnet.service.adam.mapper.AdamPayerMapper;
import com.liquidnet.service.adam.service.IAdamPayerService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.List;

/**
 * <p>
 * 付款人信息 服务实现类
 * </p>
 *
 * @author liquidnet
 * @since 2020-11-02
 */
@Service
public class AdamPayerServiceImpl implements IAdamPayerService {

    @Autowired
    AdamPayerMapper adamPayerMapper;

    @Autowired
    CustomIdGenerator customIdGenerator;

    @Override
    public AdamResultDto<AdamPayer> create(AdamPayer payer) {

        payer.setId(customIdGenerator.nextUUID(payer));
        payer.setStatus(AdamPayerConstants.StatusEnum.NORMAL.getCode());
        payer.setCreateTime(LocalDateTime.now());
        this.setName(payer);
        int i = adamPayerMapper.insert(payer);
        return i > 0 ? AdamResultDto.success(payer) : AdamResultDto.failure(AdamErrorCode.ADAM001_OPERATION_FAILED);
    }

    @Override
    public AdamResultDto delete(String id, String comId) {
        AdamPayer del = new AdamPayer();
        del.setId(id);
        del.setStatus(AdamPayerConstants.StatusEnum.DELETE.getCode());
        del.setComId(comId);
        del.setDeleteTime(LocalDateTime.now());
        int i = adamPayerMapper.updateById(del);
        return i > 0 ? AdamResultDto.success() : AdamResultDto.failure(AdamErrorCode.ADAM001_OPERATION_FAILED);
    }

    @Override
    public AdamPayer selectById(String id) {
        return adamPayerMapper.selectById(id);
    }

    @Override
    public AdamResultDto<AdamPayer> select(String id, String comId) {
        QueryWrapper<AdamPayer> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .eq(AdamPayer::getId, id)
                .eq(AdamPayer::getComId, comId)
                .eq(AdamPayer::getStatus, AdamPayerConstants.StatusEnum.NORMAL.getCode())
        ;
        AdamPayer adamPayer = adamPayerMapper.selectOne(queryWrapper);

        return adamPayer == null ? AdamResultDto.failure(AdamErrorCode.ADAM001_ILLEGAL_PARAM) : AdamResultDto.success(adamPayer);

    }

    @Override
    public List<AdamPayer> selectList(AdamPayerQuery param) {
        QueryWrapper<AdamPayer> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda()
                .eq(AdamPayer::getComId, param.getComId())
                .eq(AdamPayer::getStatus, AdamPayerConstants.StatusEnum.NORMAL.getCode())
        ;
        return adamPayerMapper.selectList(queryWrapper);
    }

    @Override
    public Page<AdamPayer> selectPage(Integer pageNum, Integer pageSize, AdamPayerQuery param) {
        //拼装查询条件
        LambdaQueryWrapper<AdamPayer> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(!StringUtils.isEmpty(param.getComId()), AdamPayer::getComId, param.getComId())
                .eq(!StringUtils.isEmpty(param.getStatus()), AdamPayer::getStatus, param.getStatus());
        String search = param.getSearch();
        if (!StringUtils.isEmpty(search)) {
            queryWrapper.and(
                    wrapper -> wrapper.like(AdamPayer::getFullName, search).or().like(AdamPayer::getLegalEntityType, search)
            );
        }

        Page<AdamPayer> dataPage = adamPayerMapper.selectPage(new Page<>(pageNum, pageSize), queryWrapper);

        return dataPage;
    }

    @Override
    public AdamResultDto<AdamPayer> update(AdamPayer payer) {
        UpdateWrapper<AdamPayer> updateWrapper = new UpdateWrapper<>();
        updateWrapper.lambda()
                .eq(AdamPayer::getId, payer.getId())
                .eq(AdamPayer::getComId, payer.getComId())
                .eq(AdamPayer::getStatus, AdamPayerConstants.StatusEnum.NORMAL.getCode());
        this.setName(payer);
        payer.setUpdateTime(LocalDateTime.now());
        int i = adamPayerMapper.update(payer, updateWrapper);
        return i > 0 ? AdamResultDto.success() : AdamResultDto.failure(AdamErrorCode.ADAM001_OPERATION_FAILED);
    }

    private void setName(AdamPayer payer) {
        if (AdamPayerConstants.LegalEntityTypeEnum.INDIVIDUAL.getCode().equals(payer.getLegalEntityType())) {
            payer.setFullName(payer.getFirstName().concat(payer.getLastName()));
        } else if (AdamPayerConstants.LegalEntityTypeEnum.COMPANY.getCode().equals(payer.getLegalEntityType())) {
            payer.setFirstName(null);
            payer.setLastName(null);
        }

    }

}
