package com.liquidnet.service.adam.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.IDGenerator;
import com.liquidnet.service.adam.entity.AdamAddresses;
import com.liquidnet.service.adam.service.IAdamAddressesService;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.adam.dto.AdamAddressesParam;
import com.liquidnet.service.adam.dto.vo.AdamAddressesVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * <p>
 * 收货地址 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2021-05-10
 */
@ApiSupport(order = 10050)
@Api(tags = "收获地址")
@Slf4j
@RestController
@RequestMapping("addr")
public class AdamAddressesController {
    @Autowired
    IAdamAddressesService adamAddressesService;

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "添加收获地址")
    @PostMapping("add")
    public ResponseDto<Object> add(@RequestBody AdamAddressesParam parameter) {
        // TODO: 2021/4/28 数据校验
        String currentUid = CurrentUtil.getCurrentUid();

        AdamAddresses adamAddresses = new AdamAddresses();
        BeanUtils.copyProperties(parameter, adamAddresses);
        adamAddresses.setUid(currentUid);
        adamAddresses.setAddressesId(String.valueOf(IDGenerator.nextSnowId()));
        adamAddresses.setCreatedAt(LocalDateTime.now());
        adamAddresses.setState(1);

        adamAddressesService.add(adamAddresses);

        return ResponseDto.success(adamAddresses.getAddressesId());
    }

    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "收获地址列表")
    @GetMapping("list")
    public ResponseDto<List<AdamAddressesVo>> list() {
        String currentUid = CurrentUtil.getCurrentUid();

        List<AdamAddressesVo> voList = new ArrayList<>();
        adamAddressesService.queryByUid(currentUid).forEach(i -> {
            voList.add(AdamAddressesVo.getNew().copy(i));
        });

        return ResponseDto.success(voList);
    }

    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "设置默认收货地址")
    @PostMapping("def/{addrId}")
    public ResponseDto<Object> def(@PathVariable String addrId) {
        String currentUid = CurrentUtil.getCurrentUid();

        AdamAddresses addresses = adamAddressesService.query(currentUid, addrId);
        if (null == addresses) return ResponseDto.failure(ErrorMapping.get("10106"));

        if (!addresses.getIsDefault()) adamAddressesService.def(currentUid, addrId);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 4)
    @ApiOperation(value = "编辑收获地址")
    @PostMapping("edit")
    public ResponseDto<Object> edit(@RequestBody AdamAddressesParam parameter) {
        // TODO: 2021/4/28 校验
        String currentUid = CurrentUtil.getCurrentUid();

        AdamAddresses updateInfo = new AdamAddresses();
        BeanUtils.copyProperties(parameter, updateInfo);
        updateInfo.setUpdatedAt(LocalDateTime.now());
        updateInfo.setUid(currentUid);
        updateInfo.setState(1);

        adamAddressesService.edit(updateInfo);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 5)
    @ApiOperation(value = "删除收获地址")
    @PostMapping("del/{addrId}")
    public ResponseDto<Object> del(@PathVariable String addrId) {

        String currentUid = CurrentUtil.getCurrentUid();

        AdamAddresses addresses = adamAddressesService.query(currentUid, addrId);
        if (null == addresses) return ResponseDto.failure(ErrorMapping.get("10106"));

        adamAddressesService.remove(currentUid, addrId);
        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 6)
    @ApiOperation(value = "收货地址详情")
    @PostMapping("query/{id}")
    public ResponseDto<AdamAddressesVo> query(@PathVariable(name = "id") String addrId,
                                              @RequestParam(name = "uid", required = false) String uid) {
        // TODO: 2021/4/28 校验

        AdamAddresses addresses = adamAddressesService.query(StringUtils.isBlank(uid) ? CurrentUtil.getCurrentUid() : uid, addrId);

        return null == addresses ? ResponseDto.failure(ErrorMapping.get(10105)) : ResponseDto.success(AdamAddressesVo.getNew().copy(addresses));
    }

    @ApiOperationSupport(order = 7)
    @ApiOperation(value = "获取默认收获地址")
    @PostMapping("query/def")
    public ResponseDto<AdamAddressesVo> queryDef() {
        return ResponseDto.success(adamAddressesService.queryDefault(CurrentUtil.getCurrentUid()));
    }
}
