package com.liquidnet.service.adam.controller;

import com.github.pagehelper.PageInfo;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.service.adam.dto.vo.AdamCollectBaseVo;
import com.liquidnet.service.adam.dto.vo.AdamCollectVo;
import com.liquidnet.service.adam.service.IAdamCollectionService;
import com.liquidnet.service.base.ResponseDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.Pattern;
import javax.validation.constraints.Size;

/**
 * <p>
 * 收藏 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2021-05-10
 */
@ApiSupport(order = 10060)
@Api(tags = "收藏")
@Slf4j
@RestController
@RequestMapping("collect")
public class AdamCollectionController {
    @Autowired
    IAdamCollectionService adamCollectionService;

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "添加收藏")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "type", value = "收藏类型[TICKET,VIDEO]", allowableValues = "TICKET,VIDEO"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "contentId", value = "内容ID[1,64]"),
    })
    @PostMapping("add")
    public ResponseDto<Object> add(//@Pattern(regexp = "\\b(TICKET,VIDEO)\\b", message = "收藏类型无效")
                                   @RequestParam String type,
                                   @Size(min = 1, max = 64, message = "收藏内容ID长度限制1-64位")
                                   @RequestParam String contentId) {
        String currentUid = CurrentUtil.getCurrentUid();

        if (adamCollectionService.exists(currentUid, type, contentId)) {
            return ResponseDto.success();
        }

        AdamCollectBaseVo collectVo = AdamCollectBaseVo.getNew();
        collectVo.setType(type);
        collectVo.setContentId(contentId);
        collectVo.setUid(currentUid);
        adamCollectionService.add(collectVo);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "删除收藏")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "type", value = "收藏类型", allowableValues = "TICKET,VIDEO"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "contentIds", value = "内容ID[多个ID用','分隔]"),
    })
    @PostMapping("del")
    public ResponseDto<Object> del(//@Pattern(regexp = "\\b(TICKET,VIDEO)\\b", message = "收藏类型无效")
                                   @RequestParam String type,
                                   @NotBlank(message = "内容ID不能为空") @RequestParam String contentIds) {
        adamCollectionService.del(CurrentUtil.getCurrentUid(), type, contentIds.split(","));

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "收藏列表")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "type", value = "收藏类型", allowableValues = "TICKET,VIDEO"),
            @ApiImplicitParam(type = "form", required = false, dataType = "Integer", name = "pageNo", value = "页码", example = "1"),
            @ApiImplicitParam(type = "form", required = false, dataType = "Integer", name = "pageSize", value = "页展示条数", example = "5"),
    })
    @PostMapping("list")
    public ResponseDto<PageInfo<AdamCollectVo>> list(//@Pattern(regexp = "\\b(TICKET,VIDEO)\\b", message = "收藏类型无效")
                                                     @RequestParam String type,
                                                     @RequestParam(defaultValue = "1", required = false) int pageNo,
                                                     @RequestParam(defaultValue = "5", required = false) int pageSize) {
        pageNo = pageNo > 0 ? pageNo : 1;
        pageSize = pageSize <= 0 || pageSize > 10 ? 5 : pageSize;

        return ResponseDto.success(adamCollectionService.queryPage(CurrentUtil.getCurrentUid(), type, pageNo, pageSize));
    }

//    @ApiOperationSupport(order = 4)
//    @ApiOperation(value = "获取特定内容的收藏用户列表 -> 返回结构待定")
//    @ApiImplicitParams({
//            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "type", value = "收藏类型", allowableValues = "TICKET,VIDEO"),
//            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "contentId", value = "内容ID[64]"),
//    })
//    @PostMapping("list/user")
//    public ResponseDto<Object> listUser(@RequestParam String type, @RequestParam String contentId) {
//        String uid = "1";
//
//
//
//        return ResponseDto.success();
//    }

    @ApiOperationSupport(order = 5)
    @ApiOperation(value = "获取收藏状态")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "type", value = "收藏类型", allowableValues = "TICKET,VIDEO"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "contentId", value = "内容ID[64]"),
    })
    @GetMapping("state")
    public ResponseDto<Boolean> state(//@Pattern(regexp = "\\b(TICKET,VIDEO)\\b", message = "收藏类型无效")
                                      @RequestParam String type,
                                      @Size(min = 1, max = 64, message = "收藏内容ID长度限制1-64位")
                                      @RequestParam String contentId) {
        return ResponseDto.success(adamCollectionService.queryState(CurrentUtil.getCurrentUid(), type, contentId));
    }
}
