package com.liquidnet.service.goblin.controller.manage;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.goblin.dto.manage.GoblinStoreMgtCertificationParam;
import com.liquidnet.service.goblin.dto.manage.GoblinStoreMgtCompleteParam;
import com.liquidnet.service.goblin.dto.vo.GoblinStoreInfoVo;
import com.liquidnet.service.goblin.service.manage.IGoblinStoreMgtCertificationService;
import com.liquidnet.service.goblin.util.GoblinMongoUtils;
import com.liquidnet.service.goblin.util.GoblinRedisUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.util.Arrays;
import java.util.List;

@ApiSupport(order = 149001)
@Api(tags = "店铺认证")
@Slf4j
@Validated
@RestController
@RequestMapping("store/mgt/cert")
public class GoblinStoreMgtCertificationController {
    @Autowired
    IGoblinStoreMgtCertificationService goblinStoreMgtCertificationService;
    @Autowired
    GoblinMongoUtils goblinMongoUtils;
    @Autowired
    GoblinRedisUtils goblinRedisUtils;

    public static final List<String> ENABLE_CERT_STATUS = Arrays.asList("0", "2");

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "店铺认证第一步:认证资料")
    @PostMapping("fstep")
    public ResponseDto<String> certificationFirstStep(@Valid @RequestBody GoblinStoreMgtCertificationParam certificationParam) {
        GoblinStoreInfoVo storeInfoVo = goblinRedisUtils.getStoreInfoVoByUid(CurrentUtil.getCurrentUid());
        if (null != storeInfoVo && !ENABLE_CERT_STATUS.contains(storeInfoVo.getStatus())) {
            log.warn("店铺认证:无效请求:店铺已认证，无需再次认证[UID={},storeId={},storeStatus={}]", CurrentUtil.getCurrentUid(), storeInfoVo.getStoreId(), storeInfoVo.getStatus());
            return ResponseDto.failure(ErrorMapping.get("149001"));
        }
        if (log.isDebugEnabled()) {
            log.debug("店铺认证[GoblinStoreMgtCertificationParam:{}]", JsonUtils.toJson(certificationParam));
        }
        storeInfoVo = goblinStoreMgtCertificationService.certificationInformationProcessing(certificationParam, storeInfoVo);
        return ResponseDto.success(storeInfoVo.getStoreId());
    }

    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "店铺认证第二步:完善信息")
    @PostMapping("sstep")
    public ResponseDto<Object> certificationSecondStep(@Valid @RequestBody GoblinStoreMgtCompleteParam completeParam) {
        GoblinStoreInfoVo storeInfoVo = goblinRedisUtils.getStoreInfoVoByUid(CurrentUtil.getCurrentUid());
        if (null == storeInfoVo) {
            log.warn("店铺认证:完善信息:店铺信息不存在，请按流程完成第一步:[UID={}]", CurrentUtil.getCurrentUid());
            return ResponseDto.failure(ErrorMapping.get("149001"));
        }
        if (!ENABLE_CERT_STATUS.contains(storeInfoVo.getStatus())) {
            log.warn("店铺认证:无效请求:店铺已认证，无需再次认证[UID={},storeId={},storeStatus={}]", CurrentUtil.getCurrentUid(), storeInfoVo.getStoreId(), storeInfoVo.getStatus());
            return ResponseDto.failure(ErrorMapping.get("149001"));
        }
        if (log.isDebugEnabled()) {
            log.debug("店铺认证[GoblinStoreMgtCompleteParam:{}]", JsonUtils.toJson(completeParam));
        }
        return ResponseDto.success(goblinStoreMgtCertificationService.certificationCompleteProcessing(completeParam, storeInfoVo));
    }

    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "店铺认证编辑中:资料详情")
    @PostMapping("einfo")
    public ResponseDto<GoblinStoreInfoVo> certificationInfo() {
        return ResponseDto.success(goblinRedisUtils.getStoreInfoVoByUid(CurrentUtil.getCurrentUid()));
    }

    @ApiOperationSupport(order = 4)
    @ApiOperation(value = "店铺认证认证中:店铺状态", notes = "认证状态[-1-未申请|0-填写资料中|1-店铺审核中|2-审核不通过|3-审核通过|4-停业|5-营业]")
    @PostMapping("status")
    public ResponseDto<String> certificationStatus() {
        GoblinStoreInfoVo storeInfoVo = goblinRedisUtils.getStoreInfoVoByUid(CurrentUtil.getCurrentUid());
        return ResponseDto.success(null == storeInfoVo ? "-1" : storeInfoVo.getStatus());
    }

    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "删除店铺")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "storeId", value = "店铺ID"),
    })
    @PostMapping("cancel")
    public ResponseDto<Object> certificationCancel(@NotBlank(message = "店铺ID不能为空") @RequestParam String storeId) {
        String currentUid = CurrentUtil.getCurrentUid();
        if (!goblinRedisUtils.hasStoreId(currentUid, storeId)) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        GoblinStoreInfoVo storeInfoVo = goblinRedisUtils.getStoreInfoVo(storeId);
        if (null == storeInfoVo || Arrays.asList("3","4","5").contains(storeInfoVo.getStatus())) {
            return ResponseDto.success();// 店铺认证通过后不允许删除
        }
        log.info("删除店铺[UID={},storeId={}]", currentUid, storeId);
        return ResponseDto.success(goblinStoreMgtCertificationService.certificationCancelProcessing(storeId, currentUid));
    }
}
