package com.liquidnet.service.goblin.controller.manage;

import com.liquidnet.common.exception.LiquidnetServiceException;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.goblin.constant.GoblinRedisConst;
import com.liquidnet.service.goblin.service.impl.manage.GoblinStoreMgtGoodsImportService;
import com.liquidnet.service.goblin.util.GoblinRedisUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.constraints.NotBlank;
import java.io.IOException;

@Api(tags = "店铺商品管理")
@Slf4j
@RestController
@RequestMapping("store/mgt/goods")
public class GoblinStoreMgtGoodsImportController {
    @Autowired
    GoblinRedisUtils goblinRedisUtils;
    @Autowired
    private GoblinStoreMgtGoodsImportService goblinStoreMgtGoodsImportService;

    @PostMapping("/upload")
    @ApiOperation(value = "批量导入数据", notes = "同一个店铺多次导入数据时间间隔最短30秒")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "File", name = "file", value = "文件"),
            @ApiImplicitParam(type = "form", required = true, dataType = "Integer", name = "dataType", value = "导入数据类型[1-商品数据]", example = "1"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "storeId", value = "店铺ID"),
    })
    public ResponseDto<String> upload(@RequestParam MultipartFile file, @RequestParam int dataType,
                                        @RequestParam @NotBlank(message = "店铺ID不能为空") String storeId) {
        String currentUid = CurrentUtil.getCurrentUid();
        if (!goblinRedisUtils.hasStoreId(currentUid, storeId)) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        String lk = GoblinRedisConst.BASIC_GOODS_SKU.concat("lk").concat(storeId);
        if (!goblinRedisUtils.redisUtil.lock(lk, 1, 30)) {
            return ResponseDto.failure(ErrorMapping.get("140000"));
        }
        String analysisResultMsg;
        try {
            switch (dataType) {
                case 1:// 模板https://img.zhengzai.tv/2022/11/09/shoppingmallimport.xlsx
                    analysisResultMsg = goblinStoreMgtGoodsImportService.goodsInformationDataAnalysisProcessing(file, currentUid, storeId);
                    break;
                case 2:
                default:
                    return ResponseDto.failure(ErrorMapping.get("149001"));
            }
        } catch (LiquidnetServiceException e) {
            goblinRedisUtils.redisUtil.uLock(lk);
            return ResponseDto.failure(e.getMessage());
        } catch (Exception e) {
            log.warn("店铺商品管理:批量导入数据:异常[UID={},dataType={},fileName={}]", currentUid, dataType, file.getOriginalFilename(), e);
            return ResponseDto.failure("数据有误，解析失败");
        } finally {
            goblinRedisUtils.redisUtil.uLock(lk);
        }
        return ResponseDto.success(analysisResultMsg);
    }
}
