package com.liquidnet.service.adam.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.service.adam.dto.AdamAddressesParam;
import com.liquidnet.service.adam.dto.vo.AdamAddressesVo;
import com.liquidnet.service.adam.service.IAdamAddressesService;
import com.liquidnet.service.adam.service.IAdamRdmService;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.util.List;

/**
 * <p>
 * 收货地址 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2021-05-10
 */
@ApiSupport(order = 10050)
@Api(tags = "收获地址")
@Slf4j
@Validated
@RestController
@RequestMapping("addr")
public class AdamAddressesController {
    @Autowired
    IAdamRdmService adamRdmService;
    @Autowired
    IAdamAddressesService adamAddressesService;

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "添加收获地址")
    @PostMapping("add")
    public ResponseDto<Object> add(@Valid @RequestBody AdamAddressesParam parameter) {
        return ResponseDto.success(adamAddressesService.add(parameter));
    }

    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "收获地址列表")
    @GetMapping("list")
    public ResponseDto<List<AdamAddressesVo>> list() {
        return ResponseDto.success(adamRdmService.getAddressesVoByUid(CurrentUtil.getCurrentUid()));
    }

    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "设置默认收货地址")
    @PostMapping("def/{addrId}")
    public ResponseDto<Object> def(@NotBlank @PathVariable String addrId) {
        String currentUid = CurrentUtil.getCurrentUid();

        AdamAddressesVo vo = adamRdmService.getAddressesVoByUidAddressesId(currentUid, addrId);
        if (null == vo) return ResponseDto.failure(ErrorMapping.get("10106"));

        if (!vo.getIsDefault()) adamAddressesService.def(currentUid, addrId);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 4)
    @ApiOperation(value = "编辑收获地址")
    @PostMapping("edit")
    public ResponseDto<Object> edit(@RequestBody @Valid AdamAddressesParam parameter) {
        if (StringUtils.isBlank(parameter.getAddressesId())) {
            return ResponseDto.failure(ErrorMapping.get("10016"));
        }
        if (null == adamRdmService.getAddressesVoByUidAddressesId(CurrentUtil.getCurrentUid(), parameter.getAddressesId())) {
            return ResponseDto.failure(ErrorMapping.get("10106"));
        }

        adamAddressesService.edit(parameter);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 5)
    @ApiOperation(value = "删除收获地址")
    @PostMapping("del/{addrId}")
    public ResponseDto<Object> del(@NotBlank @PathVariable String addrId) {
        String currentUid = CurrentUtil.getCurrentUid();

        AdamAddressesVo vo = adamRdmService.getAddressesVoByUidAddressesId(currentUid, addrId);
        if (null == vo) return ResponseDto.failure(ErrorMapping.get("10106"));

        adamAddressesService.remove(currentUid, addrId);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 6)
    @ApiOperation(value = "收货地址详情")
    @PostMapping("query/{id}")
    public ResponseDto<AdamAddressesVo> query(@NotBlank @PathVariable(name = "id") String addrId) {
        AdamAddressesVo vo = adamRdmService.getAddressesVoByUidAddressesId(CurrentUtil.getCurrentUid(), addrId);
        return null == vo ? ResponseDto.failure(ErrorMapping.get("10106")) : ResponseDto.success(vo);
    }

    @ApiOperationSupport(order = 7)
    @ApiOperation(value = "获取默认收获地址")
    @PostMapping("query/def")
    public ResponseDto<AdamAddressesVo> queryDef() {
        return ResponseDto.success(adamAddressesService.queryDefault(CurrentUtil.getCurrentUid()));
    }

    @GetMapping("query/depth")
    public ResponseDto<AdamAddressesVo> queryForDepth(@NotBlank @RequestParam String addrId,
                                                      @NotBlank @RequestParam String uid) {
        AdamAddressesVo vo = adamRdmService.getAddressesVoByUidAddressesId(uid, addrId);
        return null == vo ? ResponseDto.failure(ErrorMapping.get("10106")) : ResponseDto.success(vo);
    }
}
