package com.liquidnet.service.adam.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.SensitizeUtil;
import com.liquidnet.service.adam.dto.AdamEntersParam;
import com.liquidnet.service.adam.dto.vo.AdamEntersVo;
import com.liquidnet.service.adam.service.IAdamEntersService;
import com.liquidnet.service.adam.service.AdamRdmService;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.util.List;
import java.util.Optional;

/**
 * <p>
 * 入场人 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2021-04-28
 */
@ApiSupport(order = 10040)
@Api(tags = "入场人")
@Slf4j
@Validated
@RestController
@RequestMapping("enters")
public class AdamEntersController {
    @Autowired
    AdamRdmService adamRdmService;
    @Autowired
    IAdamEntersService adamEntersService;

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "添加入场人")
    @PostMapping("add")
    public ResponseDto<String> add(@RequestBody @Valid AdamEntersParam parameter) {
        List<AdamEntersVo> vos = adamRdmService.getEntersVoByUid(CurrentUtil.getCurrentUid());

        Optional<AdamEntersVo> any = vos.stream().filter(r -> (r.getIdCard().equals(parameter.getIdCard())) && r.getType().equals(parameter.getType())).findAny();
        if (any.isPresent()) {
            return ResponseDto.failure(ErrorMapping.get("10019"));
        }

        String entersId = adamEntersService.add(parameter);
        return ResponseDto.success(entersId);
    }

    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "入场人列表")
    @GetMapping("list")
    public ResponseDto<List<AdamEntersVo>> list() {
        List<AdamEntersVo> vos = adamRdmService.getEntersVoByUid(CurrentUtil.getCurrentUid());
        if (!CollectionUtils.isEmpty(vos)) {
            for (AdamEntersVo vo : vos) {
                vo.setName(SensitizeUtil.chineseName(vo.getName()));
                vo.setIdCard(SensitizeUtil.custom(vo.getIdCard(), 3, 2));
                vo.setMobile(SensitizeUtil.custom(vo.getMobile(), 3, 4));
            }
        }
        return ResponseDto.success(vos);
    }

    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "设置默认入场人")
    @PostMapping("def/{entersId}")
    public ResponseDto<Object> def(@NotBlank @PathVariable String entersId) {
        String currentUid = CurrentUtil.getCurrentUid();

        AdamEntersVo vo = adamRdmService.getEntersVoByUidEntersId(currentUid, entersId);
        if (null == vo) return ResponseDto.failure(ErrorMapping.get("10105"));

        if (!vo.getIsDefault()) {
            adamEntersService.def(currentUid, entersId);
        }
        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 4)
    @ApiOperation(value = "编辑入场人")
    @PostMapping("edit")
    public ResponseDto<Object> edit(@RequestBody @Valid AdamEntersParam parameter) {
        if (StringUtils.isBlank(parameter.getEntersId())) {
            return ResponseDto.failure(ErrorMapping.get("10015"));
        }
        if (null == adamRdmService.getEntersVoByUidEntersId(CurrentUtil.getCurrentUid(), parameter.getEntersId())) {
            return ResponseDto.failure(ErrorMapping.get("10105"));
        }

        adamEntersService.edit(parameter);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 5)
    @ApiOperation(value = "删除入场人")
    @PostMapping("del/{entersId}")
    public ResponseDto<Object> del(@NotBlank @PathVariable String entersId) {
        String currentUid = CurrentUtil.getCurrentUid();

        AdamEntersVo enters = adamRdmService.getEntersVoByUidEntersId(currentUid, entersId);
        if (null == enters) return ResponseDto.failure(ErrorMapping.get("10105"));

        adamEntersService.remove(currentUid, entersId);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 6)
    @ApiOperation(value = "入场人详情")
    @PostMapping("query")
    public ResponseDto<List<AdamEntersVo>> query(@NotBlank @RequestParam String entersIds) {
        List<AdamEntersVo> vos = adamRdmService.getEntersVoByUidEntersIds(CurrentUtil.getCurrentUid(), entersIds.split(","));
        if (!CollectionUtils.isEmpty(vos)) {
            for (AdamEntersVo vo : vos) {
                vo.setName(SensitizeUtil.chineseName(vo.getName()));
                vo.setIdCard(SensitizeUtil.custom(vo.getIdCard(), 3, 2));
                vo.setMobile(SensitizeUtil.custom(vo.getMobile(), 3, 4));
            }
        }
        return CollectionUtils.isEmpty(vos) ? ResponseDto.failure(ErrorMapping.get("10105")) : ResponseDto.success(vos);
    }

    @ApiOperationSupport(order = 7)
    @ApiOperation(value = "获取默认入场人")
    @PostMapping("query/def")
    public ResponseDto<AdamEntersVo> queryDef() {
        AdamEntersVo vo = adamEntersService.queryDefault(CurrentUtil.getCurrentUid());
        if (null != vo) {
            vo.setName(SensitizeUtil.chineseName(vo.getName()));
            vo.setIdCard(SensitizeUtil.custom(vo.getIdCard(), 3, 2));
            vo.setMobile(SensitizeUtil.custom(vo.getMobile(), 3, 4));
        }
        return ResponseDto.success(vo);
    }

    @GetMapping("query/depth")
    public ResponseDto<AdamEntersVo> queryForDepth(@NotBlank @RequestParam String entersId,
                                                   @NotBlank @RequestParam String uid) {
        AdamEntersVo vo = adamRdmService.getEntersVoByUidEntersId(uid, entersId);
        return null == vo ? ResponseDto.failure(ErrorMapping.get("10105")) : ResponseDto.success(vo);
    }
}
