package com.liquidnet.service.chime.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.liquidnet.commons.lang.util.StringUtil;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.chime.dto.ChimeUserInfoDto;
import com.liquidnet.service.chime.dto.ChimeUserRegisterReqDto;
import com.liquidnet.service.chime.dto.ChimeUserUpdateReqDto;
import com.liquidnet.service.chime.service.IChimeUserService;
import com.liquidnet.service.chime.vo.mongo.ChimeUserInfoVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * @author AnJiabin <anjiabin@zhengzai.tv>
 * @version V1.0
 * @Description: 社交用户控制类
 * @class: ChimeUserController
 * @Package com.liquidnet.service.chime.controller
 * @Copyright: LightNet @ Copyright (c) 2021
 * @date 2021/9/8 15:43
 */
@Api(tags = "在场-用户相关")
@RestController
@RequestMapping("user")
@Validated
@Slf4j
public class ChimeUserController {
    @Autowired
    private IChimeUserService chimeUserService;

    @ApiOperationSupport(order = 5)
    @ApiOperation(value = "社交用户注册")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "userId", value = "用户id"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "sex", value = "性别"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "tags", value = "兴趣标签[{\"val\":\"tag1\",\"desc\":\"tag描述\"},{\"val\":\"tag2\",\"desc\":\"tag描述2\"}]"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "operateType", value = "CREATE,UPDATE"),
    })
    @PostMapping(value = {"register"})
    public ResponseDto<String> register(@RequestParam @Validated String userId
            , @RequestParam @Validated String sex
            , @RequestParam @Validated String tags
            , @RequestParam @Validated String operateType
    ) {
        log.info("ChimeUserController.register userId:{}",userId);
        log.info("ChimeUserController.register sex:{}",sex);
        log.info("ChimeUserController.register tags:{}",tags);
        log.info("ChimeUserController.register operateType:{}",operateType);
        if(operateType.equalsIgnoreCase("CREATE")){
            ChimeUserRegisterReqDto reqDto = ChimeUserRegisterReqDto.getNew();
            reqDto.setUserId(userId);
            reqDto.setSex(sex);
            reqDto.setTags(tags);
            ChimeUserInfoVo chimeUserInfoVo = chimeUserService.createChimeUserInfo(reqDto);
            if(StringUtil.isNull(chimeUserInfoVo)){
                return ResponseDto.failure("chime创建社交用户失败！");
            }
            return ResponseDto.success("创建成功！");
        }else if(operateType.equalsIgnoreCase("UPDATE")) {
            ChimeUserUpdateReqDto reqDto = ChimeUserUpdateReqDto.getNew();
            reqDto.setUserId(userId);
            reqDto.setSex(sex);
            reqDto.setTags(tags);
            //更新用户信息
            Boolean rs = chimeUserService.updateUserInfo(reqDto);
            if(!rs){
                return ResponseDto.failure("更新社交用户信息失败！");
            }
            return ResponseDto.success(rs.toString());
        }
        return null;
    }
    @GetMapping("getUserByUserId")
    @ApiOperation("查询单个用户信息")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "query", dataType = "String", name = "userId", value = "用户ID", required = true)
    })
    public ResponseDto<ChimeUserInfoDto> getUserByUserId(
            @RequestParam(defaultValue = "") String userId
    ) {
        ChimeUserInfoDto userInfoDto = chimeUserService.getUserByUserId(userId);
        if(StringUtil.isNull(userInfoDto)){
            ResponseDto.failure("获取用户信息失败！userId:{}",userId);
        }
        return ResponseDto.success(userInfoDto);
    }

    @GetMapping("userLikeOperation")
    @ApiOperation("划卡计数-喜欢")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "query", dataType = "String", name = "currentUserId", value = "当前登录用户ID", required = true),
            @ApiImplicitParam(type = "query", dataType = "String", name = "targetUserId", value = "目标用户ID", required = true)
    })
    public ResponseDto<ChimeUserInfoDto> userLikeOperation(
            @RequestParam(defaultValue = "") String currentUserId,
            @RequestParam(defaultValue = "") String targetUserId
    ) {
        boolean result = chimeUserService.userLikeOperation(currentUserId,targetUserId);
        if(!result){
            ResponseDto.failure("用户划卡-喜欢-操作失败！currentUserId:{} targetUserId:{}",currentUserId,targetUserId);
        }
        return ResponseDto.success();
    }

    @GetMapping("userDislikeOperation")
    @ApiOperation("划卡计数-不喜欢")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "query", dataType = "String", name = "currentUserId", value = "当前登录用户ID", required = true),
            @ApiImplicitParam(type = "query", dataType = "String", name = "targetUserId", value = "目标用户ID", required = true)
    })
    public ResponseDto<ChimeUserInfoDto> userDisLikeOperation(
            @RequestParam(defaultValue = "") String currentUserId,
            @RequestParam(defaultValue = "") String targetUserId
    ) {
        boolean result = chimeUserService.userDisLikeOperation(currentUserId,targetUserId);
        if(!result){
            ResponseDto.failure("用户划卡-不喜欢-操作失败！currentUserId:{} targetUserId:{}",currentUserId,targetUserId);
        }
        return ResponseDto.success();
    }
}
