package com.liquidnet.service.goblin.controller;

import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.goblin.dto.vo.GoblinStoreCouponVo;
import com.liquidnet.service.goblin.dto.vo.GoblinUseResultVo;
import com.liquidnet.service.goblin.dto.vo.GoblinUserCouponVo;
import com.liquidnet.service.goblin.param.BackCouponParam;
import com.liquidnet.service.goblin.service.GoblinCouponService;
import com.liquidnet.service.goblin.util.GoblinRedisUtils;
import io.swagger.annotations.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

@Slf4j
@Api(tags = "正在下单相关-商铺")
@RestController
@RequestMapping("/store/coupon")
public class GoblinCouponController {
    @Autowired
    GoblinRedisUtils goblinRedisUtils;
    @Autowired
    GoblinCouponService goblinCouponService;

    @PostMapping("type")
    @ApiOperation("商品可参与券类型")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "spuId", value = "spuId"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "storeId", value = "店铺id"),
    })
    public ResponseDto<ArrayList<String>> getSpuType(@RequestParam("spuId") @Valid String spuId,
                                                     @RequestParam("storeId") @Valid String storeId) {
        return ResponseDto.success(goblinCouponService.getSpuType(spuId, storeId));
    }

    @PostMapping("type/list")
    @ApiOperation("券列表[根据类型]")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "spuId", value = "spuId"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "storeId", value = "店铺id"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "type", value = "券类型[1-代金券|2-折扣券|3-满减券]"),
    })
    public ResponseDto<ArrayList<GoblinStoreCouponVo>> getSpuByType(@RequestParam("spuId") @Valid String spuId,
                                                                    @RequestParam("storeId") @Valid String storeId,
                                                                    @RequestParam("type") @Valid String type) {
        return ResponseDto.success(goblinCouponService.getSpuByType(spuId, storeId, type));
    }

    @PostMapping("receive")
    @ApiOperation("订单状态")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "storeCouponId", value = "平台券id"),
//            @ApiImplicitParam(type = "form", required = true, dataType = "Integer", name = "number", value = "数量"),
    })
    public ResponseDto<Boolean> receive(@RequestParam("storeCouponId") @Valid String storeCouponId) {
//    public ResponseDto<Boolean> checkOrderResult(@RequestParam("storeCouponId") @Valid String storeCouponId,
//                                                 @RequestParam("number") @Valid Integer number) {
        String currentUid = CurrentUtil.getCurrentUid();
        GoblinStoreCouponVo storeCouponVo = goblinRedisUtils.getStoreCouponVo(storeCouponId);
        if (null == storeCouponVo) {
            log.warn("商城:用户领取优惠券:优惠券不存在:[uid={},storeCouponId={}]", currentUid, storeCouponId);
            return ResponseDto.failure(ErrorMapping.get("140050"));
        }
        if (!storeCouponVo.getState().equals("1")) {
            return ResponseDto.failure(ErrorMapping.get("140051"));
        }
        List<GoblinUserCouponVo> userCouponVos = goblinRedisUtils.getUserCouponVos(currentUid);
        if (!CollectionUtils.isEmpty(userCouponVos)) {
            int beforeSize = userCouponVos.size();
            userCouponVos.removeIf(vo -> vo.getStoreCouponId().equals(storeCouponId));
            if ((beforeSize - userCouponVos.size()) >= storeCouponVo.getReceiveLimit()) {
                return ResponseDto.failure(ErrorMapping.get("140052"));
            }
        }
        if (goblinRedisUtils.getStoreCouponStock(storeCouponId) > 0) {
            Boolean resultFlg = goblinCouponService.receiveCoupon(currentUid, userCouponVos, storeCouponVo);
            return resultFlg ? ResponseDto.success() : ResponseDto.failure();
        }
        return ResponseDto.failure(ErrorMapping.get("140053"));
    }

    @PostMapping("my/list")
    @ApiOperation("我的券券列表[根据类型]")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "type", value = "类型 1可用 2过期/已使用"),
    })
    public ResponseDto<List<GoblinUserCouponVo>> getList(@RequestParam("type") @Valid String type) {
        return ResponseDto.success(goblinCouponService.getList(type));
    }

    @PostMapping("can/use")
    @ApiOperation("是否可用券")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "Number", name = "totalPrice", value = "spuId"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "spuId", value = "逗号隔开"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "uid", value = "用户id"),
    })
    public ResponseDto<Boolean> canUse(@RequestParam("totalPrice") @Valid BigDecimal totalPrice,
                                       @RequestParam("spuId") @Valid String spuId,
                                       @RequestParam(value = "uid", required = false) @Valid String uid) {
        if (uid == null) {
            uid = CurrentUtil.getCurrentUid();
        }
        return ResponseDto.success(goblinCouponService.canUse(totalPrice, spuId, uid));
    }

    @PostMapping("useList")
    @ApiOperation("可用券列表")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "Number", name = "totalPrice", value = "spuId"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "spuId", value = "逗号隔开"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "uid", value = "用户id")
    })
    public ResponseDto<List<GoblinUserCouponVo>> useList(@RequestParam("totalPrice") @Valid BigDecimal totalPrice,
                                                         @RequestParam("spuId") @Valid String spuId,
                                                         @RequestParam(value = "uid", required = false) @Valid String uid) {
        if (uid == null) {
            uid = CurrentUtil.getCurrentUid();
        }
        return ResponseDto.success(goblinCouponService.useList(totalPrice, spuId, uid));
    }

    @PostMapping("useCoupon")
    @ApiOperation("用券")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "ucouponId", value = "券id"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "content", value = "消费内容"),
            @ApiImplicitParam(type = "form", required = true, dataType = "Number", name = "totalPrice", value = "总价格"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "spuId", value = "spuId逗号隔开"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "uid", value = "uid"),
    })
    public ResponseDto<GoblinUseResultVo> checkOrderResult(@RequestParam("ucouponId") @Valid String ucouponId,
                                                           @RequestParam("content") @Valid String content,
                                                           @RequestParam("totalPrice") @Valid BigDecimal totalPrice,
                                                           @RequestParam("spuId") @Valid String spuId,
                                                           @RequestParam("uid") @Valid String uid) {
        return ResponseDto.success(goblinCouponService.useCoupon(ucouponId, content, totalPrice, spuId, uid));
    }

    @PostMapping("backCoupon")
    @ApiOperation("退券")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<Boolean> checkOrderResult(@RequestBody @Valid List<BackCouponParam> backCouponParam) {
        if (backCouponParam.size() > 100) {
            return ResponseDto.failure("数据量不得大于100");
        }
        return ResponseDto.success(goblinCouponService.backCoupon(backCouponParam));
    }
}
