package com.liquidnet.commons.lang.util;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;

public enum PrimitiveUtil {

	/** boolean */
	BOOLEAN	(boolean.class, Boolean.class),
	/** byte */
	BYTE		(byte.class,    Byte.class),
	/** char */
	CHARACTER	(char.class,    Character.class),
	/** short */
	SHORT		(short.class,  Short.class),
	/** int */
	INTEGER	(int.class,    Integer.class),
	/** long */
	LONG		(long.class,   Long.class),
	/** float */
	FLOAT		(float.class,  Float.class),
	/** double */
	DOUBLE		(double.class, Double.class),
	;

	private static final Map<Class<?>, PrimitiveUtil> primitiveCache;
	private static final Map<Class<?>, PrimitiveUtil> wrapperCache;
	private static final Map<Class<?>, PrimitiveUtil> allCache;
	static {
		Map<Class<?>, PrimitiveUtil> pCache = new LinkedHashMap<>();
		Map<Class<?>, PrimitiveUtil> wCache = new LinkedHashMap<>();
		Map<Class<?>, PrimitiveUtil> aCache = new LinkedHashMap<>();
		for (PrimitiveUtil type : values()) {
			pCache.put(type.primitiveClass, type);
			wCache.put(type.wrapperClass, type);
		}
		aCache.putAll(pCache);
		aCache.putAll(wCache);

		primitiveCache = Collections.unmodifiableMap(pCache);
		wrapperCache = Collections.unmodifiableMap(wCache);
		allCache = Collections.unmodifiableMap(aCache);
	}

	/** 基础类型类对象 */
	private final Class<?> primitiveClass;
	/** Wrapper类型类对象 */
	private final Class<?> wrapperClass;


	private PrimitiveUtil(Class<?> primitiveClass, Class<?> wrapperClass) {
		this.primitiveClass = primitiveClass;
		this.wrapperClass = wrapperClass;
	}


	/**
	 * 是否基本类型
	 * @param cls 类对象
	 * @return
	 * <li>true：基本类型</li>
	 * <li>false：非基本类型</li>
	 */
	public static boolean isPrimitiveType(Class<?> cls) {
		return primitiveCache.containsKey(cls);
	}

	/**
	 * 是否Wrapper类型
	 * @param cls 类对象
	 * @return
	 * <li>true：Wrapper类型</li>
	 * <li>false：非Wrapper类型</li>
	 */
	public static boolean isWrapperType(Class<?> cls) {
		return wrapperCache.containsKey(cls);
	}

	public static boolean isSimple(Class<?> cls) {
		return allCache.containsKey(cls);
	}

	/**
	 * 根据基本类型或Wrapper类型取得Java基本类型枚举
	 * @param cls 基本类型或Wrapper类型
	 * @return 类型对应的Java基本类型枚举
	 */
	public static PrimitiveUtil getType(Class<?> cls) {
		return allCache.get(cls);
	}



}
