package com.liquidnet.service.sequence.service.processor;

import com.liquidnet.service.sequence.service.IDGenerator;
import com.liquidnet.service.sequence.util.SystemClock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;

@Service
public class SysSnowflakeProcessor extends IDGenerator {
    private static final Logger log = LoggerFactory.getLogger(SysSnowflakeProcessor.class);
    private static final long twepoch = 1605456000000L;
    private static final long workerIdBits = 5L;
    private static final long dataCenterIdBits = 5L;
    //// 最大支持机器节点数0~31，一共32个
    // 最大支持数据中心节点数0~31，一共32个
    @SuppressWarnings({"PointlessBitwiseExpression", "FieldCanBeLocal"})
    private static final long maxWorkerId = -1L ^ (-1L << workerIdBits);
    @SuppressWarnings({"PointlessBitwiseExpression", "FieldCanBeLocal"})
    private static final long maxDataCenterId = -1L ^ (-1L << dataCenterIdBits);
    // 序列号12位
    private static final long sequenceBits = 12L;
    // 机器节点左移12位
    private static final long workerIdShift = sequenceBits;
    // 数据中心节点左移17位
    private static final long dataCenterIdShift = sequenceBits + workerIdBits;
    // 时间毫秒数左移22位
    private static final long timestampLeftShift = sequenceBits + workerIdBits + dataCenterIdBits;
    @SuppressWarnings({"PointlessBitwiseExpression", "FieldCanBeLocal"})
    private static final long sequenceMask = -1L ^ (-1L << sequenceBits);// 4095

    private static final long workerId;
    private static final long dataCenterId;
    private static final boolean useSystemClock;
    private long sequence = 0L;
    private long lastTimestamp = -1L;

    static {
        workerId = getMachineNum() & maxWorkerId;
        dataCenterId = 1;
        useSystemClock = true;
    }

    /* ---------------------------------------------------------------------------------------- */

    @Override
    public synchronized Long nextId(String key) {
        long timestamp = genTime();
        if (timestamp < lastTimestamp) {
            if (lastTimestamp - timestamp < 2000) {
                // 容忍2秒内的回拨，避免NTP校时造成的异常
                timestamp = lastTimestamp;
            } else {
                // 如果服务器时间有问题(时钟后退) 报错。
                throw new IllegalStateException(String.format("Clock moved backwards. Refusing to generate id for %sms", lastTimestamp - timestamp));
            }
        }

        if (timestamp == lastTimestamp) {
            sequence = (sequence + 1) & sequenceMask;
            if (sequence == 0) {
                timestamp = tilNextMillis(lastTimestamp);
            }
        } else {
            sequence = 0L;
        }

        lastTimestamp = timestamp;

        return ((timestamp - twepoch) << timestampLeftShift) | (dataCenterId << dataCenterIdShift) | (workerId << workerIdShift) | sequence;
    }

    private long genTime() {
        return useSystemClock ? SystemClock.now() : System.currentTimeMillis();
    }

    private long tilNextMillis(long lastTimestamp) {
        long timestamp = genTime();
        // 循环直到操作系统时间戳变化
        while (timestamp == lastTimestamp) {
            timestamp = genTime();
        }
        if (timestamp < lastTimestamp) {
            // 如果发现新的时间戳比上次记录的时间戳数值小，说明操作系统时间发生了倒退，报错
            throw new IllegalStateException(
                    String.format("Clock moved backwards. Refusing to generate id for %sms", lastTimestamp - timestamp));
        }
        return timestamp;
    }

    private static long getMachineNum(){
        long machinePiece;
        StringBuilder sb = new StringBuilder();
        Enumeration<NetworkInterface> e = null;
        try {
            e = NetworkInterface.getNetworkInterfaces();

            while (e.hasMoreElements()) {
                NetworkInterface ni = e.nextElement();
                sb.append(ni.toString());
            }
        } catch (SocketException e1) {
            e1.printStackTrace();
        }
        machinePiece = sb.toString().hashCode();
        log.info("MachineNum={}, MachineCodeString={}", machinePiece, sb);
        return machinePiece;
    }

    /* ---------------------------------------------------------------------------------------- */

    public long getWorkerId(long id) {
        return id >> workerIdShift & ~(-1L << workerIdBits);
    }

    public long getDataCenterId(long id) {
        return id >> dataCenterIdShift & ~(-1L << dataCenterIdBits);
    }

    public long getGenerateDateTime(long id) {
        return (id >> timestampLeftShift & ~(-1L << 41L)) + twepoch;
    }

    /* ---------------------------------------------------------------------------------------- */
}
