package com.liquidnet.service.goblin.util;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.liquidnet.common.cache.redis.util.RedisUtil;
import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.commons.lang.util.RandomUtil;
import com.liquidnet.service.goblin.constant.GoblinRedisConst;
import com.liquidnet.service.goblin.constant.GoblinStatusConst;
import com.liquidnet.service.goblin.dto.GoblinStoreMarketDto;
import com.liquidnet.service.goblin.dto.manage.vo.GoblinMgtCategorySpecVo;
import com.liquidnet.service.goblin.dto.vo.*;
import com.liquidnet.service.goblin.entity.GoblinFrontBanner;
import com.liquidnet.service.goblin.entity.GoblinFrontHotWord;
import com.liquidnet.service.goblin.entity.GoblinFrontNavigation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.liquidnet.service.goblin.constant.GoblinRedisConst.REDIS_GOBLIN_TEMP_COUPON_MARKET;

@Slf4j
@Component
public class GoblinRedisUtils {
    @Autowired
    RedisUtil redisUtil;
    @Autowired
    GoblinMongoUtils goblinMongoUtils;

    private int randomMax = 10;

    public void del(String... keys) {
        redisUtil.del(keys);
    }

    /* ---------------------------------------- 预约相关 ---------------------------------------- */

    /**
     * 该skuId已开启预约
     */
    public void setAnticipateValue(String antId, String skuId){
        redisUtil.set(GoblinRedisConst.ANTICIPATE_SKUID.concat(antId).concat(skuId),1);
    }

    /**
     * 删除skuId已开启预约
     */
    public void delAnticipateValue(String antId, String skuId){
        redisUtil.del(GoblinRedisConst.ANTICIPATE_SKUID.concat(antId).concat(skuId));
    }



    /**
     * 获取关联
     */
    public boolean getAnticipate(String antId, String skuId){
       return redisUtil.get(GoblinRedisConst.ANTICIPATE_SKUID.concat(antId).concat(skuId)) != null;
    }

    public GoblinGoodAnticipateUserVo getUserAboutAut(String skuId, String uid) {
        return (GoblinGoodAnticipateUserVo) redisUtil.get(GoblinRedisConst.USER_ANTICIPATE_STATE.concat(skuId).concat(uid));
    }

    /**
     * 新增用户预约
     */
    public void setUserAboutSku(GoblinGoodAnticipateUserVo user) {
        goblinMongoUtils.setGoblinGoodsAnticipateUserVo(user);
        redisUtil.set(GoblinRedisConst.USER_ANTICIPATE_STATE.concat(user.getSkuId()).concat(user.getUid()),user);
    }

    /**
     * 删除用户预约
     */
    public void delUserAboutSku(String skuId, String uid) {
        goblinMongoUtils.delGoodsAnticipateUserVo(uid, skuId);
        redisUtil.del(GoblinRedisConst.USER_ANTICIPATE_STATE.concat(skuId).concat(uid));
    }

    /**
     * 实际预约人数修改
     */
    public void setAutIdPeople(String autId,Integer state) {
        Integer num = (Integer) redisUtil.get(GoblinRedisConst.ANTICIPATE_PEOPLE.concat(autId));
        if (state.equals(GoblinGoodAnticipateUserVo.STATE_VALID)){
            redisUtil.set(GoblinRedisConst.ANTICIPATE_PEOPLE.concat(autId),num==null? 1:num+1);
        }
        if (state.equals(GoblinGoodAnticipateUserVo.STATE_INVALID)&&num!=null){
            redisUtil.set(GoblinRedisConst.ANTICIPATE_PEOPLE.concat(autId),num-1);
        }
    }

    /**
     * 获取实际预约人数修改
     */
    public Integer getAutIdPeople(String autId) {
        return (Integer) redisUtil.get(GoblinRedisConst.ANTICIPATE_PEOPLE.concat(autId));
    }

    /**
     * 删除预约
     */
    public void delGoodsAnticipateVo(String skuId) {
        redisUtil.del(GoblinRedisConst.ANTICIPATE_SKUID.concat(skuId));
    }

    /* ---------------------------------------- sku库存相关 ---------------------------------------- */
    public void setSkuStock(String marketPre, String skuId, Integer stock) {
        String rk = GoblinRedisConst.REAL_STOCK_SKU;
        if (marketPre != null) {
            rk = rk.concat(marketPre + ":");
        }
        rk = rk.concat(skuId);
        redisUtil.set(rk, stock);
    }

    public int getSkuStock(String marketPre, String skuId) {
        String rk = GoblinRedisConst.REAL_STOCK_SKU;
        if (marketPre != null && !marketPre.equals("null")) {
            rk = rk.concat(marketPre + ":");
        }
        rk = rk.concat(skuId);
        Object obj = redisUtil.get(rk);
        if (obj == null) {
            return 0;
        } else {
            return (int) obj;
        }
    }

    public int incrSkuStock(String marketPre, String skuId, Integer stock) {
        String rk = GoblinRedisConst.REAL_STOCK_SKU;
        if (marketPre != null && !marketPre.equals("null")) {
            rk = rk.concat(marketPre + ":");
        }
        rk = rk.concat(skuId);
        return (int) redisUtil.incr(rk, stock);
    }

    public int decrSkuStock(String marketPre, String skuId, Integer stock) {
        String rk = GoblinRedisConst.REAL_STOCK_SKU;
        if (marketPre != null && !marketPre.equals("null")) {
            rk = rk.concat(marketPre + ":");
        }
        rk = rk.concat(skuId);
        return (int) redisUtil.decr(rk, stock);
    }

    //添加 skuId 相关的活动 skuId
    public void addSkuRe(String skuId, String marketSkuId) {
        String rk = GoblinRedisConst.SKU_RELATION.concat(skuId);
        List<String> list = getSkuRe(skuId);
        list.add(marketSkuId);
        redisUtil.set(rk, list);
    }

    //删除 skuId 相关的活动 skuId
    public void removeSkuRe(String skuId, String marketSkuId) {
        String rk = GoblinRedisConst.SKU_RELATION.concat(skuId);
        List<String> list = getSkuRe(skuId);
        list.remove(marketSkuId);
        redisUtil.set(rk, list);
    }

    //获取 skuId 相关的活动 skuId
    public List<String> getSkuRe(String skuId) {
        String rk = GoblinRedisConst.SKU_RELATION.concat(skuId);
        Object obj = redisUtil.get(rk);
        if (obj == null) {
            return CollectionUtil.arrayListString();
        } else {
            return ((List<String>) obj).stream().distinct().collect(Collectors.toList());
        }
    }

    /* ---------------------------------------- IOS商品价格集 ---------------------------------------- */

    public ObjectNode getIosProducts() {
        Object obj = redisUtil.get(GoblinRedisConst.BASIC_LIB_IOS_PRODUCTS);
        if (null != obj) {
            return JsonUtils.fromJson((String) obj, ObjectNode.class);
        } else {
            ObjectNode val = JsonUtils.OM().createObjectNode();
            val.put("NGoods1", 1.00);
            val.put("NGoods3", 3.00);
            val.put("NGoods6", 6.00);
            val.put("NGoods8", 8.00);
            val.put("NGoods12", 12.00);
            val.put("NGoods18", 18.00);
            val.put("NGoods25", 25.00);
            val.put("NGoods28", 28.00);
            val.put("NGoods30", 30.00);
            val.put("NGoods40", 40.00);
            val.put("NGoods45", 45.00);
            val.put("NGoods50", 50.00);
            val.put("NGoods60", 60.00);
            val.put("NGoods68", 68.00);
            val.put("NGoods73", 73.00);
            val.put("NGoods78", 78.00);
            val.put("NGoods88", 88.00);
            val.put("NGoods93", 93.00);
            val.put("NGoods98", 98.00);
            val.put("NGoods108", 108.00);
            val.put("NGoods113", 113.00);
            val.put("NGoods118", 118.00);
            val.put("NGoods123", 123.00);
            val.put("NGoods128", 128.00);
            val.put("NGoods138", 138.00);
            val.put("NGoods148", 148.00);
            val.put("NGoods153", 153.00);
            val.put("NGoods158", 158.00);
            val.put("NGoods163", 163.00);
            val.put("NGoods168", 168.00);
            val.put("NGoods178", 178.00);
            val.put("NGoods188", 188.00);
            val.put("NGoods193", 193.00);
            val.put("NGoods198", 198.00);
            val.put("NGoods208", 208.00);
            val.put("NGoods218", 218.00);
            val.put("NGoods223", 223.00);
            val.put("NGoods228", 228.00);
            val.put("NGoods233", 233.00);
            val.put("NGoods238", 238.00);
            val.put("NGoods243", 243.00);
            val.put("NGoods248", 248.00);
            val.put("NGoods253", 253.00);
            val.put("NGoods258", 258.00);
            val.put("NGoods263", 263.00);
            val.put("NGoods268", 268.00);
            val.put("NGoods273", 273.00);
            val.put("NGoods278", 278.00);
            val.put("NGoods283", 283.00);
            val.put("NGoods288", 288.00);
            val.put("NGoods298", 298.00);
            redisUtil.set(GoblinRedisConst.BASIC_LIB_IOS_PRODUCTS, val.toString());
            return val;
        }
    }

    /* ---------------------------------------- 平台标签数据源 ---------------------------------------- */

    /**
     * 普通商品标签
     * {
     * tagType:0-普通商品标签
     * tagBelong:0-普通标签
     * }
     *
     * @return List<GoblinSelfTagVo>
     */
    public List<GoblinSelfTagVo> getSelfTagVos() {
        String valStr = (String) redisUtil.get(GoblinRedisConst.BASIC_SELF_TAG);
        boolean valStrIsEmptyFlg = StringUtils.isEmpty(valStr);
        List<GoblinSelfTagVo> vos;
        if (valStrIsEmptyFlg) {
            if (!CollectionUtils.isEmpty(vos = goblinMongoUtils.getSelfTagVos("0"))) {
                redisUtil.set(GoblinRedisConst.BASIC_SELF_TAG, JsonUtils.toJson(vos), 60 * 60);
            }
        } else {
            vos = JsonUtils.fromJson(valStr, new TypeReference<List<GoblinSelfTagVo>>() {
            });
        }
        return vos;
    }

    /**
     * 专属标签
     * {
     * tagType:1-音乐人|2-艺术家|3-品牌方|4-厂牌|5-AR
     * tagBelong:1-专属标签
     * }
     *
     * @return List<GoblinSelfTagVo>
     */
    public List<GoblinSelfTagVo> getSelfExtagVos() {
        String valStr = (String) redisUtil.get(GoblinRedisConst.BASIC_SELF_EXTAG);
        List<GoblinSelfTagVo> vos;
        if (StringUtils.isEmpty(valStr)) {
            if (!CollectionUtils.isEmpty(vos = goblinMongoUtils.getSelfTagVos("1"))) {
                redisUtil.set(GoblinRedisConst.BASIC_SELF_EXTAG, JsonUtils.toJson(vos), 60 * 60);
            }
        } else {
            vos = JsonUtils.fromJson(valStr, new TypeReference<List<GoblinSelfTagVo>>() {
            });
        }
        return vos;
    }

    /* ---------------------------------------- 平台分类数据源 ---------------------------------------- */

    public List<GoblinSelfGoodsCategoryVo> getSelfGoodsCategoryVos() {
        String valStr = (String) redisUtil.get(GoblinRedisConst.BASIC_SELF_GOODS_CATEGORY);
        boolean valStrIsEmptyFlg = StringUtils.isEmpty(valStr);
        List<GoblinSelfGoodsCategoryVo> vos;
        if (valStrIsEmptyFlg) {
            if (!CollectionUtils.isEmpty(vos = goblinMongoUtils.getSelfGoodsCategoryVos())) {
                redisUtil.set(GoblinRedisConst.BASIC_SELF_GOODS_CATEGORY, JsonUtils.toJson(vos));
            }
        } else {
            vos = JsonUtils.fromJson(valStr, new TypeReference<List<GoblinSelfGoodsCategoryVo>>() {
            });
        }
        return vos;
    }

    /* ---------------------------------------- 店铺分类数据源 ---------------------------------------- */

    public boolean setStoreGoodsCategoryVos(String storeId, List<GoblinStoreGoodsCategoryVo> vos) {
        return redisUtil.set(GoblinRedisConst.BASIC_STORE_GOODS_CATEGORY.concat(storeId), JsonUtils.toJson(vos));
    }

    public List<GoblinStoreGoodsCategoryVo> getStoreGoodsCategoryVos(String storeId) {
        String rk = GoblinRedisConst.BASIC_STORE_GOODS_CATEGORY.concat(storeId);
        String valStr = (String) redisUtil.get(rk);
        boolean valStrIsEmptyFlg = StringUtils.isEmpty(valStr);
        List<GoblinStoreGoodsCategoryVo> vos = null;
        if (valStrIsEmptyFlg && !CollectionUtils.isEmpty(vos = goblinMongoUtils.getStoreGoodsCategoryVos(storeId))) {
            valStr = JsonUtils.toJson(vos);
            redisUtil.set(rk, valStr);
        } else if (!valStrIsEmptyFlg) {
            vos = JsonUtils.fromJson(valStr, new TypeReference<List<GoblinStoreGoodsCategoryVo>>() {
            });
        }
        return vos;
    }

    /* ---------------------------------------- 平台分类规格数据源 ---------------------------------------- */

    public boolean setCategorySpec(String cateId, GoblinMgtCategorySpecVo vo) {
        return redisUtil.set(GoblinRedisConst.BASIC_SELF_GOODS_CATEGORY_SPEC.concat(cateId), vo);
    }

    public GoblinMgtCategorySpecVo getCategorySpec(String cateId) {
        String rk = GoblinRedisConst.BASIC_SELF_GOODS_CATEGORY_SPEC.concat(cateId);
        GoblinMgtCategorySpecVo vo = (GoblinMgtCategorySpecVo) redisUtil.get(rk);
        if (null == vo && null != (vo = goblinMongoUtils.getCategorySpecVo(cateId))) {
            redisUtil.set(rk, vo);
        }
        return vo;
    }

    /* ---------------------------------------- 服务支持数据源 ---------------------------------------- */

    public List<GoblinServiceSupportVo> getServiceSupportVos() {
        String valStr = (String) redisUtil.get(GoblinRedisConst.BASIC_SERVICE_SUPPORT);
        boolean valStrIsEmptyFlg = StringUtils.isEmpty(valStr);
        List<GoblinServiceSupportVo> vos;
        if (valStrIsEmptyFlg) {
            if (!CollectionUtils.isEmpty(vos = goblinMongoUtils.getMgtServiceSupportVos())) {
                redisUtil.set(GoblinRedisConst.BASIC_SERVICE_SUPPORT, JsonUtils.toJson(vos));
            }
        } else {
            vos = JsonUtils.fromJson(valStr, new TypeReference<List<GoblinServiceSupportVo>>() {
            });
        }
        return vos;
    }

    /* ---------------------------------------- 店铺公告数据源 ---------------------------------------- */

    public GoblinStoreNoticeVo getStoreNoticeVo(String storeId, LocalDateTime nowTime) {
        List<GoblinStoreNoticeVo> noticeVos = this.getStoreNoticeVos(storeId, nowTime);
        int size = noticeVos.size();
        for (int i = size - 1; i >= 0; i--) {
            GoblinStoreNoticeVo noticeVo = noticeVos.get(i);
            if (noticeVo.getReleaseTime().isBefore(nowTime)) {
                noticeVos.removeIf(r -> (
                        (r.getLongLasting().equals("0") && r.getCancellTime().isBefore(nowTime))
                                || r.getReleaseTime().isBefore(noticeVo.getReleaseTime())
                ));
                this.setStoreNoticeVos(storeId, noticeVos);
                return noticeVo;
            }
        }
        return null;
    }

    public List<GoblinStoreNoticeVo> getStoreNoticeVos(String storeId, LocalDateTime nowTime) {
        String rk = GoblinRedisConst.STORE_NOTICE.concat(storeId);
        List<GoblinStoreNoticeVo> vos = (List<GoblinStoreNoticeVo>) redisUtil.get(rk);
        if (CollectionUtils.isEmpty(vos) && !CollectionUtils.isEmpty(vos = goblinMongoUtils.getStoreNoticeVos(storeId, nowTime))) {
//            redisUtil.set(rk, vos);
        }
        return vos;
    }

    public boolean setStoreNoticeVos(String storeId, List<GoblinStoreNoticeVo> vos) {
        return redisUtil.set(GoblinRedisConst.STORE_NOTICE.concat(storeId), vos);
    }

    /* ---------------------------------------- 店铺配置数据源 ---------------------------------------- */

    public List<GoblinStoreConfigVo> getStoreConfigVos(String storeId) {
        String rk = GoblinRedisConst.BASIC_STORE_CONF.concat(storeId);
        List<GoblinStoreConfigVo> vos = (List<GoblinStoreConfigVo>) redisUtil.get(rk);
        if (CollectionUtils.isEmpty(vos) && !CollectionUtils.isEmpty(vos = goblinMongoUtils.getStoreConfigVos(storeId))) {
            redisUtil.set(rk, vos);
        }
        return vos;
    }

    public boolean setStoreConfigVos(String storeId, List<GoblinStoreConfigVo> vos) {
        return redisUtil.set(GoblinRedisConst.BASIC_STORE_CONF.concat(storeId), vos);
    }

    public void delStoreConfigVos(String storeId) {
        redisUtil.del(GoblinRedisConst.BASIC_STORE_CONF.concat(storeId));
    }

    /* ---------------------------------------- 店铺数据源 ---------------------------------------- */

    private boolean setStoreIds(String uid, List<String> storeIds) {
        return redisUtil.set(GoblinRedisConst.BASIC_USTORE.concat(uid), storeIds);
    }

    public boolean addStoreId(String uid, String storeId) {
        List<String> storeIds = this.getStoreIds(uid);
        if (CollectionUtils.isEmpty(storeIds)) {
            storeIds = CollectionUtil.arrayListString();
        }
        storeIds.add(storeId);
        return this.setStoreIds(uid, storeIds);
    }

    public void delStoreIds(String uid) {
        redisUtil.del(GoblinRedisConst.BASIC_USTORE.concat(uid));
    }

    public void delStoreId(String uid, String storeId) {
        List<String> storeIds = this.getStoreIds(uid);
        if (!CollectionUtils.isEmpty(storeIds)) {
            storeIds.removeIf(r -> r.equals(storeId));
            this.setStoreIds(uid, storeIds);
        }
    }

    public List<String> getStoreIds(String uid) {
//        List<String> list = (List<String>) redisUtil.get(GoblinRedisConst.BASIC_USTORE.concat(uid));
//        return CollectionUtils.isEmpty(list) ? Arrays.asList("1") : list;
        return (List<String>) redisUtil.get(GoblinRedisConst.BASIC_USTORE.concat(uid));
    }

    public boolean hasStoreId(String uid, String storeId) {
        List<String> list = this.getStoreIds(uid);
        return !CollectionUtils.isEmpty(list) && list.contains(storeId);
    }

    public boolean setStoreInfoVo(GoblinStoreInfoVo vo) {
        List<String> storeIds = this.getStoreIds(vo.getUid());
        if (CollectionUtils.isEmpty(storeIds)) {
            storeIds = CollectionUtil.arrayListString();
        }
        storeIds.add(vo.getStoreId());
        this.setStoreIds(vo.getUid(), storeIds);
        return redisUtil.set(GoblinRedisConst.BASIC_STORE.concat(vo.getStoreId()), vo);
    }

    public void delStoreInfoVo(String storeId) {
        redisUtil.del(GoblinRedisConst.BASIC_STORE.concat(storeId));
    }

    public GoblinStoreInfoVo getStoreInfoVo(String storeId) {
        String rk = GoblinRedisConst.BASIC_STORE.concat(storeId);
        GoblinStoreInfoVo vo = (GoblinStoreInfoVo) redisUtil.get(rk);
        if (null == vo && null != (vo = goblinMongoUtils.getStoreInfoVo(storeId))) {
            redisUtil.set(rk, vo);
        }
        return vo;
    }

    public GoblinStoreInfoVo getStoreInfoVoByUid(String uid) {
        List<String> storeIds = this.getStoreIds(uid);
        GoblinStoreInfoVo storeInfoVo = null;
        if (CollectionUtils.isEmpty(storeIds)) {
            List<GoblinStoreInfoVo> vos = goblinMongoUtils.getStoreInfoVoByUid(uid);
            if (!CollectionUtils.isEmpty(vos)) {
                storeIds = CollectionUtil.arrayListString();
                for (GoblinStoreInfoVo vo : vos) {
                    storeIds.add(vo.getStoreId());
                }
                this.setStoreIds(uid, storeIds);
                storeInfoVo = vos.get(0);
            }
        } else {
            storeInfoVo = this.getStoreInfoVo(storeIds.get(0));
        }
        return storeInfoVo;
    }

    /* ---------------------------------------- 商品数据源 ---------------------------------------- */

    public boolean setGoodsInfoVo(GoblinGoodsInfoVo vo) {
        return redisUtil.set(GoblinRedisConst.BASIC_GOODS.concat(vo.getSpuId()), vo);
    }

    public void delGoodsInfoVo(String spuId) {
        redisUtil.del(GoblinRedisConst.BASIC_GOODS.concat(spuId));
    }

    public void delGoodsInfoVoByUnShelves(String spuId) {
        redisUtil.del(GoblinRedisConst.BASIC_GOODS_UNSHELVES.concat(spuId));
    }

    /**
     * 商品基础信息
     *
     * @param spuId 商品ID
     * @return GoblinGoodsInfoVo
     */
    public GoblinGoodsInfoVo getGoodsInfoVo(String spuId) {
        String pre = GoblinStatusConst.MarketPreStatus.getPre(spuId);
        if (pre != null && pre.equals(GoblinStatusConst.MarketPreStatus.MARKET_PRE_ZHENGZAI.getValue())) {
            String[] spuSplitArr = spuId.split(GoblinStatusConst.MarketPreStatus.MARKET_PRE_ZHENGZAI.getValue());
            String rk = GoblinRedisConst.BASIC_GOODS.concat(spuSplitArr[0]);
            GoblinGoodsInfoVo vo = (GoblinGoodsInfoVo) redisUtil.get(rk);
            if (null == vo && null != (vo = goblinMongoUtils.getGoodsInfoVo(spuId))) {
                redisUtil.set(rk, vo);
            }
            if (vo == null) {
                return vo;
            }
            String marketrk = GoblinRedisConst.BASIC_GOODS.concat(spuId);
            GoblinGoodsInfoVo marketVo = (GoblinGoodsInfoVo) redisUtil.get(marketrk);
            if (null == marketVo && null != (marketVo = goblinMongoUtils.getGoodsInfoVo(spuId))) {
                redisUtil.set(marketrk, marketVo);
            }
            if (marketVo == null) {
                return marketVo;
            }
            vo.setSpuId(marketVo.getSpuId());
            List<String> skuIdList = CollectionUtil.linkedListString();
            for (String skuIds : vo.getSkuIdList()) {
                skuIdList.add(skuIds.concat(pre).concat(spuSplitArr[1]));
            }
            vo.setSkuIdList(skuIdList);
            vo.setPriceGe(marketVo.getPriceGe());
            vo.setPriceLe(marketVo.getPriceLe());
            vo.setMarketId(marketVo.getMarketId());
            return vo;
        } else {
            String rk = GoblinRedisConst.BASIC_GOODS.concat(spuId);
            GoblinGoodsInfoVo vo = (GoblinGoodsInfoVo) redisUtil.get(rk);
            if (null == vo && null != (vo = goblinMongoUtils.getGoodsInfoVo(spuId))) {
                redisUtil.set(rk, vo);
            }
            return vo;
        }
    }

    public GoblinGoodsInfoVo getGoodsInfoVoByUnShelves(String spuId) {
        String pre = GoblinStatusConst.MarketPreStatus.getPre(spuId);
        if (pre != null && pre.equals(GoblinStatusConst.MarketPreStatus.MARKET_PRE_ZHENGZAI.getValue())) {
            String[] spuSplitArr = spuId.split(GoblinStatusConst.MarketPreStatus.MARKET_PRE_ZHENGZAI.getValue());
            String rk = GoblinRedisConst.BASIC_GOODS_UNSHELVES.concat(spuSplitArr[0]);
            GoblinGoodsInfoVo vo = (GoblinGoodsInfoVo) redisUtil.get(rk);
            if (null == vo && null != (vo = goblinMongoUtils.getGoodsInfoVo(spuId))) {
                redisUtil.set(rk, vo);
            }
            if (vo == null) {
                return vo;
            }
            String marketrk = GoblinRedisConst.BASIC_GOODS_UNSHELVES.concat(spuId);
            GoblinGoodsInfoVo marketVo = (GoblinGoodsInfoVo) redisUtil.get(marketrk);
            if (null == marketVo && null != (marketVo = goblinMongoUtils.getGoodsInfoVo(spuId))) {
                redisUtil.set(marketrk, marketVo);
            }
            if (marketVo == null) {
                return marketVo;
            }
            vo.setSpuId(marketVo.getSpuId());
            List<String> skuIdList = CollectionUtil.linkedListString();
            for (String skuIds : vo.getSkuIdList()) {
                skuIdList.add(skuIds.concat(pre).concat(spuSplitArr[1]));
            }
            vo.setSkuIdList(skuIdList);
            vo.setPriceGe(marketVo.getPriceGe());
            vo.setPriceLe(marketVo.getPriceLe());
            vo.setMarketId(marketVo.getMarketId());
            return vo;
        } else {
            String rk = GoblinRedisConst.BASIC_GOODS_UNSHELVES.concat(spuId);
            GoblinGoodsInfoVo vo = (GoblinGoodsInfoVo) redisUtil.get(rk);
            if (null == vo && null != (vo = goblinMongoUtils.getGoodsInfoVo(spuId))) {
                redisUtil.set(rk, vo);
            }
            return vo;
        }
    }

    public boolean setGoodsSkuInfoVo(GoblinGoodsSkuInfoVo vo) {
        return redisUtil.set(GoblinRedisConst.BASIC_GOODS_SKU.concat(vo.getSkuId()), vo);
    }

    public void delGoodsSkuInfoVo(String skuId) {
        redisUtil.del(GoblinRedisConst.BASIC_GOODS_SKU.concat(skuId));
    }

    public void delGoodsSkuInfoVoByUnShelves(String skuId) {
        redisUtil.del(GoblinRedisConst.BASIC_GOODS_SKU_UNSHELVES.concat(skuId));
    }

    /**
     * 单品信息
     *
     * @param skuId 单品ID
     * @return GoblinGoodsSkuInfoVo
     */
    public GoblinGoodsSkuInfoVo getGoodsSkuInfoVo(String skuId) {
        String pre = GoblinStatusConst.MarketPreStatus.getPre(skuId);
        if (pre != null && pre.equals(GoblinStatusConst.MarketPreStatus.MARKET_PRE_ZHENGZAI.getValue())) {

            String rk = GoblinRedisConst.BASIC_GOODS_SKU.concat(skuId.split(GoblinStatusConst.MarketPreStatus.MARKET_PRE_ZHENGZAI.getValue())[0]);
            GoblinGoodsSkuInfoVo vo = (GoblinGoodsSkuInfoVo) redisUtil.get(rk);
            if (null == vo && null != (vo = goblinMongoUtils.getGoodsSkuInfoVo(skuId))) {
                redisUtil.set(rk, vo);
            }
            if (vo == null) {
                return vo;
            }
            String marketrk = GoblinRedisConst.BASIC_GOODS_SKU.concat(skuId);
            GoblinGoodsSkuInfoVo marketVo = (GoblinGoodsSkuInfoVo) redisUtil.get(marketrk);
            if (null == marketVo && null != (marketVo = goblinMongoUtils.getGoodsSkuInfoVo(skuId))) {
                redisUtil.set(marketrk, marketVo);
            }
            if (marketVo == null) {
                return marketVo;
            }

            vo.setSpuId(marketVo.getSpuId());
            vo.setSkuId(marketVo.getSkuId());
            vo.setPrice(marketVo.getPrice());
            vo.setPriceMember(marketVo.getPriceMember());
            vo.setSkuStock(marketVo.getSkuStock());
            vo.setBuyLimit(marketVo.getBuyLimit());
            vo.setBuyRoster(marketVo.getBuyRoster());
            vo.setBuyFactor(marketVo.getBuyFactor());
            vo.setDelFlg("0");
            vo.setMarketId(marketVo.getMarketId());
            vo.setCreatedAt(LocalDateTime.now());
            return vo;
        } else {
            String rk = GoblinRedisConst.BASIC_GOODS_SKU.concat(skuId);
            GoblinGoodsSkuInfoVo vo = (GoblinGoodsSkuInfoVo) redisUtil.get(rk);
            if (null == vo && null != (vo = goblinMongoUtils.getGoodsSkuInfoVo(skuId))) {
                redisUtil.set(rk, vo);
            }
            return vo;
        }
    }

    public GoblinGoodsSkuInfoVo getGoodsSkuInfoVoByUnShelves(String skuId) {
        String pre = GoblinStatusConst.MarketPreStatus.getPre(skuId);
        if (pre != null && pre.equals(GoblinStatusConst.MarketPreStatus.MARKET_PRE_ZHENGZAI.getValue())) {
            String rk = GoblinRedisConst.BASIC_GOODS_SKU_UNSHELVES.concat(skuId.split(GoblinStatusConst.MarketPreStatus.MARKET_PRE_ZHENGZAI.getValue())[0]);
            GoblinGoodsSkuInfoVo vo = (GoblinGoodsSkuInfoVo) redisUtil.get(rk);
            if (null == vo && null != (vo = goblinMongoUtils.getGoodsSkuInfoVo(skuId))) {
                redisUtil.set(rk, vo);
            }
            if (vo == null) {
                return vo;
            }
            String marketrk = GoblinRedisConst.BASIC_GOODS_SKU_UNSHELVES.concat(skuId);
            GoblinGoodsSkuInfoVo marketVo = (GoblinGoodsSkuInfoVo) redisUtil.get(marketrk);
            if (null == marketVo && null != (marketVo = goblinMongoUtils.getGoodsSkuInfoVo(skuId))) {
                redisUtil.set(marketrk, marketVo);
            }
            if (marketVo == null) {
                return marketVo;
            }

            vo.setSpuId(marketVo.getSpuId());
            vo.setSkuId(marketVo.getSkuId());
            vo.setPrice(marketVo.getPrice());
            vo.setPriceMember(marketVo.getPriceMember());
            vo.setSkuStock(marketVo.getSkuStock());
            vo.setBuyLimit(marketVo.getBuyLimit());
            vo.setBuyRoster(marketVo.getBuyRoster());
            vo.setBuyFactor(marketVo.getBuyFactor());
            vo.setMarketId(marketVo.getMarketId());
            vo.setCreatedAt(LocalDateTime.now());
            return vo;
        } else {
            String rk = GoblinRedisConst.BASIC_GOODS_SKU_UNSHELVES.concat(skuId);
            GoblinGoodsSkuInfoVo vo = (GoblinGoodsSkuInfoVo) redisUtil.get(rk);
            if (null == vo && null != (vo = goblinMongoUtils.getGoodsSkuInfoVo(skuId))) {
                redisUtil.set(rk, vo);
            }
            return vo;
        }
    }

//    public GoblinGoodsSkuInfoVo getMgtGoodsSkuInfoVo(String skuId) {
//        String rk = GoblinRedisConst.BASIC_GOODS_SKU.concat(skuId);
//        GoblinGoodsSkuInfoVo vo = (GoblinGoodsSkuInfoVo) redisUtil.get(rk);
//        if (null == vo && null != (vo = goblinMongoUtils.getMgtGoodsSkuInfoVo(skuId))) {
//            redisUtil.set(rk, vo);
//        }
//        return vo;
//    }


    /* ----------------------------------------  ---------------------------------------- */

    /**
     * 覆盖 商铺活动vo
     *
     * @param storeMarketId
     * @param vo
     * @return
     */
    public boolean setGoblinStoreMarketVo(String storeMarketId, GoblinStoreMarketVo vo) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_STORE_MARKET.concat(storeMarketId);
        return redisUtil.set(redisKey, vo);
    }

    /**
     * 获取 商铺活动vo
     *
     * @param storeMarketId
     * @return
     */
    public GoblinStoreMarketVo getGoblinStoreMarketVo(String storeMarketId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_STORE_MARKET.concat(storeMarketId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            GoblinStoreMarketVo vo = goblinMongoUtils.getStoreMarket(storeMarketId);
            setGoblinStoreMarketVo(storeMarketId, vo);
            return vo;
        } else {
            return (GoblinStoreMarketVo) obj;
        }
    }

    /**
     * 删除 商铺活动vo
     *
     * @param storeMarketId
     * @return
     */
    public void delGoblinStoreMarketVo(String storeMarketId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_STORE_MARKET.concat(storeMarketId);
        redisUtil.del(redisKey);
    }


    /**
     * 获取 配置的商铺id
     *
     * @param marketId
     * @param storeId
     */
    public List<GoblinStoreMarketIsConfigVo> getStoreMarketIsConfig(String marketId, String storeId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_STORE_MARKET_ISCONFIG.concat(marketId).concat(":store_id:" + storeId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return new ArrayList();
        } else {
            return (List<GoblinStoreMarketIsConfigVo>) obj;
        }
    }

    public void addStoreMarketIsConfig(String marketId, String storeId, String spuId, String marketSpuId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_STORE_MARKET_ISCONFIG.concat(marketId).concat(":store_id:" + storeId);
        List<GoblinStoreMarketIsConfigVo> voList = getStoreMarketIsConfig(marketId, storeId);
        GoblinStoreMarketIsConfigVo vo = GoblinStoreMarketIsConfigVo.getNew();
        vo.setMarketSpuId(marketSpuId);
        vo.setSpuId(spuId);
        voList.add(vo);
        redisUtil.set(redisKey, voList);
    }

    public void delStoreMarketIsConfig(String marketId, String storeId, String spuId, String marketSpuId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_STORE_MARKET_ISCONFIG.concat(marketId).concat(":store_id:" + storeId);
        List<GoblinStoreMarketIsConfigVo> voList = getStoreMarketIsConfig(marketId, storeId);
        for (int i = 0; i < voList.size(); i++) {
            String itemSpuId = voList.get(i).getSpuId();
            if (spuId.equals(itemSpuId)) {
                voList.remove(i);
            }
        }
        redisUtil.set(redisKey, voList);
    }

    /**
     * 添加 活动内商品款式关联关系
     *
     * @param marketPre
     * @param marketId
     * @param vo
     */
    public void addMarketRelation(String marketPre, String marketId, GoblinMarketRelationVo vo) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_MARKET_RELATION.concat(marketPre).concat(marketId);
        List<GoblinMarketRelationVo> voList = getMarketRelation(marketPre, marketId);
        voList.add(vo);
        redisUtil.set(redisKey, voList);
    }

    /**
     * 移除 活动内商品款式关联关系
     *
     * @param marketPre
     * @param marketId
     */
    public void removeMarketRelation(String marketPre, String marketId, String spuId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_MARKET_RELATION.concat(marketPre).concat(marketId);
        List<GoblinMarketRelationVo> voList = getMarketRelation(marketPre, marketId);
        for (GoblinMarketRelationVo item : voList) {
            if (item.getSpuId().equals(spuId)) {
                voList.remove(item);
                break;
            }
        }
        redisUtil.set(redisKey, voList);
    }

    /**
     * 获取 活动内商品款式关联关系
     *
     * @param marketPre
     * @param marketId
     */
    public List<GoblinMarketRelationVo> getMarketRelation(String marketPre, String marketId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_MARKET_RELATION.concat(marketPre).concat(marketId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return new ArrayList();
        } else {
            return (List<GoblinMarketRelationVo>) obj;
        }
    }

    /**
     * 获取 当前商铺能参与的官方活动id列表
     *
     * @param storeId
     * @return
     */
    public List<String> getStoreZhengzaiRelation(String storeId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_STORE_SELF_RELATION.concat(storeId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return CollectionUtil.arrayListString();
        } else {
            return (List<String>) obj;
        }
    }

    /**
     * 正在下单 可参加商户关系 详情
     *
     * @param selfMarketId
     * @param storeId
     */
    public GoblinMarketingZhengzaiRelationVo getZhengzaiRelation(String selfMarketId, String storeId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ZHENGZAI_RELATION.concat(selfMarketId).concat(":store_id:" + storeId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (GoblinMarketingZhengzaiRelationVo) obj;
        }
    }

    /**
     * 正在下单 可参加商户 详情
     *
     * @param selfMarketId
     */
    public GoblinSelfMarketingVo getSelfMarket(String selfMarketId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_SELF_MARKET.concat(selfMarketId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (GoblinSelfMarketingVo) obj;
        }
    }

    //获取全部正在下单的活动
    public List<GoblinSelfMarketingVo> getGoblinSelfMarketingVoList() {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ZHENGZAI_LIST;
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return goblinMongoUtils.getGoblinSelfMarketingVoList();
        } else {
            return (List<GoblinSelfMarketingVo>) obj;
        }
    }

    //获取 用户sku购买个数
    public Integer getSkuCountByUid(String uid, String skuId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_BUY_COUNT.concat(uid + ":skuId:" + skuId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return 0;
        } else {
            return (Integer) obj;
        }
    }

    //获取 用户是否可购买
    public boolean getSkuCanBuyByUid(String mobile, String skuId) {
        String redisKey = GoblinRedisConst.REDIS_CAN_BUY.concat(skuId + ":").concat(mobile);
        return redisUtil.hasKey(redisKey);
    }

    // 增加 用户sku购买个数
    public int incrSkuCountByUid(String uid, String skuId, int number) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_BUY_COUNT.concat(uid + ":skuId:" + skuId);
        return (int) redisUtil.incr(redisKey, number);
    }

    // 减少 用户sku购买个数
    public int decrSkuCountByUid(String uid, String skuId, int number) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_BUY_COUNT.concat(uid + ":skuId:" + skuId);
        return (int) redisUtil.decr(redisKey, number);
    }

    //获取 sku销量
    public Integer getSkuSaleCount(String skuId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_SALE_COUNT.concat(skuId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (Integer) obj;
        }
    }

    // 增加 sku销量
    public int incrSkuSaleCount(String spuId, String skuId, int number) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_SALE_COUNT.concat(skuId);
        incrSpuSaleCount(spuId, number);
        return (int) redisUtil.incr(redisKey, number);
    }

    // 减少 sku销量
    public int decrSkuSaleCount(String spuId, String skuId, int number) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_SALE_COUNT.concat(skuId);
        decrSpuSaleCount(spuId, number);
        return (int) redisUtil.decr(redisKey, number);
    }

    //获取 spu销量
    public Integer getSpuSaleCount(String spuId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_SALE_SPU_COUNT.concat(spuId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (Integer) obj;
        }
    }

    // 增加 spu销量
    private int incrSpuSaleCount(String spuId, int number) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_SALE_SPU_COUNT.concat(spuId);
        return (int) redisUtil.incr(redisKey, number);
    }

    // 减少 spu销量
    private int decrSpuSaleCount(String spuId, int number) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_SALE_SPU_COUNT.concat(spuId);
        return (int) redisUtil.decr(redisKey, number);
    }

    // 赋值 订单相关vo
    public void setGoblinOrder(String orderId, GoblinStoreOrderVo vo) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER.concat(orderId);
        redisUtil.set(redisKey, vo);
    }


    // 赋值 订单相关Skuvo
    public void setGoblinOrderSku(String orderSkuId, GoblinOrderSkuVo vo) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_SKU.concat(orderSkuId);
        redisUtil.set(redisKey, vo);
    }

    // 赋值 快递vo
    public void addGoblinMail(String orderId, GoblinMailVo vo) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_MAIL.concat(orderId);
        List<GoblinMailVo> list = getGoblinMail(orderId);
        list.add(vo);
        redisUtil.set(redisKey, list);
    }

    // 修改 快递vo
    public void changeGoblinMail(String orderId, GoblinMailVo vo) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_MAIL.concat(orderId);
        List<GoblinMailVo> list = getGoblinMail(orderId);
        for (GoblinMailVo item : list) {
            if (item.getMailId().equals(vo.getMailId())) {
                item.setMailNo(vo.getMailNo());
                break;
            }
        }
        redisUtil.set(redisKey, list);
    }

    // 获取 快递vo
    public List<GoblinMailVo> getGoblinMail(String orderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_MAIL.concat(orderId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return ObjectUtil.goblinMailVo();
        } else {
            return (List<GoblinMailVo>) obj;
        }
    }


    // 获取 订单相关vo
    public GoblinStoreOrderVo getGoblinOrder(String orderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER.concat(orderId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (GoblinStoreOrderVo) obj;
        }
    }

    // 获取 nft订单相关vo
    public GoblinNftOrderVo getGoblinNftOrder(String orderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_NFT_ORDER_INFO.concat(orderId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (GoblinNftOrderVo) obj;
        }
    }

    // 获取 订单相关Skuvo
    public GoblinOrderSkuVo getGoblinOrderSkuVo(String orderSkuId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_SKU.concat(orderSkuId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (GoblinOrderSkuVo) obj;
        }
    }

    // 获取 未支付订单id列表
    public List<String> getUnPayOrder(String randomKey) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_UN_PAY.concat(randomKey);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return CollectionUtil.arrayListString();
        } else {
            return (List<String>) obj;
        }
    }

    // 添加 未支付订单id列表 返回 radomKey
    public int addUnPayOrder(String orderId) {
        int randomKey = RandomUtil.getRandomInt(0, randomMax);
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_UN_PAY.concat(randomKey + "");
        List<String> list = getUnPayOrder(randomKey + "");
        list.add(orderId);
        redisUtil.set(redisKey, list);
        return randomKey;
    }

    // 删除 未支付订单id列表
    public void removeGoblinOrder(String randomKey, String orderId) {
        if (randomKey == null) {
            for (int i = 0; i < randomMax; i++) {
                randomKey = i + "";
                String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_UN_PAY.concat(randomKey);
                List<String> list = getUnPayOrder(randomKey);
                if (list.contains(orderId)) {
                    list.remove(orderId);
                    redisUtil.set(redisKey, list);
                    break;
                }
            }
        } else {
            String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_UN_PAY.concat(randomKey + "");
            List<String> list = getUnPayOrder(randomKey);
            list.remove(orderId);
            redisUtil.set(redisKey, list);
        }
    }

    // 获取 订单id列表
    public List<String> getOrderList(String uid) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_LIST.concat(uid);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return CollectionUtil.arrayListString();
        } else {
            return (List<String>) obj;
        }
    }

    // 获取 nft订单id列表
    public List<String> getNftOrderList(String uid) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_NFT_ORDER_USER_ID_LIST.concat(uid);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return CollectionUtil.arrayListString();
        } else {
            return (List<String>) obj;
        }
    }

    // 添加 订单id列表
    public void addOrderList(String uid, String orderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_LIST.concat(uid);
        List<String> list = getOrderList(uid);
        list.add(orderId);
        redisUtil.set(redisKey, list);
    }

    // 删除 订单id列表
    public void removeOrderList(String uid, String orderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_LIST.concat(uid);
        List<String> list = getOrderList(uid);
        list.remove(orderId);
        redisUtil.set(redisKey, list);
    }

    // 获取 订单id列表
    public List<String> getZhengzaiOrderList(String uid) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_ZHENGZAI_LIST.concat(uid);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return CollectionUtil.arrayListString();
        } else {
            return (List<String>) obj;
        }
    }

    // 添加 订单id列表
    public void addZhengzaiOrderList(String uid, String orderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_ZHENGZAI_LIST.concat(uid);
        List<String> list = getZhengzaiOrderList(uid);
        list.add(orderId);
        redisUtil.set(redisKey, list);
    }

    // 删除 订单id列表
    public void removeZhengzaiOrderList(String uid, String orderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_ZHENGZAI_LIST.concat(uid);
        List<String> list = getZhengzaiOrderList(uid);
        list.remove(orderId);
        redisUtil.set(redisKey, list);
    }


    //主订单下包含的子订单
    public void setMasterCode(String masterCode, String orderIds) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_MASTER.concat(masterCode);
        redisUtil.set(redisKey, orderIds);
    }

    public String[] getMasterCode(String masterCode) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_MASTER.concat(masterCode);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return ((String) obj).split(",");
        }
    }

    //主订单对应的出货吗
    public void setOffCode(String writeOffCode, String masterCode) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_OFFCODE.concat(writeOffCode);
        redisUtil.set(redisKey, masterCode);
    }

    public String[] getOffCode(String writeOffCode) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_OFFCODE.concat(writeOffCode);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return getMasterCode((String) obj);
        }
    }

    //覆盖 退款订单vo
    public void setBackOrderVo(String backOrderId, GoblinBackOrderVo vo) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_BACK.concat(backOrderId);
        redisUtil.set(redisKey, vo);
    }

    //覆盖 获取订单vo
    public GoblinBackOrderVo getBackOrderVo(String backOrderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_ORDER_BACK.concat(backOrderId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (GoblinBackOrderVo) obj;
        }
    }

    // 获取 订单id下的退款订单id
    public List<String> getBackOrderByOrderId(String orderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_BACK_ORDER_ID.concat(orderId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return CollectionUtil.arrayListString();
        } else {
            return (List<String>) obj;
        }
    }

    // 添加 订单id下的退款订单id
    public void addBackOrderByOrderId(String orderId, String backOrderId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_BACK_ORDER_ID.concat(orderId);
        List<String> list = getBackOrderByOrderId(orderId);
        list.add(backOrderId);
        redisUtil.set(redisKey, list);
    }

    //商品上架、下架、删除 调用的方法
    public void deleteKeyForSelectGoods() {
        redisUtil.del(GoblinRedisConst.SELECT_GOODS_SPUIDS);
        redisUtil.del(GoblinRedisConst.SELECT_GOODS_PAGE1);
        redisUtil.del(GoblinRedisConst.SELECT_GOODS_SPUIDS_ISHAVE);
    }

    // banner 降级
    public List<GoblinFrontBanner> getListBanner() {
        List<GoblinFrontBanner> vo = (List<GoblinFrontBanner>) redisUtil.get(GoblinRedisConst.FRONT_TOP_BANNER);
        if (null == vo) {
            //降级
            return goblinMongoUtils.getListBanner();
        }
        return vo;
    }

    public List<GoblinFrontBanner> getMiddleBanner() {
        List<GoblinFrontBanner> vo = (List<GoblinFrontBanner>) redisUtil.get(GoblinRedisConst.FRONT_MIDDLE_BANNER);
        if (null == vo) {
            //降级
            return goblinMongoUtils.getMiddleBanner();
        }
        return vo;
    }

    // 热词降级
    public List<GoblinFrontHotWord> getHotWord() {
        List<GoblinFrontHotWord> vo = (List<GoblinFrontHotWord>) redisUtil.get(GoblinRedisConst.FRONT_HOTWORD);
        if (null == vo) {
            //降级
            return goblinMongoUtils.getHotWord();
        }
        return vo;
    }

    // 导航栏 降级
    //  List<GoblinFrontNavigation> list = (List<GoblinFrontNavigation>) redisUtil.get(GoblinRedisConst.FRONT_NAVIGATION);
    public List<GoblinFrontNavigation> getNavigation() {
        List<GoblinFrontNavigation> vo = (List<GoblinFrontNavigation>) redisUtil.get(GoblinRedisConst.FRONT_NAVIGATION);
        if (null == vo) {
            //降级
            return goblinMongoUtils.getNavigation();
        }
        return vo;

    }

    /* ---------------------------------------- 商铺活动:优惠券数据源 ---------------------------------------- */

    public GoblinStoreCouponVo getStoreCouponVo(String storeCouponId) {
        String rk = GoblinRedisConst.STORE_COUPON.concat(storeCouponId);
        GoblinStoreCouponVo vo = (GoblinStoreCouponVo) redisUtil.get(rk);
        if (null == vo && null != (vo = goblinMongoUtils.getStoreCouponVo(storeCouponId))) {
            redisUtil.set(rk, vo);
        }
        return vo;
    }

    public List<String> getStoreCouponSpuIds(String storeCouponId) {
        String rk = GoblinRedisConst.STORE_COUPON_RULE.concat(storeCouponId);
        String valStr = (String) redisUtil.get(rk);
        List<String> strs;
        if (StringUtils.isEmpty(valStr)) {
            GoblinStoreCouponBasicVo storeCouponBasicVo = goblinMongoUtils.getMgtStoreCouponBasicVo(storeCouponId);
            strs = null == storeCouponBasicVo ? null : storeCouponBasicVo.getSpuIdList();
            if (!CollectionUtils.isEmpty(strs)) {
                redisUtil.set(rk, JsonUtils.toJson(strs));
            }
        } else {
            strs = JsonUtils.fromJson(valStr, new TypeReference<List<String>>() {
            });
        }
        return strs;
    }

    /* ---------------------------------------- 商铺活动:优惠券剩余库存 ---------------------------------------- */

    public boolean setStoreCouponStock(String storeCouponId, Integer stock) {
        return redisUtil.set(GoblinRedisConst.STORE_COUPON_STOCK.concat(storeCouponId), stock);
    }

    public void delStoreCouponStock(String storeCouponId) {
        redisUtil.del(GoblinRedisConst.STORE_COUPON_STOCK.concat(storeCouponId));
    }

    public int getStoreCouponStock(String storeCouponId) {
        Object valObj = redisUtil.get(GoblinRedisConst.STORE_COUPON_STOCK.concat(storeCouponId));
        return null == valObj ? 0 : (int) valObj;
    }

    public int incrStoreCouponStock(String storeCouponId, Integer operStock) {
        return (int) redisUtil.incr(GoblinRedisConst.STORE_COUPON_STOCK.concat(storeCouponId), operStock);
    }

    public int decrStoreCouponStock(String storeCouponId, Integer operStock) {
        return (int) redisUtil.decr(GoblinRedisConst.STORE_COUPON_STOCK.concat(storeCouponId), operStock);
    }

    /* ---------------------------------------- 商铺活动 ---------------------------------------- */

    public boolean setStoreMarketDtos(String storeId, List<GoblinStoreMarketDto> dtos) {
        return redisUtil.set(GoblinRedisConst.STORE_MARKETS.concat(storeId), JsonUtils.toJson(dtos));
    }

    public List<GoblinStoreMarketDto> getStoreMarketDtos(String storeId) {
        String rk = GoblinRedisConst.STORE_MARKETS.concat(storeId);
        String valStr = (String) redisUtil.get(rk);
        List<GoblinStoreMarketDto> dtos;
        if (StringUtils.isEmpty(valStr)) {
            // TODO: 2022/2/17 zhanggb mongodb 查取该商品所有参与的活动并整理集合
            dtos = ObjectUtil.getGoblinStoreMarketDtoArrayList();
        } else {
            dtos = JsonUtils.fromJson(valStr, new TypeReference<List<GoblinStoreMarketDto>>() {
            });
        }
        return dtos;
    }

    public boolean addStoreMarketDto(String storeId, GoblinStoreMarketDto dto) {
        List<GoblinStoreMarketDto> dtos = this.getStoreMarketDtos(storeId);
        dtos.add(dto);
        return redisUtil.set(GoblinRedisConst.STORE_MARKETS.concat(storeId), JsonUtils.toJson(dtos));
    }

    public void delStoreMarketDto(String storeId) {
        redisUtil.del(GoblinRedisConst.STORE_MARKETS.concat(storeId));
    }

    public void delStoreMarketDto(String storeId, GoblinStoreMarketDto dto) {
        String rk = GoblinRedisConst.STORE_MARKETS.concat(storeId);
        List<GoblinStoreMarketDto> dtos = this.getStoreMarketDtos(storeId);
        if (!CollectionUtils.isEmpty(dtos)) {
            int beforeSize = dtos.size();
            dtos.removeIf(r -> r.getId().equals(dto.getId()) && r.getType().equals(dto.getType()));
            if (beforeSize > dtos.size()) {
                redisUtil.set(GoblinRedisConst.STORE_MARKETS.concat(storeId), JsonUtils.toJson(dtos));
            }
        }
    }

    //获取艺人标签redis
    public List<GoblinGoodsInfoVo> getMusicTagList(String musicTag, String performanceId) {
        List<GoblinGoodsInfoVo> listVo = ObjectUtil.getGoblinGoodsInfoVos();
//        String rk = GoblinRedisConst.REDIS_ZZ_MUSIC_TAG.concat(musicTag + ":").concat(performanceId);
//        Object obj = redisUtil.get(rk);
//        if (obj == null) {
        listVo = goblinMongoUtils.getMusicTagPGoods(musicTag, performanceId);
//            redisUtil.set(rk, listVo, 60 * 30);
        return listVo;
//        } else {
//            return (List<GoblinGoodsInfoVo>) obj;
//        }
    }

    //删除艺人标签redis
    public void delMusicTagList(String musicTag, String performanceId) {
        String rk = GoblinRedisConst.REDIS_ZZ_MUSIC_TAG.concat(musicTag + ":").concat(performanceId);
        redisUtil.del(rk);
    }

    /* ---------------------------------------- 商品活动 ---------------------------------------- */

    public List<GoblinStoreCouponVo> getGoodsMarketsForCoupon(String spuId, String storeId) {
        List<GoblinStoreMarketDto> storeMarketDtos = this.getStoreMarketDtos(storeId);
        if (CollectionUtils.isEmpty(storeMarketDtos)) {
            return ObjectUtil.getGoblinStoreCouponVoArrayList();
        }
        ArrayList<GoblinStoreCouponVo> storeCouponVos = ObjectUtil.getGoblinStoreCouponVoArrayList();
        storeMarketDtos.forEach(dto -> {// 逐个活动筛选优惠券活动处理
            if (dto.getType() == 1) {// 优惠券活动
                GoblinStoreCouponVo storeCouponVo = this.getStoreCouponVo(dto.getId());
                if (null != storeCouponVo) {
                    if ("0".equals(storeCouponVo.getUseScope())) {
                        storeCouponVos.add(storeCouponVo);
                    } else {
                        List<String> storeCouponSpuIds = this.getStoreCouponSpuIds(dto.getId());
                        if (!CollectionUtils.isEmpty(storeCouponSpuIds) && storeCouponSpuIds.contains(spuId)) {
                            storeCouponVos.add(storeCouponVo);
                        }
                    }
                }
            }
        });
        return storeCouponVos;
    }

    /* ---------------------------------------- 商城:用户优惠券 ---------------------------------------- */

    public boolean setUserCouponVos(String uid, List<GoblinUserCouponVo> vos) {
        return redisUtil.set(GoblinRedisConst.USER_COUPON.concat(uid), JsonUtils.toJson(vos));
    }

    public List<GoblinUserCouponVo> getUserCouponVos(String uid) {
        String rk = GoblinRedisConst.USER_COUPON.concat(uid);
//        long s = System.nanoTime();
        String valStr = (String) redisUtil.get(rk);
//        log.info("RDS.G耗时:{}ms", (System.nanoTime() - s) / 1000000);
        List<GoblinUserCouponVo> vos;
        if (StringUtils.isEmpty(valStr)) {
//            if (!CollectionUtils.isEmpty(vos = goblinMongoUtils.getUserCouponVos(uid))) {
//                redisUtil.set(rk, JsonUtils.toJson(vos));
//            }
            return ObjectUtil.getGoblinUserCouponVo();
        } else {
            vos = JsonUtils.fromJson(valStr, new TypeReference<List<GoblinUserCouponVo>>() {
            });
        }
        return vos;
    }

    public boolean addUserCouponVos(String uid, GoblinUserCouponVo vo) {
        if (null == vo) return false;
        List<GoblinUserCouponVo> vos = this.getUserCouponVos(uid);
        vos.add(vo);
        return redisUtil.set(GoblinRedisConst.USER_COUPON.concat(uid), JsonUtils.toJson(vos));
    }

    /* ----------------------------------------  ---------------------------------------- */

    public boolean addMarketTempCoupon(String marketId, String ucouponId) {
        String redisKey = REDIS_GOBLIN_TEMP_COUPON_MARKET.concat(marketId);
        List<String> list = getMarketTempCoupon(marketId);
        list.add(ucouponId);
        return redisUtil.set(redisKey, list, 60 * 60 * 24 * 60);
    }

    /* ----------------------------------------  临时券  ---------------------------------------- */

    public void delMarketTempCoupon(String performanceId) {
        String redisKey = REDIS_GOBLIN_TEMP_COUPON_MARKET.concat(performanceId);
        redisUtil.del(redisKey);
    }

    public List<String> getMarketTempCoupon(String performanceId) {
        String redisKey = REDIS_GOBLIN_TEMP_COUPON_MARKET.concat(performanceId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return CollectionUtil.arrayListString();
        } else {
            return (List<String>) obj;
        }
    }

    public void setTempCoupon(String ucouponId, TempCouponVo vo) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_TEMP_COUPON.concat(ucouponId);
        redisUtil.set(redisKey, vo, 60 * 60 * 24 * 60);
    }

    public TempCouponVo getTempCoupon(String ucouponId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_TEMP_COUPON.concat(ucouponId);
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (TempCouponVo) obj;
        }
    }

    public List<GoblinNftGoodsSkuListVo> getGoblinNftGoodsInfoListVo() {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_NFT_GOODS_LIST;
        Object obj = redisUtil.get(redisKey);
        if (obj == null) {
            return null;
        } else {
            return (List<GoblinNftGoodsSkuListVo>) obj;
        }
    }

    public void setGoblinNftGoodsInfoListVo(List<GoblinNftGoodsSkuListVo> spuList) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_NFT_GOODS_LIST;
        redisUtil.set(redisKey, spuList);
    }

    public void delGoblinNftGoodsInfoListVo() {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_NFT_GOODS_LIST;
        redisUtil.del(redisKey);
    }

    /* ---------------------------------------- 我的藏品 ---------------------------------------- */

    public List<String> getUserDigitalArtworkIds(String uid) {
        return (List<String>) redisUtil.get(GoblinRedisConst.USER_DIGITAL_ARTWORK_IDS.concat(uid));
    }

    public long markUnboxUserDigitalArtwork(String artworkId) {
        return redisUtil.incr(GoblinRedisConst.USER_DIGITAL_ARTWORK_UNBOX.concat(artworkId), 1);
    }

    public long countUserDigitalArtwork(String uid) {
        Object o = redisUtil.get(GoblinRedisConst.USER_DIGITAL_ARTWORK_CT.concat(uid));
        return Objects.isNull(o) ? 0 : (int) o;
    }

    public GoblinUserDigitalArtworkVo getUserDigitalArtworkVo(String artworkId) {
        return (GoblinUserDigitalArtworkVo) redisUtil.get(GoblinRedisConst.USER_DIGITAL_ARTWORK.concat(artworkId));
    }

    public boolean setUserDigitalArtworkVo(GoblinUserDigitalArtworkVo vo) {
        return redisUtil.set(GoblinRedisConst.USER_DIGITAL_ARTWORK.concat(vo.getArtworkId()), vo);
    }

    /**
     * 盲盒开启后，直接从`我的藏品列表`中移除
     * 此处通过Redis过期删除
     */
    public boolean setexUserDigitalArtworkVo(GoblinUserDigitalArtworkVo vo) {
        return redisUtil.set(GoblinRedisConst.USER_DIGITAL_ARTWORK.concat(vo.getArtworkId()), vo, 259200);
    }

    // 记录用户开通数字账户
    public void setNftNumAccount(String userId) {
        String redisKey = GoblinRedisConst.REDIS_GOBLIN_NFT_NUM_ACCOUNT.concat(userId);
        redisUtil.set(redisKey, 1);
    }

    /**
     *     private int skuType 商品类型[0-常规|1-数字藏品]
     *     private String status 审核状态[0-初始编辑|1-审核中|2-审核不通过|3-审核通过];
     *     private String shelvesStatus 单品上架状态[0-待上架|1-下架|2-违规|3-上架];
     *     private String skuAppear 是否隐藏[0-默认展示|1-隐藏];
     *     private String delFlg 删除标记[0-未删除|1-删除];
     *
     *     private LocalDateTime saleStartTime 开售时间;
     *     private LocalDateTime saleStopTime 停售时间 预留 暂时不做处理;
     *     private String soldoutStatus 是否售罄[0-否|1-是];
     *
     *     private String skuCanbuy 是否购买[0-否|1-是] 这个用来预览 前端自己判断;
     */
    // 获取盲盒下藏品的库存 各种状态下不能算库存的排除掉
    public int getSkuAllStatusStock(GoblinGoodsSkuInfoVo info) {
        if (
            info != null
            && LocalDateTime.now().isAfter(info.getSaleStartTime())
            && (null == info.getSoldoutStatus() || info.getSoldoutStatus().equals("0"))
        ) {// 可以返回库存
            return getSkuStock(null, info.getSkuId());
        } else {// 不计入库存
            return 0;
        }
    }
    // 各种状态下判断藏品是否可以展示
    public boolean getSkuAllStatusShow(GoblinGoodsSkuInfoVo info) {
        if (
            info != null
            && info.getSkuType() == 1
            && info.getStatus().equals("3")
            && info.getShelvesStatus().equals("3")
            && info.getSkuAppear().equals("0")
            && info.getDelFlg().equals("0")
        ) {
            return true;
        } else {
            return false;
        }
    }

    /* ----------------------------------------  ---------------------------------------- */
    /* ----------------------------------------  ---------------------------------------- */
}
