package com.liquidnet.service.candy.service.impl;

import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.IDGenerator;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.base.SqlMapping;
import com.liquidnet.service.base.constant.MQConst;
import com.liquidnet.service.candy.dto.CandyCouponCodeDto;
import com.liquidnet.service.candy.dto.CandyCouponInfoDto;
import com.liquidnet.service.candy.dto.CandyCouponRuleDto;
import com.liquidnet.service.candy.dto.CandyUserCouponBasicDto;
import com.liquidnet.service.candy.param.BackCouponParam;
import com.liquidnet.service.candy.service.ICandyCouponService;
import com.liquidnet.service.candy.util.CouponBaseUtil;
import com.liquidnet.service.candy.util.ObjectUtil;
import com.liquidnet.service.candy.util.QueueUtils;
import com.liquidnet.service.candy.util.RedisDataUtils;
import com.liquidnet.service.candy.vo.CandyCouponPreVo;
import com.liquidnet.service.candy.vo.CandyCouponVo;
import com.liquidnet.service.candy.vo.CandyMyCouponListVo;
import com.liquidnet.service.candy.vo.CandyUseResultVo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 券基础信息 服务实现类
 * </p>
 *
 * @author liquidnet
 * @since 2021-08-18
 */
@Service
public class CandyCouponServiceImpl implements ICandyCouponService {

    @Autowired
    private RedisDataUtils redisDataUtils;
    @Autowired
    private QueueUtils queueUtils;

    @Override
    public CandyMyCouponListVo myCoupon(Integer type) {
        String uid = CurrentUtil.getCurrentUid();
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid);
        CandyMyCouponListVo vo = CandyMyCouponListVo.getNew();
        List<CandyCouponVo> memberCoupon = ObjectUtil.getCandyCouponVoArrayList();
        List<CandyCouponVo> myCoupon = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            if (type == 1) {
                if (dtoItem.getState().equals(3) || dtoItem.getState().equals(5)) {
                    continue;
                }
            } else if (type == 2) {
                if (dtoItem.getState().equals(1)) {
                    continue;
                }
            }
            CandyCouponVo baseVo = CouponBaseUtil.getBaseCouponUserVo(dtoItem);
            if (dtoItem.getExclusive().equals(1)) { //会员券
                memberCoupon.add(baseVo);
            } else {//非会员券
                myCoupon.add(baseVo);
            }
        }
        //排序 可用->金额->类型->过期时间
        memberCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        myCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        vo.setMemberCoupon(memberCoupon);
        vo.setMyCoupon(myCoupon);
        return vo;
    }

    @Override
    public List<CandyCouponPreVo> memberCouponPre(String uid) {
        ArrayList<CandyCouponPreVo> voArrayList = ObjectUtil.getCandyCouponPreVoArrayList();

        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid);
        if (!CollectionUtil.isEmpty(dtoList)) {
            Map<String, Long> mCouponCountMap = dtoList.parallelStream()
                    .filter(r -> (r.getExclusive() == 1 && r.getState() == 1))
                    .collect(Collectors.groupingBy(CandyUserCouponBasicDto::getMcouponId, Collectors.counting()));

            mCouponCountMap.forEach((mCouponId, ct) -> voArrayList.add(CandyCouponPreVo.getNew().setMcouponId(mCouponId).setAvailableCt(ct)));
        }
        return voArrayList;
    }

    @Override
    public List<CandyCouponVo> memberCoupon(String mCouponId) {
        String uid = CurrentUtil.getCurrentUid();
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid);
        List<CandyCouponVo> memberCoupon = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            if (!mCouponId.equals(dtoItem.getMcouponId())) {
                continue;
            }
            CandyCouponVo baseVo = CouponBaseUtil.getBaseCouponUserVo(dtoItem);
            if (dtoItem.getExclusive().equals(1)) { //会员券
                memberCoupon.add(baseVo);
            }
        }
        //排序 可用->金额->类型->过期时间
        memberCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        return memberCoupon;
    }

    @Override
    public CandyMyCouponListVo preUsePerformanceCoupon(BigDecimal priceTotal, String performanceId, String timeId, String ticketId, Integer type) {
        String uid = CurrentUtil.getCurrentUid();
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid);
        CandyMyCouponListVo vo = CandyMyCouponListVo.getNew();
        List<CandyCouponVo> memberCoupon = ObjectUtil.getCandyCouponVoArrayList();
        List<CandyCouponVo> myCoupon = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            if (type == 1) {
                if (dtoItem.getState().equals(3) || dtoItem.getState().equals(5)) {
                    continue;
                }
            } else if (type == 2) {
                if (dtoItem.getState().equals(1)) {
                    continue;
                }
            }
            CandyCouponVo baseVo = CouponBaseUtil.getPerformanceCouponUserVo(dtoItem, priceTotal, performanceId, timeId, ticketId);
            if(baseVo.getCouType().equals(101)){
                continue;
            }
            if (dtoItem.getExclusive().equals(1)) { //会员券
                memberCoupon.add(baseVo);
            } else {//非会员券
                myCoupon.add(baseVo);
            }
        }

        //排序
        memberCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        myCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        vo.setMemberCoupon(memberCoupon);
        vo.setMyCoupon(myCoupon);
        return vo;
    }

    @Override
    public Integer preCanUsePerformanceCoupon(BigDecimal priceTotal, String performanceId, String timeId, String ticketId) {
        String uid = CurrentUtil.getCurrentUid();
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid);
        int canUse = 0;
        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            CandyCouponVo baseVo = CouponBaseUtil.getPerformanceCouponUserVo(dtoItem, priceTotal, performanceId, timeId, ticketId);
            if(baseVo.getCouType().equals(101)){
                continue;
            }
            if (baseVo.getState().equals(1)) { //可用
                canUse = 1;
                break;
            } else {//不可用
                continue;
            }
        }
        return canUse;
    }

    @Override
    public CandyMyCouponListVo preUseGoodCoupon(BigDecimal priceTotal, String goodId, Integer type) {
        String uid = CurrentUtil.getCurrentUid();
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid);
        CandyMyCouponListVo vo = CandyMyCouponListVo.getNew();
        List<CandyCouponVo> memberCoupon = ObjectUtil.getCandyCouponVoArrayList();
        List<CandyCouponVo> myCoupon = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            if (type == 1) {
                if (dtoItem.getState().equals(3) || dtoItem.getState().equals(5)) {
                    continue;
                }
            } else if (type == 2) {
                if (dtoItem.getState().equals(1)) {
                    continue;
                }
            }
            CandyCouponVo baseVo = CouponBaseUtil.getGoodCouponUserVo(dtoItem, priceTotal, goodId, dtoList.size());
            if(baseVo.getCouType().equals(101)){
                continue;
            }
            if (dtoItem.getExclusive().equals(1)) { //会员券
                memberCoupon.add(baseVo);
            } else {//非会员券
                myCoupon.add(baseVo);
            }
        }

        //排序
        memberCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        myCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        vo.setMemberCoupon(memberCoupon);
        vo.setMyCoupon(myCoupon);
        return vo;
    }

    @Override
    public Integer preCanUseGoodCoupon(BigDecimal priceTotal, String goodId) {
        String uid = CurrentUtil.getCurrentUid();
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid);
        int canUse = 0;
        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            CandyCouponVo baseVo = CouponBaseUtil.getGoodCouponUserVo(dtoItem, priceTotal, goodId, dtoList.size());
            if(baseVo.getCouType().equals(101)){
                continue;
            }
            if (baseVo.getState().equals(1)) { //可用
                canUse = 1;
                break;
            } else {//不可用
                continue;
            }
        }
        return canUse;
    }

    @Override
    public List<CandyCouponVo> myAdvanceCoupon(String performanceId) {
        String uid = CurrentUtil.getCurrentUid();
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid);
        List<CandyCouponVo> advanceCoupon = ObjectUtil.getCandyCouponVoArrayList();
        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            if (dtoItem.getBusiType().equals(3)) {
                CandyCouponVo baseVo = CouponBaseUtil.getPerformanceCouponUserVo(dtoItem, BigDecimal.ZERO, performanceId, "-1", "-1");
                if (baseVo.getState().equals(1)) {
                    advanceCoupon.add(baseVo);
                } else {
                    continue;
                }
            } else {
                continue;
            }
        }
        return advanceCoupon;
    }

    @Override
    public Integer stateCoupon(String uCouponId) {
        String uid = CurrentUtil.getCurrentUid();
        CandyUserCouponBasicDto dto = CouponBaseUtil.getSingleDtoByUCouponId(redisDataUtils.getCouponByUid(uid), uCouponId);
        if (dto == null) {
            return null;//券不存在
        }
        return CouponBaseUtil.getCouponState(LocalDateTime.now(), dto.getDuedAt(), dto.getBindAt(), dto.getState());
    }

    @Override
    public CandyUseResultVo useCoupon(String uCouponId, String content, String totalPrice, String performanceId, String timesId, String ticketId, String goodId) {
        String uid = CurrentUtil.getCurrentUid();
        CandyUseResultVo vo = CandyUseResultVo.getNew();
        List<CandyUserCouponBasicDto> dtoList = CouponBaseUtil.useCoupon(redisDataUtils.getCouponByUid(uid), uCouponId, content);
        if (dtoList == null) {
            vo.setCouType(-1);
            vo.setValue(BigDecimal.ZERO);
        } else {
            CandyUserCouponBasicDto dto = CouponBaseUtil.getSingleDtoByUCouponId(redisDataUtils.getCouponByUid(uid), uCouponId);
            if (dto.getBusiType().equals(3)) {
                dto.setCouType(101);
            } else {
                dto.setCouType(dto.getCouType());
            }
            vo.setCouType(dto.getCouType());
            switch (dto.getCouType()) {//券类型[1-代金券｜2-满减券｜3-兑换券｜4-折扣券|101-优先券]
                case 1:
                    vo.setValue(dto.getValFace());
                    break;
                case 2:
                    if (dto.getValOver().compareTo(new BigDecimal(totalPrice)) >= 0) {
                        vo.setValue(dto.getValMinus());
                    } else {
                        vo.setCouType(-1);
                        vo.setValue(BigDecimal.ZERO);
                    }
                    break;
                case 3:
                    vo.setValue(new BigDecimal(totalPrice));
                    break;
                case 4:
                    vo.setValue(dto.getDiscount());
                    break;
                default:
                    vo.setValue(BigDecimal.valueOf(0));
                    break;
            }

            Boolean isTarget = false;
            for (CandyCouponRuleDto ruleItem : dto.getUseRules()) {
                switch (ruleItem.getUseScope()) {
                    case 80://商品
                        isTarget = CouponBaseUtil.isTargetCoupon(ruleItem.getBusiId(), dto.getCouType(), goodId, new BigDecimal(totalPrice), dto.getValOver());
                        break;
                    case 90://演出
                        isTarget = CouponBaseUtil.isTargetCoupon(ruleItem.getBusiId(), dto.getCouType(), performanceId, new BigDecimal(totalPrice), dto.getValOver());
                        break;
                    case 91://场次
                        isTarget = CouponBaseUtil.isTargetCoupon(ruleItem.getBusiId(), dto.getCouType(), timesId, new BigDecimal(totalPrice), dto.getValOver());
                        break;
                    case 92://票
                        isTarget = CouponBaseUtil.isTargetCoupon(ruleItem.getBusiId(), dto.getCouType(), ticketId, new BigDecimal(totalPrice), dto.getValOver());
                        break;
                    case 100://票
                        isTarget = true;
                        break;
                    default:
                        isTarget = false;
                        break;
                }
                if (dto.getUseRules().size() > 1 && dto.getCouType().equals(4)) {
                    isTarget = false;
                }
            }

            if(dto.getBusiType().equals(0)){
                isTarget = true;
            }

            if (!isTarget) {
                vo.setCouType(-1);
                vo.setValue(BigDecimal.ZERO);
            }

            if (vo.getCouType() != -1) {
                redisDataUtils.setCouponByUid(uid, dtoList);
                //入库
                if (dto.getCcouponId() == null) {
                    queueUtils.sendMsgByRedis(
                            MQConst.CandyQueue.COUPON_USE.getKey(),
                            SqlMapping.get("candy_coupon.use",
                                    LocalDateTime.now(), content, uid, uCouponId
                            )
                    );
                } else {
                    queueUtils.sendMsgByRedis(
                            MQConst.CandyQueue.COUPON_USE.getKey(),
                            SqlMapping.get("candy_coupon.use_insert",
                                    uCouponId, uid, dto.getCouponId(), LocalDateTime.now(), content
                            )
                    );
                }
            }
        }
        return vo;
    }

    @Override
    public Boolean useBackCoupon(List<BackCouponParam> backCouponParams) {
        for (BackCouponParam item : backCouponParams) {
            String uid = item.getUid();
            ArrayList<String> uCouponIdList = new ArrayList(Arrays.asList(item.getuCouponIds().split(",")));
            List<CandyUserCouponBasicDto> dtoList = CouponBaseUtil.backCoupon(redisDataUtils.getCouponByUid(uid), uCouponIdList);
            if (dtoList == null) {
                return false;
            } else {
                redisDataUtils.setCouponByUid(uid, dtoList);
                //入库
                LinkedList<String> sqls = CollectionUtil.linkedListString();
                sqls.add(SqlMapping.get("candy_coupon.back"));
                LinkedList<Object[]> sqlsDataA = CollectionUtil.linkedListObjectArr();
                for (String uCouponId : uCouponIdList) {
                    sqlsDataA.add(new Object[]{
                            uid, uCouponId
                    });
                }
                String sqlData = SqlMapping.gets(sqls, sqlsDataA);
                queueUtils.sendMsgByRedis(
                        MQConst.CandyQueue.COUPON_BACK.getKey(),
                        sqlData
                );
            }
        }
        return true;
    }

    @Override
    public ResponseDto<String> receiveCoupon(String ccode) {
        String uid = CurrentUtil.getCurrentUid();
        CandyCouponCodeDto dto = redisDataUtils.getCouponByCode(ccode);
        LocalDateTime now = LocalDateTime.now();

        if (dto == null) {
            return ResponseDto.failure("兑换码有误，请重新输入");
        } else if (dto.getState().equals(0)) {//判断可领
            CandyCouponInfoDto infoDto = redisDataUtils.getCouponInfo(dto.getCouponId());
            if (!CouponBaseUtil.CandyCanReceive(infoDto)) {
                return ResponseDto.failure("兑换码已过期");
            }
            String uCouponId = IDGenerator.get32UUID();
            //构建baseDto
            CandyUserCouponBasicDto baseDto = CandyUserCouponBasicDto.getNew().copyToRedisBaseCoupon(uCouponId, uid, ccode, infoDto, infoDto.getExpireAt());
            //删除code redis
            redisDataUtils.delCouponByCode(ccode);
            //添加baseDto -> baseDtoList
            redisDataUtils.addCouponByUid(uid, baseDto);
            //入库
            queueUtils.sendMsgByRedis(
                    MQConst.CandyQueue.COUPON_RECEIVE.getKey(),
                    SqlMapping.get("candy_coupon.receive",
                            uCouponId, uid, 1, infoDto.getCouponId(), ccode, now, now, infoDto.getExpireAt()
                    )
            );
            return ResponseDto.success(uCouponId);
        }else if(dto.getState().equals(1)){
            return ResponseDto.failure("兑换已使用");
        } else {
            return ResponseDto.failure("兑换失败，请稍后再试");
        }
    }

    @Override
    public ResponseDto<CandyCouponVo> receiveCouponDetails(String ccode) {
        String uid = CurrentUtil.getCurrentUid();
        CandyCouponCodeDto dto = redisDataUtils.getCouponByCode(ccode);
        if (dto == null) {
            return ResponseDto.failure("兑换码有误，请重新输入");
        } else if (dto.getState().equals(0)) {//判断可领
            CandyCouponInfoDto infoDto = redisDataUtils.getCouponInfo(dto.getCouponId());
            if (!CouponBaseUtil.CandyCanReceive(infoDto)) {
                return ResponseDto.failure("兑换码已过期");
            }
            String uCouponId = IDGenerator.get32UUID();
            //构建baseDto
            CandyUserCouponBasicDto baseDto = CandyUserCouponBasicDto.getNew().copyToRedisBaseCoupon(uCouponId, uid, ccode, infoDto,LocalDateTime.now().plusDays(infoDto.getRedeemValidity()));
            return ResponseDto.success(CouponBaseUtil.getBaseCouponUserVo(baseDto));
        } else {
            return ResponseDto.failure("兑换失败，请稍后再试");
        }
    }

    @Override
    public ResponseDto<List<CandyCouponVo>> couponListById(String uCouponIds) {
        String uid = CurrentUtil.getCurrentUid();
        ArrayList<String> uCouponIdList = new ArrayList(Arrays.asList(uCouponIds.split(",")));
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid);
        List<CandyCouponVo> vo = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            if(uCouponIdList.contains(dtoItem.getUcouponId())) {
                CandyCouponVo baseVo = CouponBaseUtil.getBaseCouponUserVo(dtoItem);
                vo.add(baseVo);
            }else{
                continue;
            }
        }
        return ResponseDto.success(vo);
    }
}
