package com.liquidnet.service.adam.service.impl;

import com.liquidnet.common.exception.LiquidnetServiceException;
import com.liquidnet.common.mq.constant.MQConst;
import com.liquidnet.commons.lang.util.*;
import com.liquidnet.service.adam.dto.AdamThirdPartParam;
import com.liquidnet.service.adam.dto.vo.*;
import com.liquidnet.service.adam.entity.AdamEnters;
import com.liquidnet.service.adam.entity.AdamRealName;
import com.liquidnet.service.adam.service.*;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.SqlMapping;
import com.mongodb.client.model.FindOneAndUpdateOptions;
import com.mongodb.client.model.ReturnDocument;
import com.mongodb.client.result.DeleteResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.LinkedMultiValueMap;

import java.time.LocalDateTime;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * 用户 服务实现类
 * </p>
 *
 * @author liquidnet
 * @since 2021-05-10
 */
@Slf4j
@Service
public class AdamUserServiceImpl implements IAdamUserService {
    @Autowired
    Environment env;
    @Autowired
    MongoTemplate mongoTemplate;
    @Autowired
    RabbitTemplate rabbitTemplate;
    @Autowired
    IAdamEntersService adamEntersService;
    @Autowired
    IAdamRealNameService adamRealNameService;
    @Autowired
    IAdamRdmService adamRdmService;



    private static final String PHP_API_REAL_NAME = "/IdCardCheckVerification";

    @Override
//    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public AdamUserInfoVo register(String mobile) {
        LocalDateTime now = LocalDateTime.now();

        AdamUserInfoVo userInfoVo = AdamUserInfoVo.getNew();
        userInfoVo.setUid(IDGenerator.nextSnowId() + "");
        userInfoVo.setMobile(mobile);
        userInfoVo.setIsComplete(0);
        userInfoVo.setState(1);
        userInfoVo.setQrCode("lN".concat(userInfoVo.getUid()).concat(RandomStringUtils.randomAlphanumeric(5).toUpperCase()));
        userInfoVo.setCreateAt(now);

        long s = System.currentTimeMillis();
        mongoTemplate.insert(userInfoVo, AdamUserInfoVo.class.getSimpleName());
        log.debug("#MDB耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        String msg = SqlMapping.get("adam_user.add", userInfoVo.getUid(), userInfoVo.getMobile(), userInfoVo.getState(), now);
        log.debug("#SQL.GET耗时:{}ms", System.currentTimeMillis() - s);
        rabbitTemplate.convertAndSend(MQConst.EX_LNS_SQL_UCENTER, MQConst.RK_SQL_UREGISTER,
                msg
        );
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        adamRdmService.setUidByMobile(mobile, userInfoVo.getUid());
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);
        return userInfoVo;
    }

    @Override
//    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public AdamUserInfoVo register(AdamThirdPartParam param) {
        LocalDateTime now = LocalDateTime.now();

        long s = System.currentTimeMillis();
        String uid = adamRdmService.getUidByMobile(param.getMobile());
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);
        LinkedList<String> toMqSqls = new LinkedList<>();
        LinkedList<Object[]> initUserObjs = new LinkedList<>(), initThirdPartObjs = new LinkedList<>();
        AdamUserInfoVo userInfoVo;
        if (StringUtils.isEmpty(uid)) {// 手机号未注册
            userInfoVo = AdamUserInfoVo.getNew();

            userInfoVo.setUid(IDGenerator.nextSnowId() + "");
            userInfoVo.setMobile(param.getMobile());
            userInfoVo.setIsComplete(0);
            userInfoVo.setState(1);
            userInfoVo.setQrCode("lN".concat(userInfoVo.getUid()).concat(RandomStringUtils.randomAlphanumeric(5).toUpperCase()));
            userInfoVo.setCreateAt(now);

            s = System.currentTimeMillis();
            mongoTemplate.insert(userInfoVo, AdamUserInfoVo.class.getSimpleName());
            log.debug("#MDB耗时:{}ms", System.currentTimeMillis() - s);

            userInfoVo.setMobile(SensitizeUtil.custom(param.getMobile(), 3, 4));

            toMqSqls.add(SqlMapping.get("adam_user.add"));
            initUserObjs.add(new Object[]{userInfoVo.getUid(), userInfoVo.getMobile(), userInfoVo.getState(), now});
        } else {
            s = System.currentTimeMillis();
            userInfoVo = adamRdmService.getUserInfoVoByUid(uid);
            log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);
        }

        AdamThirdPartInfoVo thirdPartInfoVo = AdamThirdPartInfoVo.getNew();
        BeanUtils.copyProperties(param, thirdPartInfoVo);
        thirdPartInfoVo.setCreatedAt(now);
        thirdPartInfoVo.setUid(userInfoVo.getUid());
        thirdPartInfoVo.setState(1);// 1-绑定

        s = System.currentTimeMillis();
        mongoTemplate.insert(thirdPartInfoVo, AdamThirdPartInfoVo.class.getSimpleName());
        log.debug("#MDB耗时:{}ms", System.currentTimeMillis() - s);

        toMqSqls.add(SqlMapping.get("adam_third_party.add"));
        initThirdPartObjs.add(new Object[]{thirdPartInfoVo.getUid(), thirdPartInfoVo.getOpenId(), thirdPartInfoVo.getAvatar(), thirdPartInfoVo.getNickname(), thirdPartInfoVo.getPlatform(), thirdPartInfoVo.getState(), thirdPartInfoVo.getCreatedAt()});

        s = System.currentTimeMillis();
        adamRdmService.setUidByPlatformOpenId(param.getPlatform(), param.getOpenId(), uid);
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        rabbitTemplate.convertAndSend(MQConst.EX_LNS_SQL_UCENTER, MQConst.RK_SQL_UREGISTER,
                SqlMapping.gets(toMqSqls, initUserObjs, initThirdPartObjs)
        );
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);
        return userInfoVo;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void bindTpa(String uid, AdamThirdPartParam param) {
        LocalDateTime now = LocalDateTime.now();

        AdamThirdPartInfoVo thirdPartInfoVo = AdamThirdPartInfoVo.getNew();
        BeanUtils.copyProperties(param, thirdPartInfoVo);
        thirdPartInfoVo.setCreatedAt(now);
        thirdPartInfoVo.setUid(uid);
        thirdPartInfoVo.setState(1);// 1-绑定

        long s = System.currentTimeMillis();
        mongoTemplate.insert(thirdPartInfoVo, AdamThirdPartInfoVo.class.getSimpleName());
        log.debug("#MDB耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        adamRdmService.setUidByPlatformOpenId(param.getPlatform(), param.getOpenId(), uid);
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        adamRdmService.delThirdPartVoListByUid(uid);
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        rabbitTemplate.convertAndSend(MQConst.EX_LNS_SQL_UCENTER, MQConst.RK_SQL_UCENTER,
                SqlMapping.get(
                        "adam_third_party.add",
                        thirdPartInfoVo.getUid(), thirdPartInfoVo.getOpenId(), thirdPartInfoVo.getAvatar(), thirdPartInfoVo.getNickname(), thirdPartInfoVo.getPlatform(), thirdPartInfoVo.getState(), thirdPartInfoVo.getCreatedAt()
                )
        );
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void bindTpaForce(String bindUid, String unBindUid, AdamThirdPartParam param) {
        this.unBindTpaProcess(unBindUid, param.getPlatform());

        LocalDateTime now = LocalDateTime.now();

        AdamThirdPartInfoVo thirdPartInfoVo = AdamThirdPartInfoVo.getNew();
        BeanUtils.copyProperties(param, thirdPartInfoVo);
        thirdPartInfoVo.setCreatedAt(now);
        thirdPartInfoVo.setUid(bindUid);
        thirdPartInfoVo.setState(1);// 1-绑定

        mongoTemplate.insert(thirdPartInfoVo, AdamThirdPartInfoVo.class.getSimpleName());

        adamRdmService.setUidByPlatformOpenId(param.getPlatform(), param.getOpenId(), bindUid);

        adamRdmService.delThirdPartVoListByUid(bindUid);

        rabbitTemplate.convertAndSend(MQConst.EX_LNS_SQL_UCENTER, MQConst.RK_SQL_UCENTER,
                SqlMapping.get(
                        "adam_third_party.add",
                        thirdPartInfoVo.getUid(), thirdPartInfoVo.getOpenId(), thirdPartInfoVo.getAvatar(), thirdPartInfoVo.getNickname(), thirdPartInfoVo.getPlatform(), thirdPartInfoVo.getState(), thirdPartInfoVo.getCreatedAt()
                )
        );
    }

    /**
     * 解绑第三方账号操作
     *
     * @param uid
     * @param platform
     */
    private void unBindTpaProcess(String uid, String platform) {
        LocalDateTime now = LocalDateTime.now();

//        AdamThirdPartInfoVo unBindThirdPartInfoVo = AdamThirdPartInfoVo.getNew();
//        unBindThirdPartInfoVo.setUpdatedAt(now);
//        unBindThirdPartInfoVo.setState(2);// 2-解绑

        long s = System.currentTimeMillis();
        Document doc = mongoTemplate.getCollection(AdamThirdPartInfoVo.class.getSimpleName()).findOneAndUpdate(
                Query.query(Criteria.where("uid").is(uid).and("platform").is(platform).and("state").is(1)).getQueryObject(),
                new Document("$set", new Document("state", 2).append("updatedAt", now)),
                new FindOneAndUpdateOptions().returnDocument(ReturnDocument.AFTER)
        );
        log.debug("#MDB耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        if (null != doc) {
            adamRdmService.delUidByPlatformOpenId(platform, doc.getString("openId"));
        }
        adamRdmService.delThirdPartVoListByUid(uid);
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);

        s = System.currentTimeMillis();
        rabbitTemplate.convertAndSend(MQConst.EX_LNS_SQL_UCENTER, MQConst.RK_SQL_UCENTER,
                SqlMapping.get("adam_third_party.unbind", now, uid, platform));
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);
    }

    @Override
//    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void unBindTpa(String uid, String platform) {
        this.unBindTpaProcess(uid, platform);
    }

    @Override
//    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void close(String uid) {
        LocalDateTime now = LocalDateTime.now();
        LinkedList<String> toMqSqls = new LinkedList<>();
        LinkedList<Object[]> objsUser = new LinkedList<>(),
                objsThirdPart = new LinkedList<>(),
                objsAddresses = new LinkedList<>();

        long s = System.currentTimeMillis();
        AdamUserInfoVo userInfoVo = adamRdmService.getUserInfoVoByUid(uid);

        adamRdmService.delUidByMobile(userInfoVo.getMobile());
        List<AdamThirdPartInfoVo> thirdPartInfoVos = adamRdmService.getThirdPartVoListByUid(uid);
        if (!CollectionUtils.isEmpty(thirdPartInfoVos)) {
            thirdPartInfoVos.forEach(r -> {
                adamRdmService.delUidByPlatformOpenId(r.getPlatform(), r.getOpenId());
            });
            adamRdmService.delThirdPartVoListByUid(uid);
        }
        adamRdmService.delUserInfoVoByUid(uid);
        adamRdmService.delUserMemberVoByUid(uid);
        adamRdmService.delAddressesVoByUid(uid);
        adamRdmService.delEntersVoByUid(uid);
        log.debug("##RDS耗时:{}ms", System.currentTimeMillis() - s);

        DeleteResult deleteResult = mongoTemplate.remove(Query.query(Criteria.where("uid").is(uid)), AdamUserInfoVo.class.getSimpleName());
        toMqSqls.add(SqlMapping.get("adam_user.close"));
        objsUser.add(new Object[]{now, now, uid});

        /* ---------------------- 三方账号信息 */
        if (!CollectionUtils.isEmpty(thirdPartInfoVos)) {
            s = System.currentTimeMillis();
            mongoTemplate.remove(Query.query(Criteria.where("uid").is(uid)), AdamThirdPartInfoVo.class.getSimpleName());
            log.debug("#MDB耗时:{}ms", System.currentTimeMillis() - s);
            toMqSqls.add(SqlMapping.get("adam_third_party.close"));
            objsThirdPart.add(new Object[]{now, uid});
        }

        /* ---------------------- 用户会员信息 */
        s = System.currentTimeMillis();
        deleteResult = mongoTemplate.remove(Query.query(Criteria.where("uid").is(uid)), AdamUserMemberVo.class.getSimpleName());
        log.debug("#MDB耗时:{}ms", System.currentTimeMillis() - s);
        if (deleteResult.getDeletedCount() > 0) {
            toMqSqls.add(SqlMapping.get("adam_user_member.close"));
            objsThirdPart.add(new Object[]{now, uid});
        }

        /* ---------------------- 收货地址信息 */
        s = System.currentTimeMillis();
        deleteResult = mongoTemplate.remove(Query.query(Criteria.where("uid").is(uid).and("state").is(1)), AdamAddressesVo.class.getSimpleName());
        log.debug("#MDB耗时:{}ms", System.currentTimeMillis() - s);
        if (deleteResult.getDeletedCount() > 0) {
            toMqSqls.add(SqlMapping.get("adam_addresses.close"));
            objsAddresses.add(new Object[]{now, now, uid});
        }

        /* ---------------------- 入场人信息 */
        s = System.currentTimeMillis();
        deleteResult = mongoTemplate.remove(Query.query(Criteria.where("uid").is(uid)).getQueryObject(), AdamEnters.class.getSimpleName());
        log.debug("#MDB耗时:{}ms", System.currentTimeMillis() - s);
        if (deleteResult.getDeletedCount() > 0) {
            toMqSqls.add(SqlMapping.get("adam_enters.close"));
            objsAddresses.add(new Object[]{now, now, uid});
        }

        s = System.currentTimeMillis();
        rabbitTemplate.convertAndSend(MQConst.EX_LNS_SQL_UCENTER, MQConst.RK_SQL_UCENTER,
                SqlMapping.gets(toMqSqls, objsUser));
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);
    }

    @Override
//    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public AdamRealInfoVo identity(String uid, String name, String idCard) {
        // TODO: 2021/5/18 调用第三方认证服务

        LinkedMultiValueMap<String, String> paramsMap = new LinkedMultiValueMap<>();
        paramsMap.add("realName", name);
        paramsMap.add("cardNo", idCard);

        String respStr = null;
        try {
            long s = System.currentTimeMillis();
            respStr = HttpUtil.postToPhpApi(env.getProperty("liquidnet.url-service.url") + PHP_API_REAL_NAME, paramsMap);
            log.debug("#PHP.API耗时:{}ms", System.currentTimeMillis() - s);
            log.debug("###PHP.API[{}].RESP:{}", PHP_API_REAL_NAME, respStr);
            Map respMap = JsonUtils.fromJson(respStr, Map.class);
            if (CollectionUtils.isEmpty(respMap) || !StringUtils.equalsIgnoreCase("OK", (String) respMap.get("message"))) {
                log.warn("PHP.API实名认证失败[uid:{},name:{},idCard:{},respStr:{}]", uid, name, idCard, respStr);
                ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("10102");
                throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
            }
        } catch (Exception e) {
            log.error("PHP.API实名认证异常[uid:{},name:{},idCard:{},respStr:{}]", uid, name, idCard, respStr, e);
            ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("10102");
            throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
        }

        AdamRealName realName = new AdamRealName();
        realName.setRealNameId(IDGenerator.nextSnowId() + "");
        realName.setUid(uid);
        realName.setType("1");
        realName.setName(name);
        realName.setIdCard(idCard);
        realName.setState(1);
        realName.setCreatedAt(LocalDateTime.now());
        adamRealNameService.add(realName);

        AdamRealInfoVo realInfoVo = AdamRealInfoVo.getNew();
        realInfoVo.setName(name);
        realInfoVo.setIdCard(idCard);
        realInfoVo.setState(1);

        long s = System.currentTimeMillis();
        adamRdmService.setRealInfoVoByUid(uid, realInfoVo);
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);

        return realInfoVo;
    }
}
