package com.liquidnet.support.zuul.filter;

import com.liquidnet.common.cache.redis.util.RedisUtil;
import com.liquidnet.commons.lang.core.JwtValidator;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.support.zuul.util.PathMatchUtil;
import com.netflix.zuul.ZuulFilter;
import com.netflix.zuul.context.RequestContext;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.DigestUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.nio.charset.StandardCharsets;
import java.util.List;

@Slf4j
@Component
public class GlobalAuthFilter extends ZuulFilter {
    @Value("#{'${global.exclude.url}'.split(', ')}")
    private List<String> excludeUrl;
    @Value("#{'${global.exclude.url-pattern}'.split(', ')}")
    private List<String> excludeUrlPattern;
    private static final String CONTENT_TYPE = "application/json;charset=utf-8";
    private static final String AUTHORIZATION = "authorization";
    private static final String TOKEN_ILLEGAL = "40001";
    private static final String TOKEN_KICK = "40002";
    private static final String TOKEN_EXPIRED = "40003";

    @Autowired
    JwtValidator jwtValidator;
    @Autowired
    RedisUtil redisUtil;

    @Override
    public String filterType() {
        return "pre";
    }

    @Override
    public int filterOrder() {
        return 2;
    }

    @Override
    public boolean shouldFilter() {
        RequestContext ctx = RequestContext.getCurrentContext();
        String requestURI = ctx.getRequest().getRequestURI();

        if (excludeUrl.contains(requestURI)) {
            return false;
        }
        for (String urlPattern : excludeUrlPattern) {
            return !PathMatchUtil.isPathMatch(urlPattern, requestURI);
        }
        return true;
    }

    @Override
    public Object run() {
        RequestContext requestContext = RequestContext.getCurrentContext();
        HttpServletRequest httpServletRequest = requestContext.getRequest();
        String authorization = httpServletRequest.getHeader(AUTHORIZATION);
        try {
            log.info("authorization:{}", authorization);

            if (StringUtils.isBlank(authorization)) {

                respHandler(requestContext, TOKEN_ILLEGAL);
            } else {
                String token = authorization.substring(7);

                String ssoKey = jwtValidator.getSsoRedisKey().concat(DigestUtils.md5DigestAsHex(token.getBytes(StandardCharsets.UTF_8)));

                if (redisUtil.hasKey(ssoKey)) {
                    if ((boolean) redisUtil.get(ssoKey)) {
                        // 解析没有异常则表示token验证通过，如有必要可根据自身需求增加验证逻辑
                        Claims claims = jwtValidator.parse(token);

                        requestContext.setSendZuulResponse(true);

//                        requestContext.addZuulRequestHeader("uid", claims.get("uid").toString());

                        requestContext.addZuulRequestHeader(CurrentUtil.uTag, JsonUtils.toJson(claims));

                        requestContext.addZuulRequestHeader(CurrentUtil.uToken, token);
                    } else {
                        respHandler(requestContext, TOKEN_EXPIRED);
                    }
                } else {
                    respHandler(requestContext, TOKEN_KICK);
                }
            }
        } catch (ExpiredJwtException expiredJwtEx) {
            log.error("TOKEN已过期:{}", authorization);

            respHandler(requestContext, TOKEN_EXPIRED);
        } catch (Exception ex) {
            log.error("TOKEN验证失败:{}", authorization);

            respHandler(requestContext, TOKEN_ILLEGAL);
        }
        return null;
    }

    private void respHandler(RequestContext requestContext, String code) {
        requestContext.setSendZuulResponse(false);
        ResponseDto<Object> responseDto = ResponseDto.failure(ErrorMapping.get(code));
        requestContext.setResponseBody(JsonUtils.toJson(responseDto));
        HttpServletResponse response = requestContext.getResponse();
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.setContentType(CONTENT_TYPE);
    }
}
