package com.liquidnet.service.adam.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.SensitizeUtil;
import com.liquidnet.service.adam.dto.AdamAddressesParam;
import com.liquidnet.service.adam.dto.vo.AdamAddressesVo;
import com.liquidnet.service.adam.service.AdamRdmService;
import com.liquidnet.service.adam.service.IAdamAddressesService;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.util.List;

/**
 * <p>
 * 收货地址 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2021-05-10
 */
@ApiSupport(order = 10050)
@Api(tags = "收货地址")
@Slf4j
@Validated
@RestController
@RequestMapping("addr")
public class AdamAddressesController {
    @Autowired
    AdamRdmService adamRdmService;
    @Autowired
    IAdamAddressesService adamAddressesService;

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "添加收货地址")
    @PostMapping("add")
    public ResponseDto<Object> add(@Valid @RequestBody AdamAddressesParam parameter) {
        return ResponseDto.success(adamAddressesService.add(parameter));
    }

    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "收货地址列表")
    @GetMapping("list")
    public ResponseDto<List<AdamAddressesVo>> list() {
        List<AdamAddressesVo> vos = adamRdmService.getAddressesVoByUid(CurrentUtil.getCurrentUid());
        if (!CollectionUtils.isEmpty(vos)) {
            for (AdamAddressesVo vo : vos) {
                vo.setPhone(SensitizeUtil.custom(vo.getPhone(), 3, 4));
            }
        }
        return ResponseDto.success(vos);
    }

    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "设置默认收货地址")
    @PostMapping("def/{addrId}")
    public ResponseDto<Object> def(@NotBlank(message = "收货地址ID不能为空") @PathVariable String addrId) {
        String currentUid = CurrentUtil.getCurrentUid();

        AdamAddressesVo vo = adamRdmService.getAddressesVoByUidAddressesId(currentUid, addrId);
        if (null == vo) return ResponseDto.failure(ErrorMapping.get("10106"));

        if (!vo.getIsDefault()) adamAddressesService.def(currentUid, addrId);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 4)
    @ApiOperation(value = "编辑收货地址")
    @PostMapping("edit")
    public ResponseDto<Object> edit(@RequestBody @Valid AdamAddressesParam parameter) {
        if (StringUtils.isBlank(parameter.getAddressesId())) {
            return ResponseDto.failure(ErrorMapping.get("10016"));
        }
        if (null == adamRdmService.getAddressesVoByUidAddressesId(CurrentUtil.getCurrentUid(), parameter.getAddressesId())) {
            return ResponseDto.failure(ErrorMapping.get("10106"));
        }

        adamAddressesService.edit(parameter);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 5)
    @ApiOperation(value = "删除收货地址")
    @PostMapping("del/{addrId}")
    public ResponseDto<Object> del(@NotBlank(message = "收货地址ID不能为空") @PathVariable String addrId) {
        String currentUid = CurrentUtil.getCurrentUid();

        AdamAddressesVo vo = adamRdmService.getAddressesVoByUidAddressesId(currentUid, addrId);
        if (null == vo) return ResponseDto.failure(ErrorMapping.get("10106"));

        adamAddressesService.remove(currentUid, addrId);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 6)
    @ApiOperation(value = "收货地址详情")
    @PostMapping("query/{id}")
    public ResponseDto<AdamAddressesVo> query(@NotBlank(message = "收货地址ID不能为空") @PathVariable(name = "id") String addrId) {
        AdamAddressesVo vo = adamRdmService.getAddressesVoByUidAddressesId(CurrentUtil.getCurrentUid(), addrId);
        if (null != vo) {
            vo.setPhone(SensitizeUtil.custom(vo.getPhone(), 3, 4));

            return ResponseDto.success(vo);
        }
        return ResponseDto.failure(ErrorMapping.get("10106"));
    }

    @ApiOperationSupport(order = 7)
    @ApiOperation(value = "获取默认收货地址")
    @PostMapping("query/def")
    public ResponseDto<AdamAddressesVo> queryDef() {
        AdamAddressesVo vo = adamAddressesService.queryDefault(null, CurrentUtil.getCurrentUid(), true);
        if (null != vo) {
            vo.setPhone(SensitizeUtil.custom(vo.getPhone(), 3, 4));
        }
        return ResponseDto.success(vo);
    }

    /* ------------------------------------------------------------｜@API:RSC */

    @ApiOperationSupport(order = 99)
    @ApiOperation(value = "@API:收货地址", notes = "查取用户指定收货地址详情")
    @GetMapping("query/depth")
    public ResponseDto<AdamAddressesVo> queryForDepth(@NotBlank(message = "收货地址ID不能为空") @RequestParam String addrId,
                                                      @NotBlank(message = "用户ID不能为空") @RequestParam String uid) {
        AdamAddressesVo vo = adamRdmService.getAddressesVoByUidAddressesId(uid, addrId);
        return null == vo ? ResponseDto.failure(ErrorMapping.get("10106")) : ResponseDto.success(vo);
    }
}
