package com.liquidnet.common.swagger.annotation;

import java.lang.annotation.*;

/**
 * 接口分级注解
 * 用于标识接口的重要性级别
 * 
 * @author system
 * @since 2024-10-29
 */
@Target({ElementType.METHOD, ElementType.TYPE})
@Retention(RetentionPolicy.RUNTIME)
@Documented
public @interface ApiLevel {
    
    /**
     * 接口级别 - 默认为L99（未配置），强制开发者明确指定级别
     */
    Level level() default Level.L99;
    
    /**
     * 描述信息
     */
    String description() default "";
    
    /**
     * 接口级别枚举 - 数字越小越重要
     */
    enum Level {
        L1(1, "核心", "核心业务功能，必须稳定，影响系统可用性"),
        L2(2, "重要", "重要业务功能，影响用户体验"),
        L3(3, "普通", "常规业务功能"),
        L4(4, "低级", "非核心功能，影响范围小"),
        L88(88, "无需分级", "该接口明确标记为无需分级或暂不分级"),
        L99(99, "未配置", "该接口尚未配置分级信息");
        
        private final int value;
        private final String desc;
        private final String detail;
        
        Level(int value, String desc, String detail) {
            this.value = value;
            this.desc = desc;
            this.detail = detail;
        }
        
        public int getValue() { return value; }
        public String getDesc() { return desc; }
        public String getDetail() { return detail; }
        
        /**
         * 判断是否为未配置级别
         */
        public boolean isUnconfigured() {
            return this == L99;
        }
        
        /**
         * 判断是否为无需分级
         */
        public boolean isNoGrading() {
            return this == L88;
        }
        
        /**
         * 根据数字值获取级别
         */
        public static Level fromValue(int value) {
            for (Level level : values()) {
                if (level.value == value) {
                    return level;
                }
            }
            return L99; // 默认返回未配置
        }
        
        /**
         * 获取所有业务级别（排除L88和L99）
         */
        public static Level[] getBusinessLevels() {
            return new Level[]{L1, L2, L3, L4};
        }
    }
}