package com.liquidnet.common.swagger.controller;

import com.liquidnet.common.swagger.annotation.ApiLevel;
import com.liquidnet.common.swagger.dto.ApiLevelInfoDto;
import com.liquidnet.common.swagger.dto.ApiLevelStatisticsDto;
import com.liquidnet.common.swagger.service.ApiLevelStatisticsService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 接口分级统计控制器
 * 
 * @author system
 * @since 2024-10-29
 */
@Api(tags = "系统管理-接口分级统计")
@RestController
@RequestMapping("/levelStatistics")
@Slf4j
@ApiLevel(level = ApiLevel.Level.L2, description = "接口分级统计管理")
public class ApiLevelStatisticsController {
    
    @Autowired
    private ApiLevelStatisticsService apiLevelStatisticsService;
    
    @GetMapping("/overview")
    @ApiOperation("获取接口分级统计概览(理想是获取所有服务的数据， 但是现在还做不到预留吧)")
    @ApiLevel(level = ApiLevel.Level.L3, description = "查看系统所有接口的分级统计信息")
    public ApiLevelStatisticsDto getApiLevelStatistics() {
        try {
            return apiLevelStatisticsService.getApiLevelStatistics();
        } catch (Exception e) {
            log.error("获取接口分级统计失败", e);
            throw new RuntimeException("获取接口分级统计失败: " + e.getMessage());
        }
    }
    
    @GetMapping("/levels")
    @ApiOperation("获取所有接口级别定义")
    @ApiLevel(level = ApiLevel.Level.L4, description = "查看接口级别定义说明")
    public List<ApiLevelInfoDto> getApiLevels() {
        List<ApiLevelInfoDto> levels = new ArrayList<>();
        
        for (ApiLevel.Level level : ApiLevel.Level.values()) {
            ApiLevelInfoDto levelInfo = new ApiLevelInfoDto();
            levelInfo.setCode(level.name());                    // L1, L2, L3, L4, L88, L99
            levelInfo.setValue(level.getValue());               // 1, 2, 3, 4, 88, 99
            levelInfo.setLabel(level.getDesc());                // 核心, 重要, 普通, 低级, 无需分级, 未配置
            levelInfo.setDescription(level.getDetail());        // 详细描述
            levelInfo.setIsUnconfigured(level.isUnconfigured()); // 是否为未配置
            levelInfo.setIsNoGrading(level.isNoGrading());       // 是否为无需分级
            
            // 判断是否为业务级别 (L1-L4)
            boolean isBusinessLevel = level.getValue() >= 1 && level.getValue() <= 4;
            levelInfo.setIsBusinessLevel(isBusinessLevel);
            
            levels.add(levelInfo);
        }
        
        return levels;
    }
    

    @GetMapping("/service")
    @ApiOperation("获取本服务接口分级统计信息")
    @ApiLevel(level = ApiLevel.Level.L3, description = "获取当前服务的接口分级统计，包含按级别分组的接口列表")
    public Map<String, Object> getServiceStatistics() {
        try {
            ApiLevelStatisticsDto fullStats = apiLevelStatisticsService.getApiLevelStatistics();
            Map<String, Object> serviceStats = new HashMap<>();
            
            // 基础统计信息
            serviceStats.put("totalCount", fullStats.getTotalCount());
            serviceStats.put("levelStatistics", fullStats.getLevelStatistics());
            
            // 按级别分组的接口列表 - 动态初始化所有级别
            Map<String, List<Map<String, Object>>> apisByLevel = new HashMap<>();
            for (ApiLevel.Level level : ApiLevel.Level.values()) {
                apisByLevel.put(level.name(), new ArrayList<>());
            }
            
            // 将接口按级别分组
            for (ApiLevelStatisticsDto.ApiInfo api : fullStats.getApiList()) {
                Map<String, Object> apiInfo = new HashMap<>();
                apiInfo.put("url", api.getUrl());
                apiInfo.put("method", api.getHttpMethod());
                apiInfo.put("controller", api.getControllerName());
                apiInfo.put("methodName", api.getMethodName());
                apiInfo.put("description", api.getApiDescription());
                
                // 现在所有接口都有级别（包括L99未配置）
                String levelKey = api.getLevel().name();
                
                apisByLevel.get(levelKey).add(apiInfo);
            }
            
            serviceStats.put("apisByLevel", apisByLevel);
            
            return serviceStats;
        } catch (Exception e) {
            log.error("获取服务统计失败", e);
            throw new RuntimeException("获取服务统计失败: " + e.getMessage());
        }
    }
    

}