package com.liquidnet.service.slime.service.impl;

import com.liquidnet.common.exception.LiquidnetServiceException;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.PagedResult;
import com.liquidnet.service.slime.dto.vo.SlimeSponsorsVo;
import com.liquidnet.service.slime.service.ISlimeSponsorsService;
import com.liquidnet.service.slime.service.SlimeMongoService;
import com.liquidnet.service.slime.service.SlimeRdmService;
import com.liquidnet.service.slime.util.ObjectUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.regex.Pattern;

@Slf4j
@Service
public class SlimeSponsorsServiceImpl implements ISlimeSponsorsService {

    @Autowired
    SlimeRdmService slimeRdmService;

    @Autowired
    SlimeMongoService slimeMongoService;

    @Autowired
    MongoTemplate mongoTemplate;

    @Override
    public PagedResult<SlimeSponsorsVo> search(String name, int page, int size) {
        // 查询条件
        Query query = new Query();
        String regex = String.format("%s%s%s", "^.*", name, ".*$");
        Pattern pattern = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
        query.addCriteria(Criteria.where("name").regex(pattern));
        query.addCriteria(Criteria.where("isOnline").is(1));

        // 总数
        long count = mongoTemplate.count(query, SlimeSponsorsVo.class, SlimeSponsorsVo.class.getSimpleName());

        // 分页
        PagedResult<SlimeSponsorsVo> pagedResult = ObjectUtil.getSlimeSponsorsVoPagedResult();
        if (count > 0) {
            query.fields()
                    .include("sponsorId")
                    .include("isOnline")
                    .include("uid")
                    .include("name")
                    .include("logo")
                    .include("background");

            // 查询分页
            Pageable pageable = PageRequest.of(page - 1, size, Sort.by(Sort.Direction.DESC, "createdAt"));
            query.with(pageable);
            List<SlimeSponsorsVo> sponsorsVoList = mongoTemplate.find(query, SlimeSponsorsVo.class, SlimeSponsorsVo.class.getSimpleName());

            pagedResult.setList(sponsorsVoList).setTotal(count, size);
        }

        return pagedResult;
    }

    @Override
    public SlimeSponsorsVo sponsorInfo(String uid, String sponsorId) {
        // 当前用户是否管理该主办
        SlimeSponsorsVo sponsorsVo = this.checkSponsorAccount(uid, sponsorId);

        return sponsorsVo;
    }

    private SlimeSponsorsVo checkSponsorAccount(String uid, String sponsorId) {
        // 当前用户是否管理该主办
        SlimeSponsorsVo sponsorsVo = slimeRdmService.getSponsorsVoBySponsorId(sponsorId);
        if (null == sponsorsVo || !uid.equals(sponsorsVo.getUid())) {
            ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("13201");
            throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
        }
        return sponsorsVo;
    }
}
