package com.liquidnet.service.merchant.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.service.base.PagedResult;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.merchant.dto.param.MerchantFieldApplyParam;
import com.liquidnet.service.merchant.dto.vo.MerchantFieldAppliesVo;
import com.liquidnet.service.merchant.dto.vo.MerchantFieldCheckersVo;
import com.liquidnet.service.merchant.dto.vo.MerchantFieldsDetailsVo;
import com.liquidnet.service.merchant.dto.vo.MerchantFieldsVo;
import com.liquidnet.service.merchant.service.IMerchantFieldAppliesService;
import com.liquidnet.service.merchant.service.IMerchantFieldsService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.util.List;

/**
 * <p>
 * 场地 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2021-11-01
 */
@ApiSupport(order = 10001)
@Api(tags = "场地")
@Slf4j
@Validated
@RestController
@RequestMapping("fields")
public class MerchantFieldsController {

    @Autowired
    IMerchantFieldsService merchantFieldsService;

    @Autowired
    IMerchantFieldAppliesService merchantFieldAppliesService;

    @ApiOperationSupport(order = 10)
    @ApiOperation(value = "场地搜索")
    @GetMapping("search")
    public ResponseDto<PagedResult<MerchantFieldsVo>> search(@RequestParam(required = true) @NotBlank(message = "搜索内容不能为空") String name,
                                                             @RequestParam(defaultValue = "1", required = false) int page,
                                                             @RequestParam(defaultValue = "10", required = false) int size) {
        return ResponseDto.success(merchantFieldsService.search(name, page, size));
    }

    @ApiOperationSupport(order = 11)
    @ApiOperation(value = "申请场地")
    @PostMapping("apply")
    public ResponseDto<Object> apply(@Valid @RequestBody MerchantFieldApplyParam parameter) {
        String currentUid = CurrentUtil.getCurrentUid();

        return ResponseDto.success(merchantFieldAppliesService.apply(currentUid, parameter));
    }

    @ApiOperationSupport(order = 12)
    @ApiOperation(value = "申请场地驳回后删除")
    @PostMapping("apply/del")
    public ResponseDto<Object> applyDel(@RequestParam(required = true) @NotBlank(message = "场地申请ID不能为空") String fieldApplyId) {
        String currentUid = CurrentUtil.getCurrentUid();

        merchantFieldAppliesService.applyDel(currentUid, fieldApplyId);

        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 13)
    @ApiOperation(value = "申请场地详情")
    @PostMapping("apply/info")
    public ResponseDto<MerchantFieldAppliesVo> applyInfo(@RequestParam(required = true) @NotBlank(message = "场地申请ID不能为空") String fieldApplyId) {
        String currentUid = CurrentUtil.getCurrentUid();

        return ResponseDto.success(merchantFieldAppliesService.applyInfo(currentUid, fieldApplyId));
    }


    @ApiOperationSupport(order = 20)
    @ApiOperation(value = "场地详情")
    @GetMapping("info")
    public ResponseDto<MerchantFieldsVo> info(@RequestParam(required = true) @NotBlank(message = "场地ID不能为空") String fieldId) {
        String currentUid = CurrentUtil.getCurrentUid();

        return ResponseDto.success(merchantFieldsService.fieldInfo(currentUid, fieldId));
    }

    @ApiOperationSupport(order = 21)
    @ApiOperation(value = "场地更改是否审核演出")
    @PostMapping("editIsCheck")
    public ResponseDto<Object> editIsCheck(@RequestParam(required = true) @NotBlank(message = "场地ID不能为空") String fieldId,
                                           @RequestParam(required = true, defaultValue = "0") int isCheck) {
        String currentUid = CurrentUtil.getCurrentUid();

        merchantFieldsService.editIsCheck(currentUid, fieldId, isCheck);

        return ResponseDto.success();
    }


    @ApiOperationSupport(order = 30)
    @ApiOperation(value = "场地默认验票员列表")
    @GetMapping("checkers")
    public ResponseDto<List<MerchantFieldCheckersVo>> checkers(@RequestParam(required = true) @NotBlank(message = "场地ID不能为空") String fieldId) {
        String currentUid = CurrentUtil.getCurrentUid();

        return ResponseDto.success(merchantFieldsService.checkers(currentUid, fieldId));
    }

    @ApiOperationSupport(order = 31)
    @ApiOperation(value = "场地添加默认验票员")
    @PostMapping("checker/add")
    public ResponseDto<Object> checkerAdd(@RequestParam(required = true) @NotBlank(message = "场地ID不能为空") String fieldId,
                                          @RequestParam(required = true) @NotBlank(message = "uid不能为空") String uid,
                                          @RequestParam(required = true) @NotBlank(message = "手机号不能为空") String mobile,
                                          @RequestParam(required = true) @NotBlank(message = "姓名不能为空") String name) {
        String currentUid = CurrentUtil.getCurrentUid();

        return ResponseDto.success(merchantFieldsService.checkerAdd(currentUid, fieldId, uid, mobile, name));
    }

    @ApiOperationSupport(order = 32)
    @ApiOperation(value = "场地移除默认验票员")
    @PostMapping("checker/del")
    public ResponseDto<Object> checkerDel(@RequestParam(required = true) @NotBlank(message = "场地ID不能为空") String fieldId,
                                          @RequestParam(required = true) @NotBlank(message = "uid不能为空") String uid) {
        String currentUid = CurrentUtil.getCurrentUid();

        merchantFieldsService.checkerDel(currentUid, fieldId, uid);

        return ResponseDto.success();
    }


    @ApiOperationSupport(order = 34)
    @ApiOperation(value = "场地详情")
    @PostMapping("app/details")
    public ResponseDto<MerchantFieldsDetailsVo> fieldDetails(@RequestParam(required = true) @NotBlank(message = "场地ID不能为空") String fieldId) {
        MerchantFieldsDetailsVo vo= merchantFieldsService.fieldDetails(fieldId);
        return ResponseDto.success(vo);
    }

}
