package com.liquidnet.service.kylin.controller;


import com.github.pagehelper.PageInfo;
import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.kylin.dto.param.PayAgainParam;
import com.liquidnet.service.kylin.dto.param.PayOrderParam;
import com.liquidnet.service.kylin.dto.param.SyncOrderParam;
import com.liquidnet.service.kylin.dto.vo.returns.*;
import com.liquidnet.service.kylin.service.IKylinOrderTicketsService;
import com.liquidnet.service.kylin.service.impl.KylinOrderTicketsServiceImpl;
import io.swagger.annotations.*;
import org.apache.ibatis.annotations.Param;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.HashMap;
import java.util.List;

/**
 * <p>
 * 订单 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2021-05-20
 */
@Api(tags = "前端-订单相关")
@RestController
@RequestMapping("order")
@Validated
public class KylinOrderTicketsController {

    @Autowired
    IKylinOrderTicketsService orderTicketsService;

//    @PostMapping("pre")
//    @ApiOperation("下单")
//    @ApiResponse(code = 200, message = "接口返回对象参数")
//    public ResponseDto<PayResultVo> checkCanOrder(@RequestBody @Valid PayOrderParam payOrderParam) {
//        return orderTicketsService.checkCanOrder(payOrderParam);
//    }
//
//    @PostMapping("payAgain")
//    @ApiOperation("再次支付")
//    @ApiResponse(code = 200, message = "接口返回对象参数")
//    public ResponseDto<PayResultVo> payAgain(@RequestBody @Valid PayAgainParam payAgainParam) {
//        PayResultVo vo = orderTicketsService.payAgain(payAgainParam).getData();
//        if (null == vo) {
//            return ResponseDto.failure(ErrorMapping.get("20024"));
//        } else {
//            return ResponseDto.success(vo);
//        }
//    }
//
//    @PostMapping("syncOrder")
//    @ApiOperation("支付回调")
//    @ApiResponse(code = 200, message = "接口返回对象参数")
//    public String syncOrder(@ModelAttribute @Valid SyncOrderParam syncOrderParam) {
//        return orderTicketsService.syncOrder(syncOrderParam);
//    }

    @GetMapping("list")
    @ApiOperation("订单列表")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<PageInfo<List<KylinOrderListVo>>> orderList() {
        return ResponseDto.success(orderTicketsService.orderList());
    }

    @GetMapping("details")
    @ApiOperation("订单详情")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<OrderDetailsVo> orderDetails(@RequestParam(value = "orderId", required = false) @NotNull(message = "订单ID不能为空") String orderId) {
        OrderDetailsVo vo = orderTicketsService.orderDetails(orderId);
        if (null == vo) {
            return ResponseDto.failure(ErrorMapping.get("20003"));
        } else {
            return ResponseDto.success(vo);
        }
    }

    @GetMapping("toOrderRefundDetails")
    @ApiOperation("想要退款订单详情")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<KylinOrderTicketPreVo> toOrderRefundDetails(
            @RequestParam(value = "orderId", required = false) @NotNull(message = "订单ID不能为空") String orderId
    ) {
        return orderTicketsService.toOrderRefundDetails(orderId);
    }

    @PostMapping("sendOrderRefunds")
    @ApiOperation("发起退款申请")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "orderId", value = "订单ID", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "orderTicketEntitiesId", value = "订单入场人ID", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "reason", value = "退款申请备注", required = true),
            @ApiImplicitParam(type = "form", dataType = "String", name = "picList", value = "图片"),
            @ApiImplicitParam(type = "form", dataType = "Integer", name = "sendExpressType", value = "快递方式 1上门取件 2自主发货"),
            @ApiImplicitParam(type = "form", dataType = "String", name = "expressNumber", value = "自主发货的顺丰订单号"),
            @ApiImplicitParam(type = "form", dataType = "String", name = "expressContacts", value = "发货人姓名"),
            @ApiImplicitParam(type = "form", dataType = "String", name = "expressPhone", value = "发货人联系方式"),
            @ApiImplicitParam(type = "form", dataType = "String", name = "sendExpressAddress", value = "取件地址"),
            @ApiImplicitParam(type = "form", dataType = "String", name = "appointmentTime", value = "预约取件时间")
    })
    public ResponseDto sendOrderRefund(
            @RequestParam(value = "orderId", required = true) @NotNull(message = "订单ID不能为空") String orderId,
            @RequestParam(value = "orderTicketEntitiesId", required = true) @NotNull(message = "票单ID不能为空") String orderTicketEntitiesId,
            @RequestParam(value = "reason", required = true) @NotNull(message = "退款原因不能为空") String reason,
            @RequestParam(value = "picList", required = false) String picList,

            @RequestParam(value = "sendExpressType", required = false) Integer sendExpressType,
            @RequestParam(value = "expressNumber", required = false) String expressNumber,

            @RequestParam(value = "expressContacts", required = false) String expressContacts,
            @RequestParam(value = "expressPhone", required = false) String expressPhone,
            @RequestParam(value = "sendExpressAddress", required = false) String sendExpressAddress,
            @RequestParam(value = "appointmentTime", required = false) String appointmentTime
    ) {
        // 退款
        return orderTicketsService.sendOrderRefund(
                orderId, orderTicketEntitiesId, reason, picList,
                sendExpressType, expressNumber, expressContacts, expressPhone, sendExpressAddress, appointmentTime
        );
    }

    @GetMapping("orderRefundDetails")
    @ApiOperation("退款申请详情")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<OrderRefundVo> orderRefundDetails(@RequestParam(value = "orderId", required = true) @NotNull(message = "订单ID不能为空") String orderId,
                                                         @RequestParam(value = "orderRefundId", required = true) @NotNull(message = "退款订单ID不能为空") String orderRefundId) {
        return orderTicketsService.orderRefundDetails(orderId, orderRefundId);
    }

    @PostMapping("orderRefundWithdraw")
    @ApiOperation("发起退款撤回")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<Boolean> orderRefundWithdraw(@RequestParam(value = "orderRefundId", required = true) @NotNull(message = "退款订单ID不能为空") String orderRefundId) {
        return orderTicketsService.orderRefundWithdraw(orderRefundId);
    }


    @GetMapping("orderUnPayCount")
    @ApiOperation("待支付演出订单数量")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<HashMap> orderUnPayCount() {
        Integer unPayCount = orderTicketsService.orderUnPayCount().getData();
        HashMap<String, Integer> map = CollectionUtil.mapStringInteger();
        map.put("unPayCount", unPayCount);
        return ResponseDto.success(map);
    }

    @PostMapping("transfer")
    @ApiOperation("转赠订单")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "orderId", value = "订单id"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "transferUid", value = "转赠目标uid"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "transferMobile", value = "转赠目标手机号"),
    })
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<String> orderTransfer(@RequestParam(value = "orderId") @NotNull @NotBlank String orderId,
                                             @RequestParam(value = "transferUid") @NotNull @NotBlank String transferUid,
                                             @RequestParam(value = "transferMobile") @NotNull @NotBlank String transferMobile
    ) {
        return orderTicketsService.orderTransfer(orderId, transferUid, transferMobile);
    }

    @PostMapping("transferWithDraw")
    @ApiOperation("转赠订单撤回")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "orderId", value = "订单id"),
    })
    public ResponseDto<String> orderTransferWithdraw(@RequestParam(value = "orderId") @NotNull @NotBlank String orderId) {
        return orderTicketsService.orderTransferWithdraw(orderId);
    }

    @PostMapping("transferAccept")
    @ApiOperation("转赠订单接收")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = false, dataType = "String", name = "enterIdList", value = "逗号隔开的入场人id"),
    })
    public ResponseDto<String> orderTransferAccept(@RequestParam(required = false) String enterIdList) {
        return orderTicketsService.orderTransferAccept(enterIdList);
    }

    @PostMapping("transferReject")
    @ApiOperation("转赠订单拒收")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<String> orderTransferReject() {
        return orderTicketsService.orderTransferReject();
    }

    @PostMapping("transferBack")
    @ApiOperation("转赠订单退回")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<String> orderTransferBack(@RequestParam(value = "orderId") String orderId) {
        return orderTicketsService.orderTransferBack(orderId);
    }
}
