package com.liquidnet.support.zuul.filter;

import com.liquidnet.common.cache.redis.util.RedisUtil;
import com.liquidnet.commons.lang.core.JwtValidator;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.support.zuul.util.PathMatchUtil;
import com.netflix.zuul.ZuulFilter;
import com.netflix.zuul.context.RequestContext;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.stereotype.Component;
import org.springframework.util.DigestUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;

@Slf4j
//@Data
//@Component
//@ConfigurationProperties(prefix = "global-auth")
public class GlobalAuthFilter extends ZuulFilter {
    private List<String> includeUrl;
    private List<String> includeUrlPattern;
    private List<String> excludeUrl;
    private List<String> excludeUrlPattern;

    private static final String KYLIN_STATION_JWT_VALID = "/kylin/station/**";

    private static final String CONTENT_TYPE = "application/json;charset=utf-8";
    private static final String AUTHORIZATION = "authorization";
    private static final String TOKEN_STATUS = "token_status";

    private static final String TOKEN_ILLEGAL = "40001";
    private static final String TOKEN_KICK = "40002";
    private static final String TOKEN_INVALID = "40003";

    @Autowired
    JwtValidator jwtValidator;
    @Autowired
    RedisUtil redisUtil;

    @Override
    public String filterType() {
        return "pre";
    }

    @Override
    public int filterOrder() {
        return 2;
    }

    @Override
    public boolean shouldFilter() {
        RequestContext ctx = RequestContext.getCurrentContext();

        HttpServletRequest ctxRequest = ctx.getRequest();

        String authorization = ctxRequest.getHeader(AUTHORIZATION);

        if (StringUtils.isNotBlank(authorization) && StringUtils.length(authorization) > 7) {
            String token = authorization.substring(7);
            try {
                // 解析没有异常则表示token验证通过，如有必要可根据自身需求增加验证逻辑
                Claims claims = jwtValidator.parse(token);

                ctx.addZuulRequestHeader(CurrentUtil.uID, (String) claims.get(CurrentUtil.uID));
                ctx.addZuulRequestHeader(CurrentUtil.uTag, JsonUtils.toJson(claims));
            } catch (ExpiredJwtException expiredJwtEx) {
                ctx.addZuulRequestHeader(TOKEN_STATUS, TOKEN_INVALID);
            } catch (Exception ex) {
                ctx.addZuulRequestHeader(TOKEN_STATUS, TOKEN_ILLEGAL);
            }
            ctx.addZuulRequestHeader(CurrentUtil.uToken, token);
        } else {
            ctx.addZuulRequestHeader(TOKEN_STATUS, TOKEN_ILLEGAL);
        }

        String requestURI = ctxRequest.getRequestURI();

        if (excludeUrl.contains(requestURI)) {
            return false;
        }
        if (includeUrl.contains(requestURI)) {
            return true;
        }
        for (String urlPattern : includeUrlPattern) {
            if (PathMatchUtil.isPathMatch(urlPattern, requestURI)) {
                return true;
            }
        }
        for (String urlPattern : excludeUrlPattern) {
            if (PathMatchUtil.isPathMatch(urlPattern, requestURI)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public Object run() {
        RequestContext ctx = RequestContext.getCurrentContext();
        Map<String, String> zuulRequestHeaders = ctx.getZuulRequestHeaders();

        log.debug("lns.headers:{}", zuulRequestHeaders);

        String uToken = zuulRequestHeaders.get(CurrentUtil.uToken), uid;
        if (StringUtils.isEmpty(uToken) || StringUtils.isEmpty(uid = zuulRequestHeaders.get(CurrentUtil.uID))) {
            this.respHandler(ctx, zuulRequestHeaders.get(TOKEN_STATUS));

            return null;
        }
        if (PathMatchUtil.isPathMatch(KYLIN_STATION_JWT_VALID, ctx.getRequest().getRequestURI())) {// 专业版APP
            // adam:identity:sso:${uid}:MD5(${token})=${1-在线|0-离线}
//            String ssoUidM5TokenKey = jwtValidator.getSsoRedisKey()
//                    .concat(uid).concat(":").concat(DigestUtils.md5DigestAsHex(uToken.getBytes(StandardCharsets.UTF_8)));
//            Integer online = (Integer) redisUtil.get(ssoUidM5TokenKey);
//            if (null == online || online != 1) {
//                this.respHandler(ctx, TOKEN_INVALID);
//            } else {
                ctx.setSendZuulResponse(true);
//            }
        } else {
            // adam:identity:sso:${uid}=MD5(${token})
            String ssoKey = jwtValidator.getSsoRedisKey().concat(uid);

            String md5Token = (String) redisUtil.get(ssoKey);

            if (StringUtils.isEmpty(md5Token)) {
                // 已离线
                this.respHandler(ctx, TOKEN_INVALID);
            } else {
                // 与在线TOKEN比对
                if (md5Token.equals(DigestUtils.md5DigestAsHex(uToken.getBytes(StandardCharsets.UTF_8)))) {
                    // 一致则放行
                    ctx.setSendZuulResponse(true);
                } else {
                    // 不一致则被踢下线
                    this.respHandler(ctx, TOKEN_KICK);
                }
            }
        }
        return null;
    }

    private void respHandler(RequestContext requestContext, String code) {
        requestContext.setSendZuulResponse(false);
        ResponseDto<Object> responseDto = ResponseDto.failure(ErrorMapping.get(code));
        requestContext.setResponseBody(JsonUtils.toJson(responseDto));
        HttpServletResponse response = requestContext.getResponse();
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.setContentType(CONTENT_TYPE);
    }
}
