package com.liquidnet.service.consumer.kylin.receiver;

import com.fasterxml.jackson.databind.JsonNode;
import com.liquidnet.common.cache.redis.util.RedisUtil;
import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.HttpUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.base.SqlMapping;
import com.liquidnet.service.base.constant.MQConst;
import com.liquidnet.service.galaxy.dto.param.GalaxyArtSeriesClaimReqDto;
import com.liquidnet.service.galaxy.dto.param.GalaxyArtSeriesClaimRespDto;
import com.liquidnet.service.galaxy.dto.param.GalaxyNftUploadReqDto;
import com.liquidnet.service.galaxy.dto.param.GalaxyNftUploadRespDto;
import com.liquidnet.service.goblin.constant.GoblinRedisConst;
import com.liquidnet.service.goblin.dto.vo.GoblinGoodsInfoVo;
import com.liquidnet.service.goblin.dto.vo.GoblinGoodsSkuInfoVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.data.redis.connection.stream.StreamRecords;
import org.springframework.data.redis.core.StreamOperations;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.LinkedList;

@Slf4j
@Component
public class ConsumerGoblinBizArtworkUplRdsReceiver extends AbstractBizRedisReceiver {
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private MongoTemplate mongoTemplate;


    @Value("${liquidnet.service.galaxy.url}")
    private String serviceGalaxyUrl;

    private static final String SQL_UPDATE_GOODS_SKU_NFT = "UPDATE goblin_goods_sku_nft SET upchain=?,display_url=?,nft_url=? WHERE sku_id=? AND upchain=0 ";

    @Override
    protected String getRedisStreamKey() {
        return MQConst.GoblinQueue.BIZ_ARTWORK_UPL.getKey();
    }

    @Override
    protected String getRedisStreamGroup() {
        return MQConst.GoblinQueue.BIZ_ARTWORK_UPL.getGroup();
    }

    @Override
    protected boolean consumerMessageHandler(String msg) {
        boolean aBoolean = false;
        try {
            if (StringUtils.isEmpty(msg)) {
                log.warn("CONSUMER MSG NULL_DTO ==> [{}]:{}", this.getRedisStreamKey(), msg);
                aBoolean = true;
            } else {
                aBoolean = this.bizArtworkUplProcessing(msg);
            }
        } catch (Exception e) {
            log.error("CONSUMER MSG EX_HANDLE ==> [{}]:{}", this.getRedisStreamKey(), msg, e);
        } finally {
            if (!aBoolean) {
                HashMap<String, String> map = CollectionUtil.mapStringString();
                map.put(MQConst.QUEUE_MESSAGE_KEY, msg);
                stringRedisTemplate.opsForStream().add(StreamRecords.mapBacked(map).withStreamKey(this.getRedisStreamKey()));
            }
        }
        return aBoolean;
    }

    private boolean bizArtworkUplProcessing(String skuId) {
        GoblinGoodsSkuInfoVo mgtGoodsSkuInfoVo = this.getGoodsSkuInfoVoFromMdb(skuId);
        if (null == mgtGoodsSkuInfoVo) {
            log.warn("#CONSUMER MSG NULL_SKU[{}]:[skuId={}]", this.getRedisStreamKey(), skuId);
            return true;
        }
        int skuType = mgtGoodsSkuInfoVo.getSkuType(), upchain = mgtGoodsSkuInfoVo.getUpchain();
        String unbox = mgtGoodsSkuInfoVo.getUnbox();
        // 非数字藏品 || 盲盒 || 非声明中 || 已有声明系列ID
        if (1 != skuType || !"0".equals(unbox) || 0 != upchain || StringUtils.isNotEmpty(mgtGoodsSkuInfoVo.getSeriesId())) {
            log.warn("#CONSUMER MSG VOID_SKU[{}]:[skuId={},skuType={},unbox={},upchain={},seriesId={}]",
                    this.getRedisStreamKey(), skuId, skuType, unbox, upchain, mgtGoodsSkuInfoVo.getSeriesId());
            return true;
        }

        String displayUrl = mgtGoodsSkuInfoVo.getDisplayUrl(), nftUrl = mgtGoodsSkuInfoVo.getNftUrl();
        if (StringUtils.isBlank(displayUrl)) {// 未上传过的直接上传处理，已上传过的跳过上传直接声明
            GalaxyNftUploadReqDto galaxyNftUploadReqDto = GalaxyNftUploadReqDto.getNew();
            galaxyNftUploadReqDto.setSkuId(skuId);
            galaxyNftUploadReqDto.setOriginalDisplayUrl(mgtGoodsSkuInfoVo.getSkuPic());
            galaxyNftUploadReqDto.setOriginalNftUrl(mgtGoodsSkuInfoVo.getMaterialUrl());
            galaxyNftUploadReqDto.setRouterType(mgtGoodsSkuInfoVo.getRouteType());
            GalaxyNftUploadRespDto galaxyNftUploadRespDto = this.uploadNftMaterial(galaxyNftUploadReqDto);
            if (null == galaxyNftUploadRespDto) return false;// 上传失败，重新入队处理

            displayUrl = galaxyNftUploadRespDto.getDisplayUrl();
            nftUrl = galaxyNftUploadRespDto.getNftUrl();
        }

        GoblinGoodsInfoVo mgtGoodsInfoVo = this.getGoodsInfoVoFromMdb(mgtGoodsSkuInfoVo.getSpuId());
        String skuTitle = mgtGoodsSkuInfoVo.getName() + mgtGoodsSkuInfoVo.getSubtitle();
        GalaxyArtSeriesClaimReqDto galaxyArtSeriesClaimReqDto = GalaxyArtSeriesClaimReqDto.getNew();
        galaxyArtSeriesClaimReqDto.setAuthor(mgtGoodsInfoVo.getAuthor());
        galaxyArtSeriesClaimReqDto.setCoverUrl(displayUrl);
        galaxyArtSeriesClaimReqDto.setDisplayUrl(displayUrl);
        galaxyArtSeriesClaimReqDto.setNftDesc(skuTitle);
        galaxyArtSeriesClaimReqDto.setNftName(skuTitle);
        galaxyArtSeriesClaimReqDto.setNftUrl(nftUrl);
        galaxyArtSeriesClaimReqDto.setRouterType(mgtGoodsSkuInfoVo.getRouteType());
        galaxyArtSeriesClaimReqDto.setSellCount(String.valueOf(mgtGoodsSkuInfoVo.getPrice()));
        galaxyArtSeriesClaimReqDto.setSeriesDesc(skuTitle);
        galaxyArtSeriesClaimReqDto.setSkuId(skuId);
        galaxyArtSeriesClaimReqDto.setTotalCount(Long.valueOf(mgtGoodsSkuInfoVo.getSkuStock()));

        // 声明失败，标记`声明失败`
        upchain = null == this.claimNftSeries(galaxyArtSeriesClaimReqDto) ? 2 : 9;

        mongoTemplate.getCollection(GoblinGoodsSkuInfoVo.class.getSimpleName()).updateOne(
                Query.query(Criteria.where("skuId").is(skuId).and("delFlg").is("0")).getQueryObject(),
                Update.update("upchain", upchain).set("displayUrl", displayUrl).set("nftUrl", nftUrl).getUpdateObject()
        );
        redisUtil.del(GoblinRedisConst.BASIC_GOODS_SKU.concat(skuId));

        // Mysql持久化
        HashMap<String, String> sqlUpdateMap = CollectionUtil.mapStringString();
        LinkedList<String> toMqSqls = CollectionUtil.linkedListString();
        toMqSqls.add(SQL_UPDATE_GOODS_SKU_NFT);
        LinkedList<Object[]> updateGoodsSkuNftObjs = CollectionUtil.linkedListObjectArr();
        updateGoodsSkuNftObjs.add(new Object[]{upchain, displayUrl, nftUrl, skuId});

        sqlUpdateMap.put(MQConst.QUEUE_MESSAGE_KEY, SqlMapping.gets(toMqSqls, updateGoodsSkuNftObjs));

        StreamOperations<String, Object, Object> streamOperations = stringRedisTemplate.opsForStream();
        streamOperations.add(StreamRecords.mapBacked(sqlUpdateMap).withStreamKey(MQConst.GoblinQueue.SQL_GOODS.getKey()));

        HashMap<String, String> toQueueBeClaimQueryMsg = CollectionUtil.mapStringString();
        toQueueBeClaimQueryMsg.put(MQConst.QUEUE_MESSAGE_KEY, skuId.concat(",").concat(String.valueOf(LocalDateTime.now())));
        streamOperations.add(StreamRecords.mapBacked(toQueueBeClaimQueryMsg).withStreamKey(MQConst.GoblinQueue.BIZ_ARTWORK_CLQ.getKey()));
        return true;
    }

    /* ------------------------------------------------------------------------------------ */
    /* ------------------------------------------------------------------------------------ */
    /* ------------------------------------------------------------------------------------ */
    /* ------------------------------------------------------------------------------------ */
    /* ------------------------------------------------------------------------------------ */

    /**
     * NFT素材上传
     *
     * @param nftUploadReqDto GalaxyNftUploadReqDto
     * @return GalaxyNftUploadRespDto
     */
    public GalaxyNftUploadRespDto uploadNftMaterial(GalaxyNftUploadReqDto nftUploadReqDto) {
        String postUrl = serviceGalaxyUrl + "/galaxy/artwork/nftUpload";
        String postBody = JsonUtils.toJson(nftUploadReqDto);
        try {
            String postRespStr = HttpUtil.postRaw(postUrl, postBody, null);
            JsonNode postRespJNode = JsonUtils.fromJson(postRespStr, JsonNode.class), postRespCode;
            if (null == postRespJNode || null == (postRespCode = postRespJNode.get("code")) || !postRespCode.asText().equals("0")) {
                log.warn("#NFT素材上传:请求失败[paramsStr={},postRespStr={}]", postBody, postRespStr);
                return null;
            }
            return JsonUtils.OM().convertValue(postRespJNode.get("data"), GalaxyNftUploadRespDto.class);
        } catch (Exception e) {
            log.error("Ex.NFT素材上传:请求异常[url={},paramsStr={}],ex:{}", postUrl, postBody, e.getMessage());
            return null;
        }
    }

    /**
     * NFT系列声明
     *
     * @param seriesClaimReqDto GalaxyArtSeriesClaimReqDto
     * @return GalaxyArtSeriesClaimRespDto
     */
    public GalaxyArtSeriesClaimRespDto claimNftSeries(GalaxyArtSeriesClaimReqDto seriesClaimReqDto) {
        String postUrl = serviceGalaxyUrl + "/galaxy/artwork/seriesClaim";
        String postBody = JsonUtils.toJson(seriesClaimReqDto);
        try {
            String postRespStr = HttpUtil.postRaw(postUrl, postBody, null);
            JsonNode postRespJNode = JsonUtils.fromJson(postRespStr, JsonNode.class), postRespCode;
            if (null == postRespJNode || null == (postRespCode = postRespJNode.get("code")) || !postRespCode.asText().equals("0")) {
                log.warn("#NFT系列声明:请求失败[paramsStr={},postRespStr={}]", postBody, postRespStr);
                return null;
            }
            return JsonUtils.OM().convertValue(postRespJNode.get("data"), GalaxyArtSeriesClaimRespDto.class);
        } catch (Exception e) {
            log.error("Ex.NFT系列声明:请求异常[url={},paramsStr={}],ex:{}", postUrl, postBody, e.getMessage());
            return null;
        }
    }

    /* ------------------------------------------------------------------------------------ */
    /* ------------------------------------------------------------------------------------ */

    public GoblinGoodsSkuInfoVo getGoodsSkuInfoVoFromMdb(String skuId) {
        return mongoTemplate.findOne(Query.query(Criteria.where("skuId").is(skuId).and("delFlg").is("0")),
                GoblinGoodsSkuInfoVo.class, GoblinGoodsSkuInfoVo.class.getSimpleName());
    }

    public GoblinGoodsInfoVo getGoodsInfoVoFromMdb(String spuId) {
        return mongoTemplate.findOne(Query.query(Criteria.where("spuId").is(spuId).and("delFlg").is("0")),
                GoblinGoodsInfoVo.class, GoblinGoodsInfoVo.class.getSimpleName());
    }

    /* ------------------------------------------------------------------------------------ */
}
