package com.liquidnet.service.galaxy.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.commons.lang.util.StringUtil;
import com.liquidnet.service.adam.constant.AdamEnum;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.base.SqlMapping;
import com.liquidnet.service.base.constant.MQConst;
import com.liquidnet.service.galaxy.aop.annotation.ControllerLog;
import com.liquidnet.service.galaxy.constant.GalaxyEnum;
import com.liquidnet.service.galaxy.dto.bo.GalaxyUserInfoBo;
import com.liquidnet.service.galaxy.dto.param.GalaxyUserBindStatusQueryReqDto;
import com.liquidnet.service.galaxy.dto.param.GalaxyUserBindStatusQueryRespDto;
import com.liquidnet.service.galaxy.dto.param.GalaxyUserRegisterReqDto;
import com.liquidnet.service.galaxy.dto.param.GalaxyUserRegisterRespDto;
import com.liquidnet.service.galaxy.router.xuper.biz.XuperUserCommonBiz;
import com.liquidnet.service.galaxy.service.IGalaxyUserService;
import com.liquidnet.service.galaxy.utils.GalaxyDataUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.time.LocalDateTime;
import java.util.LinkedList;

/**
 * @author AnJiabin <anjiabin@zhengzai.tv>
 * @version V1.0
 * @Description: TODO
 * @class: GalaxyUserController
 * @Package com.liquidnet.service.galaxy.controller
 * @Copyright: LightNet @ Copyright (c) 2021
 * @date 2022/3/11 12:11
 */
@Api(tags = "NFT用户相关")
@RestController
@RequestMapping("nftUser")
@Validated
@Slf4j
public class GalaxyUserController {
    @Resource(name = "galaxyUserServiceImpl")
    private IGalaxyUserService galaxyUserService;

    @Autowired
    private XuperUserCommonBiz xuperUserCommonBiz;

    @Autowired
    private GalaxyDataUtils dataUtils;

    @ControllerLog(description = "NFT用户注册")
    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "NFT用户注册")
    @PostMapping(value = {"/que/register"})
    public ResponseDto<GalaxyUserRegisterRespDto> register(@Valid @RequestBody GalaxyUserRegisterReqDto reqDto) {
        return galaxyUserService.userRegister(reqDto);
    }

    @ControllerLog(description = "用户绑定状态查询")
    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "用户绑定状态查询")
    @PostMapping(value = {"userBindStatusQuery"})
    public ResponseDto<GalaxyUserBindStatusQueryRespDto> userBindStatusQuery(@Valid @RequestBody GalaxyUserBindStatusQueryReqDto reqDto) {
        return galaxyUserService.userBindStatusQuery(reqDto);
    }

    @ControllerLog(description = "NFT百度用户同步")
    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "NFT百度用户同步")
    @PostMapping(value = {"xuperUserSync"})
    public ResponseDto<String> xuperUserSync(@Valid @RequestBody GalaxyUserRegisterReqDto reqDto) {
        //判断百度链用户是否已经存在
        GalaxyUserInfoBo xuperUserInfoBo = dataUtils.getGalaxyUserInfo(GalaxyEnum.RouterTypeEnum.XUPER.getCode(), reqDto.getUserId());
        if(StringUtil.isNotNull(xuperUserInfoBo)&&StringUtil.isNotEmpty(xuperUserInfoBo.getBlockChainAddress())){
            log.error("NFT百度用户同步 error 该用户百度链用户已经开通成功无需再次开通 reqParam:{}", JsonUtils.toJson(reqDto));
            return ResponseDto.success("已开通成功");
        }
        //获取至信链用户开通信息
        GalaxyUserInfoBo userInfoBo = dataUtils.getGalaxyUserInfo(GalaxyEnum.RouterTypeEnum.ZXINCHAIN.getCode(), reqDto.getUserId());
        if(StringUtil.isNotNull(userInfoBo)&&StringUtil.isNotEmpty(userInfoBo.getBlockChainAddress())){
            reqDto.setRouterType(GalaxyEnum.RouterTypeEnum.XUPER.getCode());
            ResponseDto<GalaxyUserRegisterRespDto> responseDto = xuperUserCommonBiz.userRegister(reqDto);
            if(responseDto.isSuccess()){
                //同步业务账号关联关系到adam
                LinkedList<Object[]> sqlsDataA = CollectionUtil.linkedListObjectArr();
                sqlsDataA.add(new Object[]{
                        reqDto.getUserId(), AdamEnum.BizAcct.NFT_XUPER.name(), responseDto.getData().getBlockChainAddress(), null, null, 1, LocalDateTime.now()
                });
                dataUtils.getQueueUtil().sendMsgByRedis(MQConst.GoblinQueue.SQL_STORE.getKey(),
                        SqlMapping.get("adam_user_busi_acct.add",sqlsDataA)
                );
                return ResponseDto.success("开通成功");
            };
        }
        log.error("NFT百度用户同步 error 该用户至信链数字账户未开通 reqParam:{}", JsonUtils.toJson(reqDto));
        return ResponseDto.failure("开通失败");
    }
}
