package com.liquidnet.service.galaxy.utils;

import com.liquidnet.common.cache.redis.util.RedisUtil;
import com.liquidnet.commons.lang.util.*;
import com.liquidnet.service.base.SqlMapping;
import com.liquidnet.service.base.constant.MQConst;
import com.liquidnet.service.galaxy.biz.GalaxyBeanTransferBiz;
import com.liquidnet.service.galaxy.constant.GalaxyConstant;
import com.liquidnet.service.galaxy.constant.GalaxyEnum;
import com.liquidnet.service.galaxy.dto.bo.*;
import com.liquidnet.service.galaxy.dto.vo.mongo.*;
import com.mongodb.client.result.UpdateResult;
import lombok.extern.slf4j.Slf4j;
import org.bson.Document;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
public abstract class AbstractDataUtils {
    public abstract RedisUtil getRedisUtil();
    
    public abstract MongoTemplate getMongoTemplate();
    
    public abstract QueueUtil getQueueUtil();

    private long keyExpireTime = 3600*24*30;

    public void setGalaxyUserInfo(String routerType,String userId, GalaxyUserInfoBo userInfoBo) {
        String userType = GalaxyEnum.RegisterTypeEnum.PERSON.getCode();
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_USER.concat(routerType).concat(":") + userId,userInfoBo,keyExpireTime);
        //入库mongo
        GalaxyUserInfoVo userInfoVo = GalaxyUserInfoVo.getNew();
        BeanUtil.copy(userInfoBo,userInfoVo);

        userInfoVo.setCreatedAt(LocalDateTime.now());
        this.getMongoTemplate().save(userInfoVo,GalaxyUserInfoVo.class.getSimpleName());
//        insert into galaxy_user_info (mid, user_id, user_name, user_type, mobile, id_card_type
//                , id_card, mnemonic, index,user_identification, user_pub_key
//                , user_pri_key, block_chain_address, router_type,created_at, updated_at)
        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_user_info.insert"),
                    new Object[]{userId,userInfoBo.getUserName(),userType,userInfoBo.getMobile(),userInfoBo.getIdCardType()
                                 ,userInfoBo.getIdCard(),userInfoBo.getMnemonic(),userInfoBo.getIndex(),userInfoBo.getUserIdentification(),userInfoBo.getUserPubKey()
                                 ,userInfoBo.getUserPriKey(),userInfoBo.getBlockChainAddress(),userInfoBo.getRouterType(),new Date(),null
                    }
                    , MQConst.GalaxyQueue.SQL_USER_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#setGalaxyUserInfo error ==> MESSAGE:{}",e.getMessage());
        }
    }

    public GalaxyUserInfoBo getGalaxyUserInfo(String routerType,String userId) {
        Object obj = this.getRedisUtil().get(GalaxyConstant.REDIS_KEY_GALAXY_USER.concat(routerType).concat(":") + userId);
        if(obj!=null){
            return (GalaxyUserInfoBo) obj;
        }else {
            GalaxyUserInfoVo userInfoVo = this.getMongoTemplate().findOne(Query.query(Criteria.where("userId").is(userId)), GalaxyUserInfoVo.class, GalaxyUserInfoVo.class.getSimpleName());
            if (userInfoVo == null) {
                return null;
            }
            GalaxyUserInfoBo userInfoBo = GalaxyUserInfoBo.getNew();
            BeanUtil.copy(userInfoVo,userInfoBo);

            this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_USER.concat(routerType).concat(":") + userId, userInfoBo);
            return userInfoBo;
        }
    }

    public void setSeriesNftUploadBo(String routerType,String skuId, GalaxySeriesNftUploadBo seriesNftUploadBo) {
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT_UPLOAD.concat(routerType).concat(":") + skuId,seriesNftUploadBo,keyExpireTime);

        //mongo入库
        GalaxySeriesNftUploadVo seriesNftUploadVo = GalaxySeriesNftUploadVo.getNew();
        BeanUtil.copy(seriesNftUploadBo,seriesNftUploadVo);
        seriesNftUploadVo.setCreatedAt(LocalDateTime.now());
        this.getMongoTemplate().save(seriesNftUploadVo,GalaxySeriesNftUploadVo.class.getSimpleName());
    }

    public GalaxySeriesNftUploadBo getSeriesNftUploadBo(String routerType,String skuId) {
        Object obj = this.getRedisUtil().get(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT_UPLOAD.concat(routerType).concat(":") + skuId);
        if(obj!=null){
            return (GalaxySeriesNftUploadBo) obj;
        }else {
            //设置查询参数
            Criteria criteria = Criteria.where("routerType").is(routerType);
            String userId = CurrentUtil.getCurrentUid();
            if(StringUtil.isNotEmpty(skuId)){
                criteria = criteria.and("skuId").is(skuId);
            }

            GalaxySeriesNftUploadVo seriesNftUploadVo = this.getMongoTemplate().findOne(Query.query(criteria), GalaxySeriesNftUploadVo.class, GalaxySeriesNftUploadVo.class.getSimpleName());
            if (seriesNftUploadVo == null) {
                return null;
            }
            GalaxySeriesNftUploadBo seriesNftUploadBo = GalaxySeriesNftUploadBo.getNew();
            BeanUtil.copy(seriesNftUploadVo,seriesNftUploadBo);

            this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT_UPLOAD.concat(routerType).concat(":") + skuId, seriesNftUploadBo);
            return seriesNftUploadBo;
        }
    }

    public void setSeriesInfoBo(String routerType,String seriesCode, GalaxySeriesInfoBo seriesInfoBo) {
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES.concat(routerType).concat(":") + seriesCode,seriesInfoBo,keyExpireTime);

//        insert into galaxy_series_info (mid, series_info_id, series_name, series_code, series_id, total_count
//                , crt_count,series_claim_task_id, series_claim_status, cover_url, series_desc
//                , nft_count,router_type, created_at, updated_at)
        Integer nftCount = seriesInfoBo.getNftCount();//默认为一个系列包含一个NFT

        //mongo入库
        GalaxySeriesInfoVo seriesInfoVo = GalaxySeriesInfoVo.getNew();
        BeanUtil.copy(seriesInfoBo,seriesInfoVo);
        seriesInfoVo.setCreatedAt(DateUtil.format(LocalDateTime.now(), DateUtil.Formatter.yyyyMMddHHmmss));
        this.getMongoTemplate().save(seriesInfoVo,GalaxySeriesInfoVo.class.getSimpleName());

        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_series_info.insert"),
                    new Object[]{seriesInfoBo.getSeriesInfoId(),seriesInfoBo.getSeriesName(),seriesCode,null,seriesInfoBo.getTotalCount()
                            ,seriesInfoBo.getCrtCount(),seriesInfoBo.getSeriesClaimTaskId(),seriesInfoBo.getSeriesClaimStatus(),seriesInfoBo.getCoverUrl(),seriesInfoBo.getSeriesDesc()
                            ,nftCount,seriesInfoBo.getRouterType(),LocalDateTime.now(),null
                    }
                    , MQConst.GalaxyQueue.SQL_SERIES_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#setSeriesInfoBo error ==> MESSAGE:{}",e.getMessage());
        }
    }

    public GalaxySeriesInfoBo getSeriesInfoBo(String routerType, String seriesCode) {
        Object obj = this.getRedisUtil().get(GalaxyConstant.REDIS_KEY_GALAXY_SERIES.concat(routerType).concat(":") + seriesCode);
        if(obj!=null){
            return (GalaxySeriesInfoBo) obj;
        }else {
            GalaxySeriesInfoVo seriesInfoVo = this.getMongoTemplate().findOne(Query.query(Criteria.where("seriesCode").is(seriesCode)), GalaxySeriesInfoVo.class, GalaxySeriesInfoVo.class.getSimpleName());
            if (seriesInfoVo == null) {
                return null;
            }
            GalaxySeriesInfoBo seriesInfoBo = GalaxySeriesInfoBo.getNew();
            BeanUtil.copy(seriesInfoVo,seriesInfoBo);
            this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES.concat(routerType).concat(":") + seriesCode, seriesInfoBo);
            return seriesInfoBo;
        }
    }

    public void setSeriesNftInfoBo(String routerType, String skuId, GalaxySeriesInfoBo seriesInfoBo, GalaxySeriesNftInfoBo seriesNftInfoBo) {
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT.concat(routerType).concat(":") + skuId,seriesNftInfoBo,keyExpireTime);

        //初始化nft发行绑定索引,未发行则初始化为0，
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_PUBLISH_NFT.concat(routerType).concat(":").concat(skuId),seriesNftInfoBo.getNftIdBeginIndex()-1);

        //mongo入库
        GalaxySeriesNftInfoVo seriesNftInfoVo = GalaxySeriesNftInfoVo.getNew();
        BeanUtil.copy(seriesNftInfoBo,seriesNftInfoVo);
        seriesNftInfoVo.setCreatedAt(LocalDateTime.now());
        this.getMongoTemplate().save(seriesNftInfoVo,GalaxySeriesNftInfoVo.class.getSimpleName());

        try{
//            insert into galaxy_series_nft_info (mid, series_nft_info_id, series_info_id, sku_id, series_name, series_code
//                    ,series_id, nft_total_count, nft_crt_count,nft_id_begin,nft_id_begin_index,nft_publish_number,original_nft_url
//                    , original_display_url,author, nft_name, nft_url, display_url
//                    , nft_desc, nft_flag, sell_count, nft_hash,router_type
//                    , created_at, updated_at)
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_series_nft_info.insert"),
                    new Object[]{seriesNftInfoBo.getSeriesNftInfoId(),seriesInfoBo.getSeriesInfoId(),skuId,seriesInfoBo.getSeriesName(),seriesInfoBo.getSeriesCode()
                            ,seriesInfoBo.getSeriesId(),seriesNftInfoBo.getNftTotalCount(),seriesNftInfoBo.getNftCrtCount(),seriesNftInfoBo.getNftIdBegin(),seriesNftInfoBo.getNftIdBeginIndex(),seriesNftInfoBo.getNftPublishNumber(),seriesNftInfoBo.getOriginalNftUrl()
                            ,seriesNftInfoBo.getOriginalDisplayUrl(),seriesNftInfoBo.getAuthor(),seriesNftInfoBo.getNftName(),seriesNftInfoBo.getNftUrl(),seriesNftInfoBo.getDisplayUrl()
                            ,seriesNftInfoBo.getNftDesc(),seriesNftInfoBo.getNftFlag(),seriesNftInfoBo.getSellCount(),seriesNftInfoBo.getNftHash(),seriesInfoBo.getRouterType()
                            ,LocalDateTime.now(),null
                    }
                    , MQConst.GalaxyQueue.SQL_SERIES_INFO.getKey());
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#setSeriesInfoBo error ==> MESSAGE:{}",e.getMessage());
        }
    }


    public GalaxySeriesNftInfoBo getSeriesNftInfoBo(String routerType, String skuId) {
        Object obj = this.getRedisUtil().get(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT.concat(routerType).concat(":") + skuId);
        if(obj!=null){
            return (GalaxySeriesNftInfoBo) obj;
        }else {
            GalaxySeriesNftInfoVo seriesNftInfoVo = this.getMongoTemplate().findOne(Query.query(Criteria.where("skuId").is(skuId)), GalaxySeriesNftInfoVo.class, GalaxySeriesNftInfoVo.class.getSimpleName());
            if (seriesNftInfoVo == null) {
                return null;
            }
            GalaxySeriesNftInfoBo seriesNftInfoBo = GalaxySeriesNftInfoBo.getNew();
            BeanUtil.copy(seriesNftInfoVo,seriesNftInfoBo);
            this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT.concat(routerType).concat(":") + skuId, seriesNftInfoBo);
            return seriesNftInfoBo;
        }
    }

    public void updateSeriesClaimStatus(String routerType,String seriesCode, GalaxySeriesInfoBo seriesInfoBo) {
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES.concat(routerType).concat(":") + seriesCode,seriesInfoBo,keyExpireTime);

        Query query = Query.query(Criteria.where("seriesCode").is(seriesInfoBo.getSeriesCode()));
        Update update = Update.fromDocument(Document.parse(JsonUtils.toJson(seriesInfoBo)));
        String nowTimeStr = DateUtil.Formatter.yyyyMMddHHmmss.format(LocalDateTime.now());
        update.set("updatedAt",nowTimeStr);
        UpdateResult result = this.getMongoTemplate().updateFirst(query,update, GalaxySeriesInfoVo.class,GalaxySeriesInfoVo.class.getSimpleName());
        log.info("updateSeriesClaimStatus result:{}",result.toString());

        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_series_info.updateSeriesIdAndStatus"),
                    new Object[]{seriesInfoBo.getSeriesId(),seriesInfoBo.getChainTimestamp(),seriesInfoBo.getTradeHash(),seriesInfoBo.getSeriesClaimStatus(),nowTimeStr,seriesInfoBo.getSeriesCode()}
                    , MQConst.GalaxyQueue.SQL_SERIES_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#updateSeriesClaimStatus error ==> MESSAGE:{}",e.getMessage());
        }
        //更新系列对应的系列nft状态
        List<String> skuIdList = this.getSkuIdListBySeriesCode(seriesCode);
        skuIdList.stream().forEach(skuId -> {
            GalaxySeriesNftInfoBo seriesNftInfoBo = this.getSeriesNftInfoBo(routerType,skuId);
            seriesNftInfoBo.setSeriesId(seriesInfoBo.getSeriesId());
            seriesNftInfoBo.setUpdatedAt(LocalDateTime.now());
            this.updateSeriesNftStatus(routerType,skuId,seriesNftInfoBo);
        });
    }

    public void updateSeriesNftStatus(String routerType,String skuId, GalaxySeriesNftInfoBo seriesNftInfoBo) {
        //更新系列nft状态
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT.concat(routerType).concat(":") + skuId,seriesNftInfoBo,keyExpireTime);

        Query query = Query.query(Criteria.where("skuId").is(skuId));
        Update update = Update.fromDocument(Document.parse(JsonUtils.toJson(seriesNftInfoBo)));
        String nowTimeStr = DateUtil.Formatter.yyyyMMddHHmmss.format(seriesNftInfoBo.getUpdatedAt());
        update.set("routerType",routerType);
        update.set("updatedAt",nowTimeStr);
        UpdateResult result = this.getMongoTemplate().updateFirst(query,update, GalaxySeriesNftInfoVo.class,GalaxySeriesNftInfoVo.class.getSimpleName());
        log.info("updateSeriesNftStatus result:{}",result.toString());

        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_series_nft_info.updateSeriesIdAndStatus"),
                    new Object[]{seriesNftInfoBo.getSeriesId(),nowTimeStr,seriesNftInfoBo.getSkuId()}
                    , MQConst.GalaxyQueue.SQL_SERIES_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#updateSeriesNftStatus error ==> MESSAGE:{}",e.getMessage());
        }
    }

    //根据系列Id获取
    public List<String> getSkuIdListBySeriesCode(String seriesCode){
        //根据演出id获取用户信息 只查询5条
        List<String> skuIdList = new ArrayList<>();
        //获取总数
        Criteria criteria = Criteria.where("seriesCode").is(seriesCode);

        Query query = Query.query(criteria);

        long startTime = System.currentTimeMillis();
        List<GalaxySeriesNftInfoVo> seriesNftInfoVoList = this.getMongoTemplate().find(query, GalaxySeriesNftInfoVo.class, GalaxySeriesNftInfoVo.class.getSimpleName());
        long endTime = System.currentTimeMillis();
        log.info("getSkuIdListBySeriesCode 根据系列code获取skuId 耗时：{}ms",endTime-startTime);

        //根据用户id获取头像信息
        if(StringUtil.isNotNull(seriesNftInfoVoList)){
            skuIdList = seriesNftInfoVoList.stream().map(GalaxySeriesNftInfoVo::getSkuId).collect(Collectors.toList());
        }
        return skuIdList;
    }

    public void setNftOrderBo(String routerType,String nftOrderPayId, GalaxyNftOrderBo nftOrderBo) {

        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_TRADE_ORDER.concat(routerType).concat(":") + nftOrderPayId,nftOrderBo,keyExpireTime);
        /**
         * 入库mongo
         */
        GalaxyNftOrderVo nftOrderVo = GalaxyNftOrderVo.getNew();
        BeanUtil.copy(nftOrderBo,nftOrderVo);
        this.getMongoTemplate().save(nftOrderVo,GalaxyNftOrderVo.class.getSimpleName());

//        insert into galaxy_nft_order_info (nft_order_pay_id, user_id, sku_id, series_name, series_id,
//                series_code, nft_id,nft_price, nft_publish_task_id, nft_publish_from_address,
//                nft_publish_to_address,nft_publish_chain_timestamp, nft_publish_status, nft_publish_trade_hash,nft_buy_task_id,
//                nft_buy_from_address, nft_buy_to_address, nft_buy_chain_timestamp,nft_buy_status, nft_buy_trade_hash,
//                nft_buy_pay_task_id, router_type, created_at,updated_at)
        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_nft_order_info.insert"),
                    new Object[]{nftOrderPayId,nftOrderBo.getUserId(),nftOrderBo.getSkuId(),nftOrderBo.getSeriesName(),nftOrderBo.getSeriesId()
                            ,nftOrderBo.getSeriesCode(),nftOrderBo.getNftId(),nftOrderBo.getNftName(),nftOrderBo.getNftPrice(),nftOrderBo.getNftPublishTaskId(),nftOrderBo.getNftPublishFromAddress()
                            ,nftOrderBo.getNftPublishToAddress(),nftOrderBo.getNftPublishChainTimestamp(),nftOrderBo.getNftPublishStatus(),nftOrderBo.getNftPublishTradeHash(),nftOrderBo.getNftBuyTaskId()
                            ,nftOrderBo.getNftBuyFromAddress(),nftOrderBo.getNftBuyToAddress(),nftOrderBo.getNftBuyChainTimestamp(),nftOrderBo.getNftBuyStatus(),nftOrderBo.getNftBuyTradeHash(),
                            nftOrderBo.getNftBuyPayTaskId(),nftOrderBo.getRouterType(),LocalDateTime.now(),null
                    }
                    , MQConst.GalaxyQueue.SQL_NFT_ORDER_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#setNftOrderBo error ==> MESSAGE:{}",e.getMessage());
        }

        //更新系列已发行信息
        try {
            this.updateSeriesCrtCount(routerType,nftOrderBo.getSeriesCode());
        } catch (Exception e) {
            log.error(e.getMessage(),e);
        }
        //更新系列nft已发行信息
        try {
            this.updateSeriesNftCrtCount(routerType,nftOrderBo.getSkuId());
        } catch (Exception e) {
            log.error(e.getMessage(),e);
        }

    }

    public GalaxyNftOrderBo getNftOrderBo(String routerType, String nftOrderPayId) {
        Object obj = this.getRedisUtil().get(GalaxyConstant.REDIS_KEY_GALAXY_TRADE_ORDER.concat(routerType).concat(":") + nftOrderPayId);
        if(obj!=null){
            return (GalaxyNftOrderBo) obj;
        }else {
            GalaxyNftOrderVo nftOrderVo = this.getMongoTemplate().findOne(Query.query(Criteria.where("nftOrderPayId").is(nftOrderPayId)), GalaxyNftOrderVo.class, GalaxyNftOrderVo.class.getSimpleName());
            if (nftOrderVo == null) {
                return null;
            }
            GalaxyNftOrderBo nftOrderBo = GalaxyNftOrderBo.getNew();
            BeanUtil.copy(nftOrderVo,nftOrderBo);
            this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_TRADE_ORDER.concat(routerType).concat(":") + nftOrderPayId, nftOrderBo);
            return nftOrderBo;
        }
    }

    /**
     * 更新nft订单发行数据
     * @param routerType
     * @param nftOrderPayId
     * @param nftOrderBo
     */
    public void updateNftOrderPublishInfo(String routerType,String nftOrderPayId,GalaxyNftOrderBo nftOrderBo) {
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_TRADE_ORDER.concat(routerType).concat(":") + nftOrderPayId,nftOrderBo,keyExpireTime);

        Query query = Query.query(Criteria.where("nftOrderPayId").is(nftOrderBo.getNftOrderPayId()));
        Update update = Update.fromDocument(Document.parse(JsonUtils.toJson(nftOrderBo)));
        String nowTimeStr = DateUtil.Formatter.yyyyMMddHHmmss.format(nftOrderBo.getUpdatedAt());
        update.set("updatedAt",nowTimeStr);
        UpdateResult result = this.getMongoTemplate().updateFirst(query,update, GalaxyNftOrderVo.class,GalaxyNftOrderVo.class.getSimpleName());
        log.info("updateNftOrderPublishInfo result:{}",result.toString());

        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_nft_order_info.updateNftPublishInfo"),
                    new Object[]{nftOrderBo.getNftId(),nftOrderBo.getNftPublishTaskId(),nftOrderBo.getNftPublishChainTimestamp()
                            ,nftOrderBo.getNftPublishStatus(),nftOrderBo.getNftPublishTradeHash()
                            ,LocalDateTime.now(),nftOrderBo.getNftOrderPayId()}
                    , MQConst.GalaxyQueue.SQL_NFT_ORDER_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#updateNftOrderPublishInfo error ==> MESSAGE:{}",e.getMessage());
        }

        if(nftOrderBo.getNftPublishStatus().equalsIgnoreCase(GalaxyEnum.TaskStatusEnum.TASK_SUCCESS.getCode())){
            GalaxyNftTradeVo nftTradeVo = GalaxyBeanTransferBiz.buildNftTradeVo(GalaxyEnum.TradeTypeEnum.TRADE_PUBLISH,nftOrderBo);
            this.setNftTradeInfo(routerType,nftTradeVo);
        }
    }

    /**
     * 更新nft订单购买数据
     * @param routerType
     * @param nftOrderPayId
     * @param nftOrderBo
     */
    public void updateNftOrderBuyInfo(String routerType,String nftOrderPayId,GalaxyNftOrderBo nftOrderBo) {
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_TRADE_ORDER.concat(routerType).concat(":") + nftOrderPayId,nftOrderBo,keyExpireTime);

        Query query = Query.query(Criteria.where("nftOrderPayId").is(nftOrderPayId));
        Update update = Update.fromDocument(Document.parse(JsonUtils.toJson(nftOrderBo)));
        String nowTimeStr = DateUtil.Formatter.yyyyMMddHHmmss.format(nftOrderBo.getUpdatedAt());
        update.set("updatedAt",nowTimeStr);
        UpdateResult result = this.getMongoTemplate().updateFirst(query,update, GalaxyNftOrderVo.class,GalaxyNftOrderVo.class.getSimpleName());
        log.info("updateNftOrderBuyInfo result:{}",result.toString());

        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_nft_order_info.updateNftBuyInfo"),
                    new Object[]{nftOrderBo.getNftBuyTaskId(),nftOrderBo.getNftBuyChainTimestamp(),nftOrderBo.getNftBuyStatus()
                            ,nftOrderBo.getNftBuyTradeHash(),nftOrderBo.getNftBuyPayTaskId()
                            ,LocalDateTime.now(),nftOrderBo.getNftOrderPayId()}
                    , MQConst.GalaxyQueue.SQL_NFT_ORDER_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#setNftTradeInfo error ==> MESSAGE:{}",e.getMessage());
        }

        if(nftOrderBo.getNftBuyStatus().equalsIgnoreCase(GalaxyEnum.TaskStatusEnum.TASK_SUCCESS.getCode())){
            GalaxyNftTradeVo nftTradeVo = GalaxyBeanTransferBiz.buildNftTradeVo(GalaxyEnum.TradeTypeEnum.TRADE_BUY,nftOrderBo);
            this.setNftTradeInfo(routerType,nftTradeVo);
        }
    }

    public void setGalaxyNftOrderBindBo(String routerType,String nftOrderPayId, GalaxyNftOrderBindBo nftOrderBindBo) {
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_PUBLISH_ORDER.concat(routerType).concat(":") + nftOrderPayId,nftOrderBindBo,keyExpireTime);
        //入库mongo
        GalaxyNftOrderBindVo nftOrderBindVo = GalaxyNftOrderBindVo.getNew();
        BeanUtil.copy(nftOrderBindBo,nftOrderBindVo);
        this.getMongoTemplate().save(nftOrderBindVo,GalaxyNftOrderBindVo.class.getSimpleName());
    }

    public GalaxyNftOrderBindBo getGalaxyNftOrderBindBo(String routerType,String nftOrderPayId) {
        String redisKey = GalaxyConstant.REDIS_KEY_GALAXY_PUBLISH_ORDER.concat(routerType).concat(":") + nftOrderPayId;
        Object obj = this.getRedisUtil().get(redisKey);
        if(obj!=null){
            return (GalaxyNftOrderBindBo) obj;
        }else {
            GalaxyNftOrderBindVo nftOrderBindVo = this.getMongoTemplate().findOne(Query.query(Criteria.where("nftOrderPayId").is(nftOrderPayId)), GalaxyNftOrderBindVo.class, GalaxyNftOrderBindVo.class.getSimpleName());
            if (nftOrderBindVo == null) {
                return null;
            }
            GalaxyNftOrderBindBo nftOrderBindBo = GalaxyNftOrderBindBo.getNew();
            BeanUtil.copy(nftOrderBindVo,nftOrderBindBo);

            this.getRedisUtil().set(redisKey, nftOrderBindBo,keyExpireTime);
            return nftOrderBindBo;
        }
    }

    public void setNftTradeInfo(String routerType, GalaxyNftTradeVo nftTradeVo) {
        try{
            /**
             * 发行成功记录入库mongo
             */
            this.getMongoTemplate().save(nftTradeVo,GalaxyNftTradeVo.class.getSimpleName());

//            insert into galaxy_nft_trade_info (user_id, nft_id, nft_name, series_name, series_id,
//                    series_code, trade_hash,chain_timestamp, from_address, to_address,
//                    trade_price, trade_type, router_type,created_at, updated_at)
            /**
             * 发行成功记录入库mysql
             */
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_nft_trade_info.insert"),
                    new Object[]{nftTradeVo.getUserId(),nftTradeVo.getNftId(),nftTradeVo.getNftName(),nftTradeVo.getSeriesName(),nftTradeVo.getSeriesId(),
                            nftTradeVo.getSeriesCode(),nftTradeVo.getTradeHash(),nftTradeVo.getChainTimestamp(),nftTradeVo.getFromAddress(),nftTradeVo.getToAddress(),
                            nftTradeVo.getTradePrice(),nftTradeVo.getTradeType(),nftTradeVo.getRouterType(),LocalDateTime.now(),nftTradeVo.getUpdatedAt()}
                    , MQConst.GalaxyQueue.SQL_NFT_TRADE_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#setNftTradeInfo error ==> MESSAGE:{}",e.getMessage());
        }
    }

    public void setNftOrderFailLogVo(String routerType, GalaxyNftOrderFailLogVo nftOrderFailLogVo) {
        /**
         * 入库mongo
         */
        nftOrderFailLogVo.setCreatedAt(LocalDateTime.now());
        this.getMongoTemplate().save(nftOrderFailLogVo,GalaxyNftOrderFailLogVo.class.getSimpleName());

//        insert into galaxy_nft_order_fail_log (nft_order_pay_id, user_id, nft_id, nft_name, series_name,
//                                              series_id,series_code, taskId, fail_reason_desc, fail_reason_desc_second,
//                                              deal_with_status,trade_type, router_type, created_at, updated_at)
        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_nft_order_fail_log.insert"),
                    new Object[]{nftOrderFailLogVo.getNftOrderPayId(),nftOrderFailLogVo.getUserId(),nftOrderFailLogVo.getNftId(),nftOrderFailLogVo.getNftName(),nftOrderFailLogVo.getSeriesName(),
                            nftOrderFailLogVo.getSeriesId(),nftOrderFailLogVo.getSeriesCode(),nftOrderFailLogVo.getTaskId(),nftOrderFailLogVo.getFailReasonDesc(),nftOrderFailLogVo.getFailReasonDescSecond(),
                            nftOrderFailLogVo.getDealWithStatus(),nftOrderFailLogVo.getTradeType(),nftOrderFailLogVo.getRouterType(),LocalDateTime.now(),null
                    }
                    , MQConst.GalaxyQueue.SQL_NFT_ORDER_FAIL_LOG.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#setNftOrderFailLogVo error ==> MESSAGE:{}",e.getMessage());
        }
    }

    public long incrNftIdNo(String routerType,String skuId) {
        return this.getRedisUtil().incr(GalaxyConstant.REDIS_KEY_GALAXY_PUBLISH_NFT.concat(routerType).concat(":").concat(skuId), 1);
    }

    public long getNftIdNo(String routerType,String skuId) {
        Object o = this.getRedisUtil().get(GalaxyConstant.REDIS_KEY_GALAXY_PUBLISH_NFT.concat(routerType).concat(":").concat(skuId));
        return o==null?-1:(int)o;
    }

    /**
     * 更新系列已购买信息
     * @param routerType
     * @param seriesCode
     */
    public void updateSeriesCrtCount(String routerType,String seriesCode) {
        GalaxySeriesInfoBo seriesInfoBo = this.getSeriesInfoBo(routerType,seriesCode);
        seriesInfoBo.setCrtCount(seriesInfoBo.getCrtCount()+1);
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES.concat(routerType).concat(":") + seriesCode,seriesInfoBo,keyExpireTime);

        Query query = Query.query(Criteria.where("seriesCode").is(seriesInfoBo.getSeriesCode()));
        Update update = Update.fromDocument(Document.parse(JsonUtils.toJson(seriesInfoBo)));
        String nowTimeStr = DateUtil.Formatter.yyyyMMddHHmmss.format(LocalDateTime.now());
        update.set("updatedAt",nowTimeStr);
        UpdateResult result = this.getMongoTemplate().updateFirst(query,update, GalaxySeriesInfoVo.class,GalaxySeriesInfoVo.class.getSimpleName());
        log.info("updateSeriesCrtCount result:{}",result.toString());

        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_series_info.updateSeriesCrtCount"),
                    new Object[]{seriesInfoBo.getCrtCount(),LocalDateTime.now(),seriesCode}
                    , MQConst.GalaxyQueue.SQL_SERIES_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#updateSeriesCrtCount error ==> MESSAGE:{}",e.getMessage());
        }
    }

    /**
     * 更新系列nft已购买信息
     * @param routerType
     * @param skuId
     */
    public void updateSeriesNftCrtCount(String routerType,String skuId) {
        GalaxySeriesNftInfoBo seriesNftInfoBo = this.getSeriesNftInfoBo(routerType,skuId);
        seriesNftInfoBo.setNftCrtCount(seriesNftInfoBo.getNftCrtCount()+1);
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT.concat(routerType).concat(":") + skuId,seriesNftInfoBo,keyExpireTime);

        Query query = Query.query(Criteria.where("skuId").is(skuId));
        Update update = Update.fromDocument(Document.parse(JsonUtils.toJson(seriesNftInfoBo)));
        String nowTimeStr = DateUtil.Formatter.yyyyMMddHHmmss.format(LocalDateTime.now());
        update.set("updatedAt",nowTimeStr);
        UpdateResult result = this.getMongoTemplate().updateFirst(query,update, GalaxySeriesNftInfoVo.class,GalaxySeriesNftInfoVo.class.getSimpleName());
        log.info("updateSeriesNftCrtCount result:{}",result.toString());

        try{
            this.getQueueUtil().sendMySqlRedis(
                    SqlMapping.get("galaxy_series_nft_info.updateSeriesNftCrtCount"),
                    new Object[]{seriesNftInfoBo.getNftCrtCount(),LocalDateTime.now(),skuId}
                    , MQConst.GalaxyQueue.SQL_SERIES_INFO.getKey()
            );
        }catch(Exception e){
            log.error(e.getMessage(),e);
            log.error("#GalaxySeriesNftInfoBo error ==> MESSAGE:{}",e.getMessage());
        }
    }

    public void setSeriesNftUploadTempSecretBo(String uploadSeriesName,GalaxySeriesNftUploadTempSecretBo seriesNftUploadTempSecretBo){
        long keyExpireTime = 320;
        this.getRedisUtil().set(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT_UPLOAD_TEMP_SECRET.concat(":") + uploadSeriesName,seriesNftUploadTempSecretBo,keyExpireTime);
    }

    public GalaxySeriesNftUploadTempSecretBo getSeriesNftUploadTempSecretBo(String uploadSeriesName){
        Object obj = this.getRedisUtil().get(GalaxyConstant.REDIS_KEY_GALAXY_SERIES_NFT_UPLOAD_TEMP_SECRET.concat(":") + uploadSeriesName);
        if(obj!=null){
            return (GalaxySeriesNftUploadTempSecretBo) obj;
        }else{
            return null;
        }
    }
}
