package com.liquidnet.service.kylin.controller;

import com.liquidnet.common.cache.redis.util.RedisUtil;
import com.liquidnet.commons.lang.core.JwtValidator;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.DateUtil;
import com.liquidnet.commons.lang.util.HttpUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.kylin.dto.param.CheckPerformanceRelationParam;
import com.liquidnet.service.kylin.dto.param.KylinStationCheckOrderParam;
import com.liquidnet.service.kylin.dto.vo.*;
import com.liquidnet.service.kylin.service.IKylinOrderTicketEntitiesService;
import com.liquidnet.service.kylin.utils.DataUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.util.CollectionUtils;
import org.springframework.util.DigestUtils;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

@Api(tags = "验票")
@Slf4j
@RestController
@RequestMapping("station")
public class KylinStationController {
    @Autowired
    Environment environment;
    @Autowired
    JwtValidator jwtValidator;
    @Autowired
    MongoTemplate mongoTemplate;
    @Autowired
    RedisUtil redisUtil;
    @Autowired
    DataUtils dataUtils;
    @Autowired
    IKylinOrderTicketEntitiesService kylinOrderTicketEntitiesService;

    @ApiOperation(value = "手机号密码登录")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", dataType = "String", name = "mobile", value = "手机号", example = "11122233344"),
            @ApiImplicitParam(type = "form", dataType = "String", name = "passwd", value = "密码", example = "123456"),
    })
    @PostMapping("login")
    public ResponseDto<String> login(@RequestParam String mobile, @RequestParam String passwd) {
        log.info("mobile:{},passwd:{}", mobile, passwd);

        return this.loginVerification(mobile, passwd, true);
    }

    @ApiOperation(value = "手机号验证码登录")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "body", required = true, dataType = "String", name = "mobile", value = "手机号", example = "11122233344"),
            @ApiImplicitParam(type = "body", required = true, dataType = "String", name = "code", value = "验证码", example = "111111"),
    })
    @PostMapping("login/sms")
    public ResponseDto<String> loginBySms(@RequestParam String mobile, @RequestParam String code) {
        log.info("mobile:{},code:{}", mobile, code);

        return this.loginVerification(mobile, code, false);
    }

    @ApiOperation(value = "登出")
    @PostMapping("out")
    public ResponseDto<String> out() {
        String checkUserId = CurrentUtil.getCurrentUid();

        String token = CurrentUtil.getToken();

        String ssoKeyUidM5Token = jwtValidator.getSsoRedisKey().concat(CurrentUtil.getCurrentUid()).concat(
                DigestUtils.md5DigestAsHex(token.getBytes(StandardCharsets.UTF_8))
        );

        log.info("###logout:checkUserId:{}\nssoKey:{}\ntoken:{}", checkUserId, ssoKeyUidM5Token, token);

        redisUtil.set(ssoKeyUidM5Token, false);

        return ResponseDto.success();
    }

    /* ------------------------------------------------------------------ */

    @ApiOperation(value = "演出列表")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "body", required = true, dataType = "String", name = "mod", value = "模块[recent-近期的,down-已下载,over-历史]", allowableValues = "recent,down,over"),
    })
    @GetMapping("performances/{mod}")
    public ResponseDto<List<KylinStationPerformanceVo>> performances(@PathVariable String mod) {
        List<KylinStationPerformanceVo> voList = new ArrayList<>();
        String currentUid = CurrentUtil.getCurrentUid();
        try {
            switch (mod) {
                case "recent":
                    log.info(":::performances/recent:{}", currentUid);
                    // 查取当前用户下关联演出ID列表
                    KylinCheckUserPerformanceVo checkUserRelationVo = dataUtils.getCheckUserRelationVo(currentUid);

                    if (null != checkUserRelationVo) {
                        List<CheckPerformanceRelationParam> performanceRelationList = checkUserRelationVo.getRelationParams();
                        if (!CollectionUtils.isEmpty(performanceRelationList)) {
                            LocalDateTime tmpDt = LocalDateTime.of(LocalDate.now(), LocalTime.of(0, 0, 0, 0));
                            String tmpDtStr = DateUtil.format(tmpDt, DateUtil.Formatter.yyyyMMddHHmmss);

                            // 转换Map<performanceId, canDownTime>
                            Map<String, String> performanceRelationMap = performanceRelationList.stream().collect(Collectors.toMap(CheckPerformanceRelationParam::getPerformanceId, CheckPerformanceRelationParam::getCanDownTime));

                            // 查取指定条件的演出列表
                            Query performancesVoQuery = Query.query(Criteria.where("performancesId").in(
                                    performanceRelationList.stream().map(CheckPerformanceRelationParam::getPerformanceId).toArray()
                            ).and("timeEnd").gt(tmpDtStr));
                            performancesVoQuery.with(Sort.by(Sort.Order.asc("timeEnd"), Sort.Order.desc("sort")));
                            voList = mongoTemplate.find(performancesVoQuery, KylinStationPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());

                            // 查取演出对应的订单票明细
                            Query orderTicketEntitiesVoQuery = Query.query(Criteria.where("performanceId").in(
                                    voList.stream().map(KylinStationPerformanceVo::getPerformancesId).toArray()
                            ).and("isPayment").is(1));
                            List<KylinOrderTicketEntitiesVo> oteVoList = mongoTemplate.find(orderTicketEntitiesVoQuery, KylinOrderTicketEntitiesVo.class, KylinOrderTicketEntitiesVo.class.getSimpleName());
                            // 转换订单票明细结构为Map<performanceId, List<KylinOrderTicketEntitiesVo>>
                            Map<String, List<KylinOrderTicketEntitiesVo>> oteVoMap = oteVoList.stream().collect(Collectors.groupingBy(KylinOrderTicketEntitiesVo::getPerformanceId));

                            // 查取订单对应票种
                            Query performanceTicketVoQuery = Query.query(Criteria.where("ticketsId").in(
                                    oteVoList.stream().filter(distinctByKey(KylinOrderTicketEntitiesVo::getTicketId)).map(KylinOrderTicketEntitiesVo::getTicketId).toArray()
                            ));
                            List<KylinTicketVo> performanceTicketVoList = mongoTemplate.find(performanceTicketVoQuery, KylinTicketVo.class, KylinTicketVo.class.getSimpleName());
                            // 转换票种信息结构为Map<ticketsId, ticketVo>
                            Map<String, KylinTicketVo> performanceTicketMap = performanceTicketVoList.stream().collect(Collectors.toMap(KylinTicketVo::getTicketsId, Function.identity(), (k1, k2) -> k2));

                            // 补充演出列表票种统计
                            voList.forEach(r -> {
                                // 演出的所有订单票明细
                                List<KylinOrderTicketEntitiesVo> performanceTicketEntitiesVoList = oteVoMap.get(r.getPerformancesId());
                                // 订单票明细按票种分组
                                Map<String, List<KylinOrderTicketEntitiesVo>> performanceTicketEntitiesVoMap = performanceTicketEntitiesVoList.stream().collect(Collectors.groupingBy(KylinOrderTicketEntitiesVo::getTicketId));

                                List<KylinStationTicketVo> ticketVoList = new ArrayList<>();

                                for (Map.Entry<String, List<KylinOrderTicketEntitiesVo>> entry : performanceTicketEntitiesVoMap.entrySet()) {
                                    KylinTicketVo ticketVo = performanceTicketMap.get(entry.getKey());

                                    KylinStationTicketVo stationTicketVo = KylinStationTicketVo.getNew();
                                    stationTicketVo.setTicketId(ticketVo.getTicketsId());
                                    stationTicketVo.setTitle(ticketVo.getTitle());
                                    stationTicketVo.setPrice(ticketVo.getPrice());
                                    stationTicketVo.setUseStart(ticketVo.getUseStart());
                                    stationTicketVo.setUseEnd(ticketVo.getUseEnd());

                                    List<KylinOrderTicketEntitiesVo> subPerformanceTicketEntitiesVoList = entry.getValue();
                                    // 订单票明细按出票状态分组
                                    Map<Integer, List<KylinOrderTicketEntitiesVo>> subStatusPerformanceTicketEntitiesVoMap
                                            = subPerformanceTicketEntitiesVoList.stream().collect(Collectors.groupingBy(KylinOrderTicketEntitiesVo::getStatus));

                                    stationTicketVo.setNumber(subPerformanceTicketEntitiesVoList.size());
                                    stationTicketVo.setPriceSum(stationTicketVo.getPrice().multiply(BigDecimal.valueOf(stationTicketVo.getNumber())));
                                    List<KylinOrderTicketEntitiesVo> checkedEntitiesVoList = subStatusPerformanceTicketEntitiesVoMap.get(1);
                                    stationTicketVo.setCheckedNum(CollectionUtils.isEmpty(checkedEntitiesVoList) ? 0 : checkedEntitiesVoList.size());
                                    List<KylinOrderTicketEntitiesVo> remainderEntitiesVoList = subStatusPerformanceTicketEntitiesVoMap.get(0);
                                    stationTicketVo.setRemainderNum(CollectionUtils.isEmpty(remainderEntitiesVoList) ? 0 : remainderEntitiesVoList.size());

                                    ticketVoList.add(stationTicketVo);
                                }

                                r.setTicketVoList(ticketVoList);
                                r.setCanDownTime(performanceRelationMap.get(r.getPerformancesId()));
                            });
                        }
                    }
                    break;
                case "down":
                    log.info(":::performances/down:{}", currentUid);
                    break;
                case "over":
                    log.info(":::performances/over:{}", currentUid);
                    break;
                default:
                    log.info(":::performances/default:{}", currentUid);
                    break;
            }
        } catch (Exception e) {
            log.error("验票:查取演出列表异常:/station/performances/?", e);
        }
        return ResponseDto.success(voList);
    }

    @ApiOperation(value = "下载验票数据")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "body", required = true, dataType = "String", name = "type", value = "类型[101-音乐节,102小型演出(livehouse演出),103巡演]", allowableValues = "101,102,103"),
            @ApiImplicitParam(type = "body", required = true, dataType = "String", name = "performanceId", value = "演出ID[64]"),
    })
    @GetMapping("download/{type}/{performanceId}")
    public ResponseDto<List<KylinStationCheckOrderVo>> downloadTicketData(@PathVariable String type, @PathVariable String performanceId) {
        log.info("type:{},performanceId:{}", type, performanceId);

        List<KylinStationCheckOrderVo> voList = new ArrayList<>();

        String currentUid = CurrentUtil.getCurrentUid();

        // 查取当前用户下关联演出ID列表
        KylinCheckUserPerformanceVo checkUserRelationVo = dataUtils.getCheckUserRelationVo(currentUid);
        List<CheckPerformanceRelationParam> relationParams;
        if (null == checkUserRelationVo || CollectionUtils.isEmpty(relationParams = checkUserRelationVo.getRelationParams())) {
            return ResponseDto.failure(ErrorMapping.get("20606"));
        }
        Optional<CheckPerformanceRelationParam> optional = relationParams.stream().findFirst().filter(r -> r.getPerformanceId().equals(performanceId));
        if (!optional.isPresent()) {
            return ResponseDto.failure(ErrorMapping.get("20607"));
        }
        CheckPerformanceRelationParam relationParam = optional.get();
        LocalDateTime canDownDt = DateUtil.Formatter.yyyyMMddHHmmss.parse(relationParam.getCanDownTime());
        if (canDownDt.isAfter(LocalDateTime.now())) {
            return ResponseDto.failure(ErrorMapping.get("20608"));
        }

        // 查取演出对应的订单票明细
        Query orderTicketEntitiesVoQuery = Query.query(Criteria.where("performanceId").is(performanceId).and("isPayment").is(1));
        voList = mongoTemplate.find(orderTicketEntitiesVoQuery, KylinStationCheckOrderVo.class, KylinOrderTicketEntitiesVo.class.getSimpleName());

        Query query = Query.query(Criteria.where("orderTicketsId").in(
                voList.stream().map(KylinStationCheckOrderVo::getOrderId).toArray()
        ));
        query.fields().include("orderTicketsId").include("qrCode");
        List<KylinOrderTicketVo> orderTicketVoList = mongoTemplate.find(query, KylinOrderTicketVo.class, KylinOrderTicketVo.class.getSimpleName());

        for (KylinOrderTicketVo t : orderTicketVoList)
            for (KylinStationCheckOrderVo r : voList)
                if (r.getOrderId().equals(t.getOrderTicketsId())) r.setQrCode(t.getQrCode());

        return ResponseDto.success(voList);
    }

    @ApiOperation(value = "上载验票数据")
    @PostMapping("upload")
    public ResponseDto<Boolean> uploadTicketData(@RequestBody KylinStationCheckOrderParam checkDataParam) {
        log.info("checkDataParams:{}", JsonUtils.toJson(checkDataParam));

        kylinOrderTicketEntitiesService.updateByStation(checkDataParam);

        return ResponseDto.success(true);
    }

    /* ------------------------------------------------------------------ */

    private ResponseDto<String> loginVerification(String mobile, String identity, boolean isPasswd) {
        KylinCheckUserVo checkUserVo = mongoTemplate.findOne(Query.query(Criteria.where("mobile").is(mobile)),
                KylinCheckUserVo.class, KylinCheckUserVo.class.getSimpleName());

        if (null == checkUserVo) {
            return ResponseDto.failure(ErrorMapping.get("20601"));
        }

        if (isPasswd) {// 校验密码
            if (checkUserVo.getPwd().equals(DigestUtils.md5DigestAsHex(identity.getBytes()))) {
                return this.loginAuthentication(checkUserVo);
            }
            return ResponseDto.failure(ErrorMapping.get("20602"));
        } else {// 校验验证码
            if (Arrays.asList("dev", "test").contains(environment.getProperty("spring.profiles.active"))) {
                if ("111111".equals(identity)) {
                    return this.loginAuthentication(checkUserVo);
                }
                return ResponseDto.failure(ErrorMapping.get("20002"));
            }

            try {
                LinkedMultiValueMap<String, String> paramsMap = new LinkedMultiValueMap<>();
                paramsMap.add("mobile", mobile);
                paramsMap.add("code", identity);
                LinkedMultiValueMap<String, String> headersMap = new LinkedMultiValueMap<>();
                headersMap.add("token", null);

                // TODO: 2021/5/12
                String respStr = HttpUtil.get("https://service.zhengzai.tv/smsValidation", paramsMap, headersMap);
                log.info("###PHP.API.RESP:{}", respStr);

                Map respMap = JsonUtils.fromJson(respStr, Map.class);

                if (StringUtils.equalsIgnoreCase("OK", (String) respMap.get("message"))) {
                    return this.loginAuthentication(checkUserVo);
                }
                return ResponseDto.failure(ErrorMapping.get("20002"));
            } catch (Exception e) {
                log.error("验证码验证异常[mobile:{},code:{}]", mobile, identity, e);
                return ResponseDto.failure(ErrorMapping.get("20002"));
            }
        }
    }

    private ResponseDto<String> loginAuthentication(KylinCheckUserVo checkUserVo) {
        String uid = checkUserVo.getCheckUserId();

        String ssoKeyUid = jwtValidator.getSsoRedisKey().concat(uid);

        Map<String, Object> claimsMap = new HashMap<>();
        claimsMap.put("uid", uid);
        claimsMap.put("mobile", checkUserVo.getMobile());
        claimsMap.put("nickname", checkUserVo.getName());

        String token = jwtValidator.create(claimsMap);

        String ssoKeyUidM5Token = ssoKeyUid.concat(DigestUtils.md5DigestAsHex(token.getBytes(StandardCharsets.UTF_8)));

        redisUtil.set(ssoKeyUidM5Token, true, jwtValidator.getExpireTtl() * 60);

        return ResponseDto.success(token);
    }

    private static <T> Predicate<T> distinctByKey(Function<? super T, Object> keyExtractor) {
        Map<Object, Boolean> seen = new ConcurrentHashMap<>();
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }
}
