package com.liquidnet.service.order.utils;

import com.liquidnet.service.goblin.dto.vo.GoblinOrderSkuVo;
import com.liquidnet.service.goblin.entity.GoblinOrderSku;
import com.liquidnet.service.goblin.param.GoblinOrderSqlParam;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Slf4j
public class PaySubjectUtils {

    /**
     * 清理 subject，只保留中文、英文、数字和空格，其他字符替换为 -
     * @param subject 原始商品名称
     * @return 清理后的 subject
     */
    public static String cleanSubject(String subject) {
        if (subject == null || subject.trim().isEmpty()) {
            return "";
        }

        try {
            subject = subject.trim();

            // 长度限制（支付宝限制256字符）
            if (subject.length() > 100) {
                subject = subject.substring(0, 100);
            }

            // 正则表达式：匹配非中文、非英文、非数字、非空格的字符
            // 中文范围：\u4e00-\u9fa5
            // 英文范围：a-zA-Z
            // 数字范围：0-9
            // 空格：保留
            Pattern pattern = Pattern.compile("[^\\u4e00-\\u9fa5a-zA-Z0-9]");

            // 替换所有不符合的字符为 -
            String cleaned = pattern.matcher(subject).replaceAll("-");

            // 处理多个连续的 - 替换为单个 -
            cleaned = cleaned.replaceAll("-+", "-");

            // 去除首尾的 -（保留空格）
            cleaned = cleaned.replaceAll("^-|-$", "");

            return "正在现场-" + cleaned;
        }catch (Exception e){
            log.error("error", e);
        }
        return "";
    }

    /**
     * 将订单 SKU 列表汇总并格式化为 "正在现场-spuName-num,spuName-num..." 的字符串。
     * 相同 spuName 的数量会自动累加。
     *
     * @param sqlParams 订单参数列表，可能为 null 或包含 null 元素
     * @return 格式化后的字符串，若无有效 SKU 则返回 "正在现场-"
     */
    public static String formatSpuNameNumByGoblinOrderSqlParamList(List<GoblinOrderSqlParam> sqlParams) {
        if (sqlParams == null || sqlParams.isEmpty()) {
            return "";
        }

        try {
            // 聚合 spuName -> totalNum（安全处理 null）
            Map<String, Integer> aggregated = sqlParams.stream()
                    .filter(Objects::nonNull)
                    .map(GoblinOrderSqlParam::getOrderSkuList)
                    .filter(Objects::nonNull)
                    .flatMap(List::stream)
                    .filter(Objects::nonNull)
                    .filter(sku -> sku.getSpuName() != null && !sku.getSpuName().trim().isEmpty())
                    .collect(Collectors.groupingBy(
                            GoblinOrderSku::getSpuName,
                            Collectors.summingInt(sku -> sku.getNum() == null ? 0 : sku.getNum())
                    ));

            // 按 spuName 排序（可选，使输出更稳定）
            return aggregated.entrySet().stream()
                    .sorted(Map.Entry.comparingByKey()) // 可选：按商品名排序
                    .map(entry -> entry.getKey() + "-" + entry.getValue())
                    .collect(Collectors.joining(","));
        }catch (Exception e) {
            log.error("error", e);
        }
        return "";
    }


    public static String formatSpuNameNumSummaryFromSkuVos(List<GoblinOrderSkuVo> skuVos) {
        if (skuVos == null || skuVos.isEmpty()) {
            return "";
        }

        Map<String, Integer> aggregated = skuVos.stream()
                .filter(Objects::nonNull)
                .filter(vo -> vo.getSpuName() != null && !vo.getSpuName().trim().isEmpty())
                .collect(Collectors.groupingBy(
                        GoblinOrderSkuVo::getSpuName,
                        Collectors.summingInt(vo -> vo.getNum() == null ? 0 : vo.getNum())
                ));

        String detailPart = aggregated.entrySet().stream()
                .sorted(Map.Entry.comparingByKey())
                .map(e -> e.getKey() + "-" + e.getValue())
                .collect(Collectors.joining(","));

        return detailPart;
    }

    public static void main(String[] args) {
        String text = "「发呆的下午」| 北京的第一次“发呆冲击” 鸟撞x虎啸春x大叫控";
        String text2 = "大波浪乐队 2021全新专辑《新逻辑 不止一面》全国巡演郑州";
        String text3 = "Matt吕彦良2025-Fresh-ME-新鲜之旅巡演-Refresh-with-Matt-Lv-长沙站Matt吕彦良2025-Fresh-ME-新鲜之旅巡演-Refr鲜之旅鲜之旅鲜之旅鲜之旅鲜之旅鲜之旅鲜之旅鲜之旅鲜之旅鲜之旅鲜之旅鲜之旅";
        System.out.println(cleanSubject(text));
        System.out.println(cleanSubject(text2));
        System.out.println(cleanSubject(text3));
    }

}


