package com.liquidnet.service.dragon.channel.strategy.impl;

import com.alibaba.fastjson.JSON;
import com.alipay.api.AlipayApiException;
import com.alipay.api.internal.util.AlipaySignature;
import com.liquidnet.common.exception.LiquidnetServiceException;
import com.liquidnet.commons.lang.util.DateUtil;
import com.liquidnet.commons.lang.util.StringUtil;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.dragon.channel.alipay.biz.AlipayBiz;
import com.liquidnet.service.dragon.channel.alipay.constant.AlipayConstant;
import com.liquidnet.service.dragon.channel.alipay.strategy.AlipayStrategyContext;
import com.liquidnet.service.dragon.channel.strategy.IPayChannelStrategy;
import com.liquidnet.service.dragon.channel.strategy.annotation.StrategyPayChannelHandler;
import com.liquidnet.service.dragon.constant.DragonConstant;
import com.liquidnet.service.dragon.constant.DragonErrorCodeEnum;
import com.liquidnet.service.dragon.dto.DragonOrdersDto;
import com.liquidnet.service.dragon.dto.DragonPayBaseReqDto;
import com.liquidnet.service.dragon.dto.DragonPayBaseRespDto;
import com.liquidnet.service.dragon.dto.DragonPayOrderQueryRespDto;
import com.liquidnet.service.dragon.utils.DataUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletRequest;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;

/**
 * @author AnJiabin <anjiabin@zhengzai.tv>
 * @version V1.0
 * @Description: TODO
 * @class: PayChannelStrategyAlipayImpl
 * @Package com.liquidnet.service.dragon.channel.strategy.impl
 * @Copyright: LightNet @ Copyright (c) 2021
 * @date 2021/7/13 13:06
 */
@Slf4j
@Component
@StrategyPayChannelHandler(DragonConstant.PayChannelEnum.ALIPAY)
public class PayChannelStrategyAlipayImpl implements IPayChannelStrategy {
    @Autowired
    private AlipayStrategyContext alipayStrategyContext;

    @Autowired
    private DataUtils dataUtils;

    @Autowired
    private AlipayBiz alipayBiz;

    @Value("${liquidnet.dragon.wepay.partnerKey}")
    private String partnerKey;

    @Override
    public ResponseDto<DragonPayBaseRespDto> dragonPay(DragonPayBaseReqDto dragonPayBaseReqDto) {
        return alipayStrategyContext.getStrategy(dragonPayBaseReqDto.getDeviceFrom()).dragonPay(dragonPayBaseReqDto);
    }

    @Override
    public void dragonNotify(HttpServletRequest request,String payType,String deviceFrom) {
        Map<String, String[]> requestParams = request.getParameterMap();
        Map<String, String> notifyMap = new HashMap<String, String>();
        notifyMap = alipayBiz.parseNotifyMsg(requestParams);
        log.info("dragonNotify-->alipay json : {}", JSON.toJSONString(notifyMap));

        log.info("接收到{}支付结果{}", payType, notifyMap);

        String returnStr = null;
        String code = notifyMap.get("out_trade_no");

        // 根据银行订单号获取支付信息
        DragonOrdersDto dragonOrdersDto = dataUtils.getPayOrderByCode(code);
        if (dragonOrdersDto == null) {
            throw new LiquidnetServiceException(DragonErrorCodeEnum.TRADE_ERROR_NOT_EXISTS.getCode(),DragonErrorCodeEnum.TRADE_ERROR_NOT_EXISTS.getMessage());
        }

        if (DragonConstant.PayStatusEnum.STATUS_PAID.getCode().equals(dragonOrdersDto.getStatus())) {
            throw new LiquidnetServiceException(DragonErrorCodeEnum.TRADE_ERROR_HAS_PAID.getCode(),DragonErrorCodeEnum.TRADE_ERROR_HAS_PAID.getMessage());
        }

        try {
            if (AlipaySignature.rsaCheckV1(notifyMap, partnerKey, "UTF-8", "RSA2")){
                String tradeStatus = notifyMap.get("trade_status");

                if (AlipayConstant.AlipayTradeStateEnum.TRADE_FINISHED.name().equals(tradeStatus)) {
                    // 判断该笔订单是否在商户网站中已经做过处理
                    // 如果没有做过处理，根据订单号（out_trade_no）在商户网站的订单系统中查到该笔订单的详细，并执行商户的业务程序
                    // 请务必判断请求时的total_fee、seller_id与通知时获取的total_fee、seller_id为一致的
                    // 如果有做过处理，不执行商户的业务程序

                    // 注意：
                    // 退款日期超过可退款期限后（如三个月可退款），支付宝系统发送该交易状态通知
                } else if (AlipayConstant.AlipayTradeStateEnum.TRADE_SUCCESS.name().equals(tradeStatus)) {

                    String gmtPaymentStr = notifyMap.get("gmt_payment");// 付款时间
                    LocalDateTime timeEnd = null;
                    if (!StringUtil.isEmpty(gmtPaymentStr)) {
                        timeEnd = DateUtil.Formatter.yyyyMMddHHmmss.parse(gmtPaymentStr);
                    }
                    completeSuccessOrder(dragonOrdersDto, notifyMap.get("trade_no"), timeEnd, notifyMap.toString());
                    returnStr = "success";
                } else {
//                    completeFailOrder(dragonOrdersDto, notifyMap.toString());
                    returnStr = "fail";
                }
            } else {// 验证失败
                throw new LiquidnetServiceException(DragonErrorCodeEnum.TRADE_ALIPAY_SIGN_ERROR.getCode(),DragonErrorCodeEnum.TRADE_ALIPAY_SIGN_ERROR.getMessage());
            }
        } catch (AlipayApiException e) {
            log.error("验签失败：" , e);
            throw new LiquidnetServiceException(DragonErrorCodeEnum.TRADE_ALIPAY_SIGN_ERROR.getCode(),DragonErrorCodeEnum.TRADE_ALIPAY_SIGN_ERROR.getMessage());
        }

        log.info("返回支付通道{}信息{}", payType, returnStr);
    }

    @Override
    public DragonPayOrderQueryRespDto checkOrderStatus(String code) {
        Map<String, Object> result = alipayBiz.tradeQuery(code);
        DragonPayOrderQueryRespDto respDto = new DragonPayOrderQueryRespDto();
        return respDto;
    }

    /**
     * 支付成功方法
     * @param dragonOrdersDto
     * @param bankTrxNo
     * @param timeEnd
     * @param bankReturnMsg
     */
    @Transactional(rollbackFor = Exception.class)
    void completeSuccessOrder(DragonOrdersDto dragonOrdersDto, String bankTrxNo, LocalDateTime timeEnd, String bankReturnMsg) {
        log.info("订单支付成功!");
        dragonOrdersDto.setPaymentAt(timeEnd);
        dragonOrdersDto.setPaymentId(bankTrxNo);// 设置银行流水号
//        DragonOrdersDto.(bankReturnMsg);
        dragonOrdersDto.setStatus(Integer.parseInt(DragonConstant.PayStatusEnum.STATUS_PAID.getCode()));
        dataUtils.updateOrderStatus(dragonOrdersDto.getCode(),dragonOrdersDto);

        String notifyUrl = dragonOrdersDto.getNotifyUrl();

    }

    /**
     * 支付失败方法
     *
     * @param DragonOrdersDto
     */
//    private void completeFailOrder(DragonOrdersDto dragonOrdersDto, String bankReturnMsg) {
//        log.info("订单支付失败!");
////        DragonOrdersDto.(bankReturnMsg);
//        dragonOrdersDto.setStatus(Integer.parseInt(DragonConstant.PayStatusEnum.STATUS_PAID.getCode()));
//        dataUtils.updateOrderStatus(dragonOrdersDto.getCode(),dragonOrdersDto);
//
//        String notifyUrl = dragonOrdersDto.getNotifyUrl();
//    }
}
