package com.liquidnet.service.merchant.service.impl;

import com.liquidnet.common.exception.LiquidnetServiceException;
import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.IDGenerator;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.SqlMapping;
import com.liquidnet.service.base.constant.MQConst;
import com.liquidnet.service.merchant.dto.param.MerchantSponsorApplyParam;
import com.liquidnet.service.merchant.dto.vo.MerchantSponsorAppliesVo;
import com.liquidnet.service.merchant.service.IMerchantSponsorsService;
import com.liquidnet.service.merchant.service.MerchantRdmService;
import com.liquidnet.service.merchant.util.QueueUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.IntStream;

@Slf4j
@Service
public class MerchantSponsorsServiceImpl implements IMerchantSponsorsService {
    @Autowired
    MerchantRdmService merchantRdmService;

    @Autowired
    QueueUtil queueUtils;

    @Override
    public String apply(String uid, MerchantSponsorApplyParam parameter) {
        LinkedList<String> toMqSqls = CollectionUtil.linkedListString();
        LinkedList<Object[]> sponsorApplyInsertObjs = CollectionUtil.linkedListObjectArr();

        // 申请主办 vos 上限
        List<MerchantSponsorAppliesVo> sponsorAppliesVos = merchantRdmService.getSponsorAppliesVosByUid(uid);
        if (!CollectionUtils.isEmpty(sponsorAppliesVos)) {
            if (sponsorAppliesVos.size() >= 10) {
                ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("13021");
                throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
            }
        }

        LocalDateTime now = LocalDateTime.now();

        // 申请主办 vo
        MerchantSponsorAppliesVo sponsorAppliesVo = MerchantSponsorAppliesVo.getNew();
        BeanUtils.copyProperties(parameter, sponsorAppliesVo);
        sponsorAppliesVo.setSponsorApplyId(IDGenerator.nextSnowId());
        sponsorAppliesVo.setApplyStatus(0);
        sponsorAppliesVo.setApplyType("create");
        sponsorAppliesVo.setReject("");
        sponsorAppliesVo.setUid(uid);
        sponsorAppliesVo.setCreatedAt(now);
        if (parameter.getSkipCompany() < 1) {
            sponsorAppliesVo.setCompanyName("");
            sponsorAppliesVo.setLicenseCode("");
            sponsorAppliesVo.setLicenseImg("");
            sponsorAppliesVo.setLegalName("");
            sponsorAppliesVo.setLegalIdentity("");
            sponsorAppliesVo.setLegalIdentityObverse("");
            sponsorAppliesVo.setLegalIdentityReverse("");
        } else {
            if (sponsorAppliesVo.getCompanyName().isEmpty()
                    || sponsorAppliesVo.getLicenseCode().isEmpty()
                    || sponsorAppliesVo.getLicenseImg().isEmpty()
                    || sponsorAppliesVo.getLegalName().isEmpty()
                    || sponsorAppliesVo.getLegalIdentity().isEmpty()
                    || sponsorAppliesVo.getLegalIdentityObverse().isEmpty()
                    || sponsorAppliesVo.getLegalIdentityReverse().isEmpty()) {
                ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("13022");
                throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
            }
        }

        // 申请主办 redis
        long s = System.currentTimeMillis();
        merchantRdmService.addSponsorAppliesVoByUid(uid, sponsorAppliesVos, sponsorAppliesVo);
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);

        // 申请主办 sql
        toMqSqls.add(SqlMapping.get("merchant_sponsor_applies.insert"));
        sponsorApplyInsertObjs.add(new Object[]{
                sponsorAppliesVo.getSponsorApplyId(), sponsorAppliesVo.getApplyStatus(), sponsorAppliesVo.getApplyType(), sponsorAppliesVo.getReject(), sponsorAppliesVo.getUid(),
                sponsorAppliesVo.getName(), sponsorAppliesVo.getLogo(), sponsorAppliesVo.getBackground(), sponsorAppliesVo.getDescription(),
                sponsorAppliesVo.getContactName(), sponsorAppliesVo.getContactIdentity(), sponsorAppliesVo.getContactIdentityObverse(), sponsorAppliesVo.getContactIdentityReverse(), sponsorAppliesVo.getContactEmail(),
                sponsorAppliesVo.getCompanyName(), sponsorAppliesVo.getLicenseCode(), sponsorAppliesVo.getLicenseImg(),
                sponsorAppliesVo.getLegalName(), sponsorAppliesVo.getLegalIdentity(), sponsorAppliesVo.getLegalIdentityObverse(), sponsorAppliesVo.getLegalIdentityReverse(), sponsorAppliesVo.getCreatedAt()
        });

        // mq
        s = System.currentTimeMillis();
        queueUtils.sendMsgByRedis(
                MQConst.MerchantQueue.SQL_MERCHANT_SPONSOR.getKey(),
                SqlMapping.gets(toMqSqls, sponsorApplyInsertObjs)
        );
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);

        return sponsorAppliesVo.getSponsorApplyId();
    }

    @Override
    public void applyDel(String uid, String sponsorApplyId) {
        // 查找场地申请，是否存在且已经驳回
        List<MerchantSponsorAppliesVo> sponsorAppliesVos = merchantRdmService.getSponsorAppliesVosByUid(uid);
        if (CollectionUtils.isEmpty(sponsorAppliesVos)) {
            ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("13023");
            throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
        }
        int idx = IntStream.range(0, sponsorAppliesVos.size())
                .filter(i -> sponsorAppliesVos.get(i).getSponsorApplyId().equals(sponsorApplyId))
                .findFirst()
                .orElse(-1);
        if (idx < 0) {
            ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("13023");
            throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
        }
        MerchantSponsorAppliesVo sponsorAppliesVo = sponsorAppliesVos.get(idx);
        if (sponsorAppliesVo.getApplyStatus() != 2) {
            ErrorMapping.ErrorMessage errorMessage = ErrorMapping.get("13024");
            throw new LiquidnetServiceException(errorMessage.getCode(), errorMessage.getMessage());
        }

        LocalDateTime now = LocalDateTime.now();
        sponsorAppliesVo.setUpdatedAt(now);
        sponsorAppliesVo.setDeletedAt(now);

        // 场地默认验票员 redis
        long s = System.currentTimeMillis();
        merchantRdmService.delSponsorAppliesVoByUid(uid, sponsorAppliesVos, sponsorAppliesVo);
        log.debug("#RDS耗时:{}ms", System.currentTimeMillis() - s);

        // 场地默认验票员 sql
        LinkedList<String> toMqSqls = CollectionUtil.linkedListString();
        LinkedList<Object[]> sponsorAppliesUpdateObjs = CollectionUtil.linkedListObjectArr();
        toMqSqls.add(SqlMapping.get("merchant_sponsor_applies.update"));
        sponsorAppliesUpdateObjs.add(new Object[]{
                sponsorAppliesVo.getUpdatedAt(), sponsorAppliesVo.getDeletedAt(), sponsorAppliesVo.getSponsorApplyId()
        });

        // mq
        s = System.currentTimeMillis();
        queueUtils.sendMsgByRedis(
                MQConst.MerchantQueue.SQL_MERCHANT_SPONSOR.getKey(),
                SqlMapping.gets(toMqSqls, sponsorAppliesUpdateObjs)
        );
        log.debug("#MQ耗时:{}ms", System.currentTimeMillis() - s);
    }
}
