package com.liquidnet.service.goblin.controller;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.common.exception.constant.ErrorCode;
import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.goblin.dto.vo.GoblinGoodsSkuInfoVo;
import com.liquidnet.service.goblin.dto.vo.GoblinStoreCouponVo;
import com.liquidnet.service.goblin.dto.vo.GoblinStoreInfoVo;
import com.liquidnet.service.goblin.dto.vo.GoblinUserCouponVo;
import com.liquidnet.service.goblin.service.GoblinCouponService;
import com.liquidnet.service.goblin.util.GoblinRedisUtils;
import com.liquidnet.service.goblin.util.ObjectUtil;
import io.swagger.annotations.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.validation.constraints.NotBlank;
import java.util.ArrayList;
import java.util.List;

@ApiSupport(order = 145001)
@Api(tags = "商城:POS机相关")
@Slf4j
@RestController
@RequestMapping("store/pos")
public class GoblinPosController {
    @Autowired
    GoblinRedisUtils goblinRedisUtils;
    @Autowired
    GoblinCouponService goblinCouponService;

    @PostMapping("coupon/receive")
    @ApiOperation("领取券")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "storeCouponId", value = "平台券id"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "uid", value = "UID"),
    })
    public ResponseDto<Boolean> checkOrderResult(@NotBlank(message = "参数无效:storeCouponId") @RequestParam("storeCouponId") String storeCouponId,
                                                 @NotBlank(message = "参数无效:uid") @RequestParam("uid") String uid) {
        String currentUid = CurrentUtil.getCurrentUid();
        GoblinStoreInfoVo storeInfoVo = goblinRedisUtils.getStoreInfoVoByUid(currentUid);
        GoblinStoreCouponVo storeCouponVo = goblinRedisUtils.getStoreCouponVo(storeCouponId);
        if (null == storeCouponVo) {
            log.warn("商城:POS机相关:领取券:优惠券不存在:[uid={},storeCouponId={}]", uid, storeCouponId);
            return ResponseDto.failure(ErrorMapping.get("140050"));
        } else if (null == storeInfoVo) {
            log.warn("商城:POS机相关:领取券:未创建店铺不可领取优惠券:[uid={},storeCouponId={}]", uid, storeCouponId);
            return ResponseDto.failure(ErrorCode.HTTP_PARAM_ERROR.getCode(), "店铺不存在");
        } else if (!storeCouponVo.getStoreId().equals(storeInfoVo.getStoreId())) {
            log.warn("商城:POS机相关:领取券:非本店铺优惠券不可领取:[uid={},storeId={},storeCouponId={}]", uid, storeInfoVo.getStoreId(), storeCouponId);
            return ResponseDto.failure(ErrorCode.HTTP_PARAM_ERROR.getCode(), "非本店铺优惠券不可领取");
        } else if (!storeCouponVo.getState().equals("1")) {
            return ResponseDto.failure(ErrorMapping.get("140051"));
        }
        List<GoblinUserCouponVo> userCouponVos = goblinRedisUtils.getUserCouponVos(uid);
        if (!CollectionUtils.isEmpty(userCouponVos)) {
            int beforeSize = userCouponVos.size();
            userCouponVos.removeIf(vo -> vo.getStoreCouponId().equals(storeCouponId));
            if ((beforeSize - userCouponVos.size()) >= storeCouponVo.getReceiveLimit()) {
                return ResponseDto.failure(ErrorMapping.get("140052"));
            }
        }
        if (storeCouponVo.getStock().equals(0) || goblinRedisUtils.getStoreCouponStock(storeCouponId) > 0) {
            Boolean resultFlg = goblinCouponService.receiveCoupon(uid, userCouponVos, storeCouponVo);
            return resultFlg ? ResponseDto.success() : ResponseDto.failure();
        }
        return ResponseDto.failure(ErrorMapping.get("140053"));
    }

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "SKU明细")
    @PostMapping("goods/skus")
    public ResponseDto<List<Object>> list(@NotBlank(message = "参数无效:skuIds") @RequestBody String skuids) {
        log.info("商城:POS机相关:SKU列表[skuIds={}]", skuids);
        String currentUid = CurrentUtil.getCurrentUid();
        GoblinStoreInfoVo storeInfoVo = goblinRedisUtils.getStoreInfoVoByUid(currentUid);
        if (null == storeInfoVo) {
            log.warn("商城:POS机相关:SKU明细:店铺不存在[uid={},skuIds={}]", currentUid, skuids);
            return ResponseDto.failure(ErrorCode.HTTP_PARAM_ERROR.getCode(), "店铺不存在");
        }
        String storeId = storeInfoVo.getStoreId();
        String[] skuIdArr = skuids.split(",");
        ArrayList<Object> skuVoList = CollectionUtil.arrayListObject();
        for (String skuId : skuIdArr) {
            GoblinGoodsSkuInfoVo skuInfoVo = goblinRedisUtils.getGoodsSkuInfoVo(skuId);
            if (storeId.equals(skuInfoVo.getStoreId())) {
                skuVoList.add(skuInfoVo);
            }
        }
        return ResponseDto.success(skuVoList);
    }
}
