package com.liquidnet.service.candy.service.impl;

import com.fasterxml.jackson.core.type.TypeReference;
import com.liquidnet.commons.lang.util.*;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.base.SqlMapping;
import com.liquidnet.service.base.constant.MQConst;
import com.liquidnet.service.candy.dto.CandyCouponCodeDto;
import com.liquidnet.service.candy.dto.CandyCouponInfoDto;
import com.liquidnet.service.candy.dto.CandyCouponRuleDto;
import com.liquidnet.service.candy.dto.CandyUserCouponBasicDto;
import com.liquidnet.service.candy.param.BackCouponParam;
import com.liquidnet.service.candy.service.ICandyCouponService;
import com.liquidnet.service.candy.util.CouponBaseUtil;
import com.liquidnet.service.candy.util.ObjectUtil;
import com.liquidnet.service.candy.util.QueueUtils;
import com.liquidnet.service.candy.util.RedisDataUtils;
import com.liquidnet.service.candy.vo.CandyCouponPreVo;
import com.liquidnet.service.candy.vo.CandyCouponVo;
import com.liquidnet.service.candy.vo.CandyMyCouponListVo;
import com.liquidnet.service.candy.vo.CandyUseResultVo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.MultiValueMap;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 券基础信息 服务实现类
 * </p>
 *
 * @author liquidnet
 * @since 2021-08-18
 */
@Service
public class CandyCouponServiceImpl implements ICandyCouponService {

    @Autowired
    private RedisDataUtils redisDataUtils;
    @Autowired
    private QueueUtils queueUtils;
    @Value("${liquidnet.service.adam.url}")
    private String adamUrl;

    @Override
    public CandyMyCouponListVo myCoupon(Integer type) {
        String uid = CurrentUtil.getCurrentUid();
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid, userCreateTime);
        CandyMyCouponListVo vo = CandyMyCouponListVo.getNew();
        List<CandyCouponVo> memberCoupon = ObjectUtil.getCandyCouponVoArrayList();
        List<CandyCouponVo> myCoupon = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            CandyCouponVo baseVo = CouponBaseUtil.getBaseCouponUserVo(dtoItem);
            if (baseVo == null) {
                continue;
            }
            if (type == 1) {
                if (baseVo.getState().equals(3) || dtoItem.getState().equals(5)) {
                    continue;
                }
            } else if (type == 2) {
                if (baseVo.getState().equals(1)) {
                    continue;
                }
            }
            if (dtoItem.getExclusive().equals(1)) { //会员券
                memberCoupon.add(baseVo);
            } else {//非会员券
                myCoupon.add(baseVo);
            }
        }
        //排序 可用->金额->类型->过期时间
        memberCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        myCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        vo.setMemberCoupon(memberCoupon);
        vo.setMyCoupon(myCoupon);
        return vo;
    }

    @Override
    public List<CandyCouponPreVo> memberCouponPre(String uid) {
        ArrayList<CandyCouponPreVo> voArrayList = ObjectUtil.getCandyCouponPreVoArrayList();
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid, userCreateTime);
        if (!CollectionUtil.isEmpty(dtoList)) {
            Map<String, Long> mCouponCountMap = dtoList.stream()
                    .filter(r -> (r.getExclusive() == 1 && r.getState() == 1))
                    .collect(Collectors.groupingBy(CandyUserCouponBasicDto::getMcouponId, Collectors.counting()));

            mCouponCountMap.forEach((mCouponId, ct) -> voArrayList.add(CandyCouponPreVo.getNew().setMcouponId(mCouponId).setAvailableCt(ct)));
        }
        return voArrayList;
    }

    @Override
    public List<CandyCouponVo> memberCoupon(String mCouponId) {
        String uid = CurrentUtil.getCurrentUid();
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid, userCreateTime);
        List<CandyCouponVo> memberCoupon = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            if (!mCouponId.equals(dtoItem.getMcouponId())) {
                continue;
            }
            CandyCouponVo baseVo = CouponBaseUtil.getBaseCouponUserVo(dtoItem);
            if (baseVo == null) {
                continue;
            }
            if (dtoItem.getExclusive().equals(1)) { //会员券
                memberCoupon.add(baseVo);
            }
        }
        //排序 可用->金额->类型->过期时间
        memberCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        return memberCoupon;
    }

    @Override
    public CandyMyCouponListVo preUsePerformanceCoupon(BigDecimal priceTotal, String performanceId, String timeId, String ticketId, Integer type) {
        String uid = CurrentUtil.getCurrentUid();
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid, userCreateTime);
        CandyMyCouponListVo vo = CandyMyCouponListVo.getNew();
        List<CandyCouponVo> memberCoupon = ObjectUtil.getCandyCouponVoArrayList();
        List<CandyCouponVo> myCoupon = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            CandyCouponVo baseVo = CouponBaseUtil.getPerformanceCouponUserVo(dtoItem, priceTotal, performanceId, timeId, ticketId);
            if (type == 1) {
                if (baseVo.getState().equals(3) || dtoItem.getState().equals(5) || baseVo.getState().equals(21)) {
                    continue;
                }
            } else if (type == 2) {
                if (baseVo.getState().equals(1)) {
                    continue;
                }
            }
            if (baseVo.getCouType().equals(101) || baseVo.getCouType().equals(3)) {
                continue;
            }
            if (dtoItem.getExclusive().equals(1)) { //会员券
                memberCoupon.add(baseVo);
            } else {//非会员券
                myCoupon.add(baseVo);
            }
        }

        //排序
        memberCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        myCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        vo.setMemberCoupon(memberCoupon);
        vo.setMyCoupon(myCoupon);
        return vo;
    }

    @Override
    public Integer preCanUsePerformanceCoupon(BigDecimal priceTotal, String performanceId, String timeId, String ticketId) {
        String uid = CurrentUtil.getCurrentUid();
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid, userCreateTime);
        int canUse = 0;
        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            CandyCouponVo baseVo = CouponBaseUtil.getPerformanceCouponUserVo(dtoItem, priceTotal, performanceId, timeId, ticketId);
            if (baseVo.getCouType().equals(101) || baseVo.getCouType().equals(3) || baseVo.getBusiType() == 2) {
                continue;
            }
            if (baseVo.getState().equals(1)) { //可用
                canUse = 1;
                break;
            } else {//不可用
                continue;
            }
        }
        return canUse;
    }

    @Override
    public CandyMyCouponListVo preUseGoodCoupon(BigDecimal priceTotal, String goodId, Integer type, String uid) {
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid, userCreateTime);
        CandyMyCouponListVo vo = CandyMyCouponListVo.getNew();
        List<CandyCouponVo> memberCoupon = ObjectUtil.getCandyCouponVoArrayList();
        List<CandyCouponVo> myCoupon = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            CandyCouponVo baseVo = CouponBaseUtil.getGoodCouponUserVo(dtoItem, priceTotal, goodId, dtoList.size());
            if (type == 1) {
                if (baseVo.getState().equals(3) || dtoItem.getState().equals(5) || baseVo.getState().equals(21)) {
                    continue;
                }
            } else if (type == 2) {
                if (baseVo.getState().equals(1)) {
                    continue;
                }
            }
            if (baseVo.getCouType().equals(101)) {
                continue;
            }
            if (dtoItem.getExclusive().equals(1)) { //会员券
                memberCoupon.add(baseVo);
            } else {//非会员券
                myCoupon.add(baseVo);
            }
        }

        //排序
        memberCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        myCoupon.sort(Comparator.comparing(CandyCouponVo::getState));
        vo.setMemberCoupon(memberCoupon);
        vo.setMyCoupon(myCoupon);
        return vo;
    }

    @Override
    public Integer preCanUseGoodCoupon(BigDecimal priceTotal, String goodId, String uid) {
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid, userCreateTime);
        int canUse = 0;
        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            CandyCouponVo baseVo = CouponBaseUtil.getGoodCouponUserVo(dtoItem, priceTotal, goodId, dtoList.size());
            if (baseVo.getCouType().equals(101) || baseVo.getBusiType() == 1 || baseVo.getBusiType() == 3) {
                continue;
            }
            if (baseVo.getState().equals(1)) { //可用
                canUse = 1;
                break;
            } else {//不可用
                continue;
            }
        }
        return canUse;
    }

    @Override
    public List<CandyCouponVo> myAdvanceCoupon(String performanceId) {
        String uid = CurrentUtil.getCurrentUid();
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid, userCreateTime);
        List<CandyCouponVo> advanceCoupon = ObjectUtil.getCandyCouponVoArrayList();
        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            if (dtoItem.getBusiType().equals(3)) {
                CandyCouponVo baseVo = CouponBaseUtil.getPerformanceCouponUserVo(dtoItem, BigDecimal.ZERO, performanceId, "-1", "-1");
                if (baseVo.getState().equals(1)) {
                    advanceCoupon.add(baseVo);
                } else {
                    continue;
                }
            } else {
                continue;
            }
        }
        return advanceCoupon;
    }

    @Override
    public Integer stateCoupon(String uCouponId) {
        String uid = CurrentUtil.getCurrentUid();
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        CandyUserCouponBasicDto dto = CouponBaseUtil.getSingleDtoByUCouponId(redisDataUtils.getCouponByUid(uid, userCreateTime), uCouponId);
        if (dto == null) {
            return null;//券不存在
        }
        return CouponBaseUtil.getCouponState(LocalDateTime.now(), dto.getDuedAt(), dto.getBindAt(), dto.getState());
    }

    @Override
    public CandyUseResultVo useCoupon(String uCouponId, String content, String totalPrice, String performanceId, String timesId, String ticketId, String goodId, String uid) {
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        CandyUseResultVo vo = CandyUseResultVo.getNew();
        List<String> targetIds = CollectionUtil.linkedListString();
        BigDecimal fullValue = BigDecimal.ZERO;
        List<CandyUserCouponBasicDto> dtoList = CouponBaseUtil.useCoupon(redisDataUtils.getCouponByUid(uid, userCreateTime), uCouponId, content);
        if (dtoList == null) {
            vo.setCouType(-1);
            vo.setValue(BigDecimal.ZERO);
        } else {
            CandyUserCouponBasicDto dto = CouponBaseUtil.getSingleDtoByUCouponId(redisDataUtils.getCouponByUid(uid, userCreateTime), uCouponId);
            if (dto.getBusiType().equals(3)) {
                dto.setCouType(101);
            } else {
                dto.setCouType(dto.getCouType());
            }
            vo.setCouType(dto.getCouType());
            switch (dto.getCouType()) {//券类型[1-代金券｜2-满减券｜3-兑换券｜4-折扣券|101-优先券]
                case 1:
                    vo.setValue(dto.getValFace());
                    break;
                case 2:
                    if (dto.getValOver().compareTo(new BigDecimal(totalPrice)) <= 0) {
                        vo.setValue(dto.getValMinus());
                        fullValue = dto.getValOver();
                    } else {
                        vo.setCouType(-1);
                        vo.setValue(BigDecimal.ZERO);
                    }
                    break;
                case 3:
                    vo.setValue(new BigDecimal(totalPrice));
                    break;
                case 4:
                    vo.setValue(dto.getDiscount());
                    break;
                default:
                    vo.setValue(BigDecimal.valueOf(0));
                    break;
            }

            Boolean isTarget = false;
            for (CandyCouponRuleDto ruleItem : dto.getUseRules()) {
                switch (ruleItem.getUseScope()) {
                    case 80://商品
                        isTarget = CouponBaseUtil.isTargetCoupon(ruleItem.getBusiId(), dto.getCouType(), goodId, new BigDecimal(totalPrice), dto.getValOver());
                        break;
                    case 90://演出
                        isTarget = CouponBaseUtil.isTargetCoupon(ruleItem.getBusiId(), dto.getCouType(), performanceId, new BigDecimal(totalPrice), dto.getValOver());
                        break;
                    case 91://场次
                        isTarget = CouponBaseUtil.isTargetCoupon(ruleItem.getBusiId(), dto.getCouType(), timesId, new BigDecimal(totalPrice), dto.getValOver());
                        break;
                    case 92://票
                        isTarget = CouponBaseUtil.isTargetCoupon(ruleItem.getBusiId(), dto.getCouType(), ticketId, new BigDecimal(totalPrice), dto.getValOver());
                        break;
                    case 100://票
                        isTarget = true;
                        break;
                    default:
                        isTarget = false;
                        break;
                }
                if (dto.getUseRules().size() > 1 && dto.getCouType().equals(4)) {
                    isTarget = false;
                }

                if (isTarget && !ruleItem.getBusiId().equals("")) {
                    targetIds.add(ruleItem.getBusiId());
                }
            }

            if (dto.getBusiType().equals(0)) {
                isTarget = true;
            }

            if (!isTarget) {
                vo.setCouType(-1);
                vo.setValue(BigDecimal.ZERO);
            }

            if (vo.getCouType() != -1) {
                redisDataUtils.setCouponByUid(uid, dtoList);
                LocalDateTime now = LocalDateTime.now();
                //入库
                if (dto.getCcouponId() == null || (null != dto.getCcouponId() && dto.getUsedAt() != null)) {// 是公有券且为使用后退回的券，则更新
                    queueUtils.sendMsgByRedis(
                            MQConst.CandyQueue.COUPON_USE.getKey(),
                            SqlMapping.get("candy_coupon.use",
                                    now, content, uid, uCouponId
                            )
                    );
                } else {
                    queueUtils.sendMsgByRedis(
                            MQConst.CandyQueue.COUPON_USE.getKey(),
                            SqlMapping.get("candy_coupon.use_insert",
                                    uCouponId, uid, dto.getCouponId(), dto.getBindAt(), dto.getDuedAt(), now, content, now
                            )
                    );
                }
            }
        }
        vo.setFullValue(fullValue);
        vo.setTargetIds(targetIds);
        return vo;
    }

    @Override
    public Boolean useBackCoupon(List<BackCouponParam> backCouponParams) {
        for (BackCouponParam item : backCouponParams) {
            String uid = item.getUid();
            LocalDateTime userCreateTime;
            try {
                MultiValueMap<String, String> params = CollectionUtil.linkedMultiValueMapStringString();
                params.add("uid", uid);
                MultiValueMap<String, String> headers = CollectionUtil.linkedMultiValueMapStringString();
                headers.add("Accept", "application/json;charset=UTF-8");
                String returnData = HttpUtil.post(adamUrl.concat("/adam/rsc/inquire/createdAt"), params, headers);
                userCreateTime = JsonUtils.fromJson(returnData, new TypeReference<ResponseDto<LocalDateTime>>() {
                }).getData();
            } catch (Exception e) {
                userCreateTime = null;
            }
            ArrayList<String> uCouponIdList = new ArrayList(Arrays.asList(item.getuCouponIds().split(",")));
            List<CandyUserCouponBasicDto> dtoList = CouponBaseUtil.backCoupon(redisDataUtils.getCouponByUid(uid, userCreateTime), uCouponIdList);
            if (dtoList == null) {
                return false;
            } else {
                redisDataUtils.setCouponByUid(uid, dtoList);
                //入库
                LinkedList<String> sqls = CollectionUtil.linkedListString();
                sqls.add(SqlMapping.get("candy_coupon.back"));
                LinkedList<Object[]> sqlsDataA = CollectionUtil.linkedListObjectArr();
                for (String uCouponId : uCouponIdList) {
                    sqlsDataA.add(new Object[]{
                            uid, uCouponId
                    });
                }
                String sqlData = SqlMapping.gets(sqls, sqlsDataA);
                queueUtils.sendMsgByRedis(
                        MQConst.CandyQueue.COUPON_BACK.getKey(),
                        sqlData
                );
            }
        }
        return true;
    }

    @Override
    public ResponseDto<String> receiveCoupon(String ccode) {
        String uid = CurrentUtil.getCurrentUid();
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        String mobile = (String) CurrentUtil.getTokenClaims().get(CurrentUtil.TOKEN_MOBILE);

        CandyCouponCodeDto dto = redisDataUtils.getCouponByCode(ccode);
        LocalDateTime now = LocalDateTime.now();

        if (dto == null) {
            return ResponseDto.failure("兑换码有误，请重新输入");
        } else if (dto.getState().equals(0)) {//判断可领
            CandyCouponInfoDto infoDto = redisDataUtils.getCouponInfo(dto.getCouponId());
            if (!CouponBaseUtil.CandyCanReceive(infoDto)) {
                return ResponseDto.failure("兑换码不可用");
            }
            String uCouponId = IDGenerator.get32UUID();
            //构建baseDto
            CandyUserCouponBasicDto baseDto = CandyUserCouponBasicDto.getNew().copyToRedisBaseCoupon(uCouponId, uid, ccode, infoDto, infoDto.getExpireAt());
            //删除code redis
            redisDataUtils.delCouponByCode(ccode);
            //添加baseDto -> baseDtoList
            redisDataUtils.addCouponByUid(uid, baseDto, userCreateTime);
            //入库
            LinkedList<String> toMqSqls = CollectionUtil.linkedListString();
            LinkedList<Object[]> objsUserCoupon = CollectionUtil.linkedListObjectArr(),
                    objsCouponCode = CollectionUtil.linkedListObjectArr();
            toMqSqls.add(SqlMapping.get("candy_coupon.receive"));
            objsUserCoupon.add(new Object[]{uCouponId, uid, 1, infoDto.getCouponId(), ccode, infoDto.getEffectAt(), now, infoDto.getExpireAt()});
            toMqSqls.add(SqlMapping.get("candy_coupon_code.receive"));// state:状态[0-可用|1-已用]
            objsCouponCode.add(new Object[]{1, uCouponId, uid, mobile, now, now, dto.getCouponId(), ccode});
            queueUtils.sendMsgByRedis(
                    MQConst.CandyQueue.COUPON_RECEIVE.getKey(),
                    SqlMapping.gets(toMqSqls, objsUserCoupon, objsCouponCode)
            );
            return ResponseDto.success(uCouponId);
        } else if (dto.getState().equals(1)) {
            return ResponseDto.failure("兑换码已使用");
        } else if (dto.getState() == 3) {
            return ResponseDto.failure("兑换码已过期");
        } else if (dto.getState() == 4) {
            return ResponseDto.failure("兑换码已失效");
        } else {
            return ResponseDto.failure("兑换失败，请稍后再试");
        }
    }

    @Override
    public ResponseDto<CandyCouponVo> receiveCouponDetails(String ccode) {
        String uid = CurrentUtil.getCurrentUid();
        CandyCouponCodeDto dto = redisDataUtils.getCouponByCode(ccode);
        if (dto == null) {
            return ResponseDto.failure("兑换码有误，请重新输入");
        } else if (dto.getState().equals(0)) {//判断可领
            CandyCouponInfoDto infoDto = redisDataUtils.getCouponInfo(dto.getCouponId());
            if (!CouponBaseUtil.CandyCanReceive(infoDto)) {
                return ResponseDto.failure("兑换码不可用");
            }
            String uCouponId = IDGenerator.get32UUID();
            //构建baseDto
            CandyUserCouponBasicDto baseDto = CandyUserCouponBasicDto.getNew().copyToRedisBaseCoupon(uCouponId, uid, ccode, infoDto, LocalDateTime.now().plusDays(infoDto.getRedeemValidity()));
            return ResponseDto.success(CouponBaseUtil.getBaseCouponUserVo(baseDto));
        } else {
            return ResponseDto.failure("兑换失败，请稍后再试");
        }
    }

    @Override
    public ResponseDto<List<CandyCouponVo>> couponListById(String uCouponIds, String uid) {
        LocalDateTime userCreateTime;
        try {
            userCreateTime = getCreatedAt(uid);
        } catch (Exception e) {
            userCreateTime = LocalDateTime.of(2222,1,1,0,0,0);
        }
        ArrayList<String> uCouponIdList = new ArrayList(Arrays.asList(uCouponIds.split(",")));
        List<CandyUserCouponBasicDto> dtoList = redisDataUtils.getCouponByUid(uid, userCreateTime);
        List<CandyCouponVo> vo = ObjectUtil.getCandyCouponVoArrayList();

        for (CandyUserCouponBasicDto dtoItem : dtoList) {
            if (uCouponIdList.contains(dtoItem.getUcouponId())) {
                CandyCouponVo baseVo = CouponBaseUtil.getBaseCouponUserVo(dtoItem);
                if (baseVo == null) {
                    continue;
                }
                vo.add(baseVo);
            } else {
                continue;
            }
        }
        return ResponseDto.success(vo);
    }


    private LocalDateTime getCreatedAt(String uid){
        MultiValueMap<String, String> params = CollectionUtil.linkedMultiValueMapStringString();
        params.add("uid", uid);
        MultiValueMap<String, String> headers = CollectionUtil.linkedMultiValueMapStringString();
        headers.add("Accept", "application/json;charset=UTF-8");
        String returnData = HttpUtil.post(adamUrl.concat("/adam/rsc/inquire/createdAt"), params, headers);
        return JsonUtils.fromJson(returnData, new TypeReference<ResponseDto<LocalDateTime>>() {}).getData();
    }
}
