package com.liquidnet.service.goblin.controller;

import com.liquidnet.common.exception.LiquidnetServiceException;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.StringUtil;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.goblin.dto.vo.*;
import com.liquidnet.service.goblin.entity.GoblinFrontHotWord;
import com.liquidnet.service.goblin.service.impl.GoblinFrontServiceImpl;
import com.liquidnet.service.goblin.util.GoblinRedisUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

/**
 * @author zhangfuxin
 * @Description:前端接口
 * @date 2021/12/27 下午6:25
 */
@Slf4j
@Api(tags = "前端接口管理")
@RestController
@RequestMapping("/front")
public class GoblinFrontController {
    @Autowired
    private GoblinRedisUtils goblinRedisUtils;
    @Autowired
    private GoblinFrontServiceImpl goblinFrontService;

    @GetMapping("getTopBanner")
    @ApiOperation("获取顶部banner")
    public ResponseDto<ArrayList<GoblinFrontBannerVo>> getBanner() {
        return ResponseDto.success(goblinFrontService.getListBanner());
    }

    @GetMapping("getMiddleBanner")
    @ApiOperation("获取中部banner")
    public ResponseDto<ArrayList<GoblinFrontBannerVo>> getMiddleBanner() {
        return ResponseDto.success(goblinFrontService.getMiddleBanner());
    }

    @GetMapping("getHotWord")
    @ApiOperation("获得热词")
    public ResponseDto<GoblinFrontHotWord> getHotWord() {
        return ResponseDto.success(goblinFrontService.getHotWord());
    }

    @GetMapping("getNavigation")
    @ApiOperation("获得金刚栏")
    public ResponseDto<GoblinFrontNavigationVoo> getNavigation() {
        return ResponseDto.success(goblinFrontService.getNavigation());
    }

    @GetMapping("getSeckilll")
    @ApiOperation("获得秒杀首页要用的")
    public ResponseDto<GoblinFrontSeckillVo> getSeckilll() throws ParseException {
        return ResponseDto.success(goblinFrontService.getSeckilll());
    }

    @GetMapping("getSelectGoods")
    @ApiOperation("获得精选商品列表(pageNumber从0开始)")
    public ResponseDto<GoblinFrontSelectGoodVo> getSelectGoods(@RequestParam(name = "pageSize", required = true) int pageSize, @RequestParam(name = "pageNumber", required = true) int pageNumber) throws ParseException {
        return ResponseDto.success(goblinFrontService.getSelectGoods(pageNumber, pageSize));
    }

    @GetMapping("getGoodsDetail")
    @ApiOperation("获得商品详情")
    public ResponseDto<GoblinFrontGoodDetailVo> getGoodsDetail(@RequestParam(name = "spuId", required = true) String spuId) throws ParseException {
        GoblinFrontGoodDetailVo vo = goblinFrontService.getGoodsDetail(spuId);
        if(vo==null){
            return ResponseDto.failure("商品不存在");
        }
        return ResponseDto.success(vo);
    }

    @GetMapping("getGoodsDetail/scanner")
    @ApiOperation(value = "获得商品详情/条形码识别", notes = "条码识别SKU时使用")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "form", required = true, dataType = "String", name = "barCode", value = "SKU条码"),
            @ApiImplicitParam(paramType = "form", required = true, dataType = "String", name = "marketId", value = "活动id"),
    })
    public ResponseDto<GoblinFrontGoodDetailVo> getGoodsDetailByScanner(@RequestParam String barCode, @RequestParam("marketId") String marketId) {
        String currentUid = CurrentUtil.getCurrentUid();
        GoblinStoreInfoVo storeInfoVo = goblinRedisUtils.getStoreInfoVoByUid(currentUid);
        if (storeInfoVo == null) {
            return ResponseDto.failure("店铺不存在");
        }
        GoblinFrontGoodDetailVo vo;
        try {
            vo = goblinFrontService.getGoodsDetailByScanner(barCode, storeInfoVo.getStoreId(), marketId);
            if (vo == null) {
                return ResponseDto.failure("商品不存在");
            }
        } catch (LiquidnetServiceException e) {
            return ResponseDto.failure(e.getMessage());
        } catch (Exception e) {
            log.error("Ex.扫条码识别商品:发生异常[barCode={},marketId={}]", barCode, marketId, e);
            return ResponseDto.failure("识别失败");
        }
        return ResponseDto.success(vo);
    }

    @GetMapping("getMoudleIndex")
    @ApiOperation("获取组件排序")
    public ResponseDto getMoudleIndex() throws ParseException {
        return ResponseDto.success(goblinFrontService.getMoudleIndex());
    }

    @GetMapping("getCube")
    @ApiOperation("获取魔方")
    public ResponseDto<GoblinFrontCubeVo> getCube() throws ParseException {
        return ResponseDto.success(goblinFrontService.getCube());
    }

    @GetMapping("getRecommend")
    @ApiOperation("获取新品推荐")
    public ResponseDto<GoblinFrontCubeVo> getRecommend() throws ParseException {
        return ResponseDto.success(goblinFrontService.getRecommend());
    }

    @GetMapping("getCategoryList")
    @ApiOperation("获取分类列表商品（ 1、销量优先、2、新品优先、3、价格降序、4、价格升序）")
    public ResponseDto<GoblinFrontCategoryListVo> getCategoryList(@RequestParam(name = "type", required = true) String type, @RequestParam(name = "categoryId", required = true) String categoryId, @RequestParam(name = "page", required = true) int page, @RequestParam(name = "pageSize", required = true) int pageSize) throws ParseException {
        return ResponseDto.success(goblinFrontService.getCategoryList(type, categoryId, page, pageSize));
    }

    @GetMapping("getMusicList")
    @ApiOperation("获取音乐标签列表（ 1、销量优先、2、新品优先、3、价格降序、4、价格升序）")
    public ResponseDto<GoblinFrontCategoryListVo> getMusicList(@RequestParam(name = "type", required = true) String type, @RequestParam(name = "musicId", required = true) String musicId, @RequestParam(name = "page", required = true) int page, @RequestParam(name = "pageSize", required = true) int pageSize) throws ParseException {
        return ResponseDto.success(goblinFrontService.getMusic(musicId, type, page, pageSize));
    }

    @GetMapping("getCategory")
    @ApiOperation("获取分类子集")
    public ResponseDto<GoblinCategoryzfVo> getCategory(@RequestParam(name = "categoryId", required = true) String categoryId) throws ParseException {
        return ResponseDto.success(goblinFrontService.getCategory(categoryId));
    }

    @GetMapping("getStore")
    @ApiOperation("获得店铺详情")
    public ResponseDto<GoblinStoreInfoVo> getStore(@RequestParam(name = "storeId", required = true) String storeId) throws ParseException {
        return ResponseDto.success(goblinFrontService.getStore(storeId));
    }

    @GetMapping("getStoreCategory")
    @ApiOperation("获得店铺分类")
    public ResponseDto<List<GoblinStoreGoodsCategoryVo>> getStoreCategory(@RequestParam(name = "storeId", required = true) String storeId) throws ParseException {
        return ResponseDto.success(goblinFrontService.getStoreCategory(storeId));
    }

    @GetMapping("getStoreGoodes")
    @ApiOperation("获得店铺商品")
    public ResponseDto<GoblinFrontCategoryListVo> getStoreGoodes(@RequestParam(name = "storeId", required = true) String storeId, @RequestParam(name = "categoryId", required = false) String categoryId, @RequestParam(name = "name", required = false) String name) throws ParseException {
        return ResponseDto.success(goblinFrontService.getStoreGoodes(storeId, categoryId, name));
    }

    @GetMapping("searchGoodesName")
    @ApiOperation("搜索商品名字、或商铺名字")
    public ResponseDto<GoblinGoodsInfoListVoo> searchGoodesName(@RequestParam(name = "name", required = true) String name,@RequestParam(name = "page", required = true) int page,@RequestParam(name = "pageSize", required = true) int pageSize) throws ParseException {
        if (StringUtil.isNotBlank(name)) {
            return ResponseDto.success(goblinFrontService.searchGoodesName(name,page,pageSize));
        }
        return ResponseDto.success(null);
    }

    @GetMapping("tagPerformance")
    @ApiOperation("标签，演出id获取演出列表")
    public ResponseDto<List<GoblinGoodsInfoVo>> getGoodByMusicTagP(@RequestParam(name = "musicTag", required = true) String musicTag,
                                                                      @RequestParam(name = "performanceId", required = true) String performanceId) {
        return ResponseDto.success(goblinFrontService.getGoodByMusicTagP(musicTag, performanceId));
    }


}
