package com.liquidnet.service.kylin.utils;

import com.liquidnet.common.cache.redis.util.RedisUtil;
import com.liquidnet.commons.lang.util.DateUtil;
import com.liquidnet.service.kylin.constant.KylinRedisConst;
import com.liquidnet.service.kylin.dto.vo.express.KylinOrderExpressVo;
import com.liquidnet.service.kylin.dto.vo.middle.KylinTicketTimesVo;
import com.liquidnet.service.kylin.dto.vo.middle.KylinTicketVo;
import com.liquidnet.service.kylin.dto.vo.mongo.*;
import com.liquidnet.service.kylin.dto.vo.returns.KylinOrderListVo;
import com.liquidnet.service.kylin.dto.vo.returns.KylinOrderRefundsVo;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.*;
import java.util.regex.Pattern;

@Component
public class DataUtils {

    @Autowired
    private MongoTemplate mongoTemplate;
    @Autowired
    private RedisUtil redisUtil;

    /**
     * 获取普通剩余库存
     *
     * @param ticketId 票id
     * @return 普通剩余库存
     */
    public int getSurplusGeneral(String ticketId) {
        return (int) redisUtil.get(KylinRedisConst.PERFORMANCES_INVENTORY + ticketId + ":" + KylinRedisConst.SURPLUS_GENERAL);
    }

    /**
     * 判断 当前库存是否售罄
     *
     * @param ticketId 票id
     * @return boolean
     */
    public boolean ticketIsSoldOut(String ticketId) {
        if (0 >= getSurplusGeneral(ticketId)) {
            return true;
        } else {
            return false;
        }
    }

    public void setOrderList(String userId, List<KylinOrderListVo> vo) {
        redisUtil.set(KylinRedisConst.ORDER_LIST + userId, vo);
    }

    public List<KylinOrderListVo> getOrderList(String userId) {
        if (redisUtil.hasKey(KylinRedisConst.ORDER_LIST + userId)) {
        } else {
            List<KylinOrderListVo> voList = mongoTemplate.find(Query.query(Criteria.where("userId").is(userId))
                            .with(Sort.by(Sort.Direction.DESC, "createdAt")).limit(40),
                    KylinOrderListVo.class, KylinOrderTicketVo.class.getSimpleName());
            redisUtil.set(KylinRedisConst.ORDER_LIST + userId, voList);
        }
        return (List<KylinOrderListVo>) redisUtil.get(KylinRedisConst.ORDER_LIST + userId);
    }

    /**
     * 根据订单id 获取 订单vo 详情
     *
     * @param orderId
     * @return
     */
    public KylinOrderTicketVo getOrderTicketVo(String orderId) {
        if (redisUtil.hasKey(KylinRedisConst.ORDER + orderId)) {
        } else {
            KylinOrderTicketVo ticketData = mongoTemplate.findOne(Query.query(Criteria.where("orderTicketsId").is(orderId)), KylinOrderTicketVo.class, KylinOrderTicketVo.class.getSimpleName());
            List<KylinOrderTicketEntitiesVo> kylinOrderTicketEntitiesVoList = mongoTemplate.find(Query.query(Criteria.where("orderId").is(orderId)), KylinOrderTicketEntitiesVo.class, KylinOrderTicketEntitiesVo.class.getSimpleName());
            ticketData.setEntitiesVoList(kylinOrderTicketEntitiesVoList);
            redisUtil.set(KylinRedisConst.ORDER + orderId, ticketData);
        }
        return (KylinOrderTicketVo) redisUtil.get(KylinRedisConst.ORDER + orderId);
    }

    /**
     * 删除订单redis
     *
     * @param orderId
     */
    public void delOrderTicketRedis(String orderId) {
        redisUtil.del(KylinRedisConst.ORDER + orderId);
    }


    /**
     * 获取 验票账号关系 vo
     *
     * @param checkUserId
     * @return
     */
    public KylinCheckUserPerformanceVo getCheckUserRelationVo(String checkUserId) {
        if (redisUtil.hasKey(KylinRedisConst.CHECK_USER_RELATION + checkUserId)) {
        } else {
            KylinCheckUserPerformanceVo ticketData = mongoTemplate.findOne(Query.query(Criteria.where("checkUserId").is(checkUserId)), KylinCheckUserPerformanceVo.class, KylinCheckUserPerformanceVo.class.getSimpleName());
            redisUtil.set(KylinRedisConst.CHECK_USER_RELATION + checkUserId, ticketData);
        }
        return (KylinCheckUserPerformanceVo) redisUtil.get(KylinRedisConst.CHECK_USER_RELATION + checkUserId);
    }

    /**
     * 获取入场须知
     *
     * @param enterInfoId
     * @return
     */
    public String getEnterInfo(String enterInfoId) {
        if (redisUtil.hasKey(KylinRedisConst.ENTER_INFO + enterInfoId)) {
        } else {
            return "入场须知";
        }
        return (String) redisUtil.get(KylinRedisConst.ENTER_INFO + enterInfoId);
    }


    /**
     * 删除订单redis
     *
     * @param orderEntitiesId
     */
    public void delOrderTicketEntitiesRedis(String orderEntitiesId) {
        redisUtil.del(KylinRedisConst.ORDER_ENTITIES + orderEntitiesId);
    }

    /**
     * 根据演出id 获取 演出vo 详情
     *
     * @param performanceId
     * @return
     */
    public KylinPerformanceVo getPerformanceVo(String performanceId) {
        if (redisUtil.hasKey(KylinRedisConst.PERFORMANCES + performanceId)) {
        } else {
            KylinPerformanceVo performanceData = mongoTemplate.findOne(Query.query(Criteria.where("performancesId").is(performanceId)), KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());
            redisUtil.set(KylinRedisConst.PERFORMANCES + performanceId, performanceData);
        }
        return (KylinPerformanceVo) redisUtil.get(KylinRedisConst.PERFORMANCES + performanceId);
    }

    /**
     * 获取分销用户名字
     *
     * @param agentId
     * @return
     */
    public String getAgentInfoName(String agentId) {
        String redisKey = KylinRedisConst.PERFORMANCES_AGENT_INFO.concat(agentId);
//        String name = (String) redisUtil.hget(redisKey, "name");
        String name = "";
        return name;
    }

    /**
     * 获取演出列表redis
     *
     * @param cityName
     */
    public List<KylinPerformanceVo> getPerformancesListOfcityName(String cityName) {
        String redisKey = KylinRedisConst.PERFORMANCES_LIST_CITYNAME.concat(cityName);
        if (!redisUtil.hasKey(redisKey)) {
            // 固定查询条件
            Query query = getCommonWhere();
            // 其他条件
            Pattern cityNameCompile = Pattern.compile("^.*" + cityName + ".*$", Pattern.CASE_INSENSITIVE);
            query.addCriteria(Criteria.where("cityName").regex(cityNameCompile));
            // 排序
            Sort sortName = Sort.by(Sort.Direction.ASC, "timeStart");
            query.with(sortName);
            // 不要查询的字段
            query.fields().exclude("details");
            query.fields().exclude("noticeImage");
            query.fields().exclude("ticketTimeList");
            query.fields().exclude("describeElectronic");
            List<KylinPerformanceVo> performancesList = mongoTemplate.find(query, KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());
            redisUtil.set(redisKey, performancesList);
        }
        return (List<KylinPerformanceVo>) redisUtil.get(redisKey);
    }

    /**
     * 获取系统推荐演出列表redis
     */
    public List<KylinPerformanceVo> getPerformancesListIsSystemRecommend() {
        String redisKey = KylinRedisConst.PERFORMANCES_LIST_SYSTEM_RECOMMEND;
        if (!redisUtil.hasKey(redisKey)) {
            // 固定条件
            Query query = getCommonWhere();
            // 排序
            Sort sortName = Sort.by(Sort.Direction.ASC, "timeStart");
            Pageable pageable = PageRequest.of(0, 8, sortName);
            query.with(pageable);
            // 不要查询的字段
            query.fields().exclude("details");
            query.fields().exclude("noticeImage");
            query.fields().exclude("ticketTimeList");
            query.fields().exclude("describeElectronic");
            // 推荐
            List<KylinPerformanceVo> systemRecommendList = mongoTemplate.find(query, KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());
            redisUtil.set(redisKey, systemRecommendList);
        }
        return (List<KylinPerformanceVo>) redisUtil.get(redisKey);
    }

    /**
     * 获取演出预告列表redis
     */
    public HashMap<String, Object> getPerformancesListNotice() {
        String redisKey = KylinRedisConst.PERFORMANCES_LIST_NOTICE;
        if (!redisUtil.hasKey(redisKey)) {
            // 固定条件
            Query query = getCommonWhere();

            // 今天的
            HashMap toDayTime = DateUtil.oneDayStartEnd();
            String nowTimeTStr = (String) toDayTime.get("startStr");
            String toDayEndTimeStr = (String) toDayTime.get("endStr");
            query.addCriteria(Criteria.where("sellTime").gte(nowTimeTStr).lt(toDayEndTimeStr));
            query.fields().exclude("details");
            query.fields().exclude("noticeImage");
            query.fields().exclude("ticketTimeList");
            query.fields().exclude("describeElectronic");
            List<KylinPerformanceVo> toDayList = mongoTemplate.find(query, KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());

            // 固定条件
            Query queryT = getCommonWhere();

            // 三天的
            Calendar cal = Calendar.getInstance();
            cal.setTime((Date) toDayTime.get("end"));
            Date beforeDayEnd = DateUtil.getBeforeDayEnd(cal, 3);
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            String threeDaysLaterStr = sdf.format(beforeDayEnd);

            queryT.addCriteria(Criteria.where("sellTime").gte(toDayEndTimeStr).lt(threeDaysLaterStr));
            queryT.fields().exclude("details");
            queryT.fields().exclude("noticeImage");
            queryT.fields().exclude("ticketTimeList");
            queryT.fields().exclude("describeElectronic");
            List<KylinPerformanceVo> threeDaysList = mongoTemplate.find(queryT, KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());

            HashMap<String, Object> info = new HashMap<>();
            info.put("toDayList", toDayList);
            info.put("threeDaysList", threeDaysList);
            redisUtil.set(redisKey, info);
        }
        return (HashMap<String, Object>) redisUtil.get(redisKey);
    }

    /**
     * 获取推荐演出列表redis
     */
    public List<KylinPerformanceVo> getPerformancesListIsRecommend() {
        String redisKey = KylinRedisConst.PERFORMANCES_LIST_RECOMMEND;
        if (!redisUtil.hasKey(redisKey)) {
            // 固定条件
            Query query = getCommonWhere();
            // 排序
            Sort sortName = Sort.by(Sort.Direction.DESC, "isRecommend");
            query.with(sortName);
            // 不要查询的字段
            query.fields().exclude("details");
            query.fields().exclude("noticeImage");
            query.fields().exclude("ticketTimeList");
            query.fields().exclude("describeElectronic");
            // 推荐
            query.addCriteria(Criteria.where("isRecommend").gte(1));
            List<KylinPerformanceVo> recommendList = mongoTemplate.find(query, KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());
            redisUtil.set(redisKey, recommendList);
        }
        return (List<KylinPerformanceVo>) redisUtil.get(redisKey);
    }

    /**
     * 获取会员专属演出列表redis
     */
    public List<KylinPerformanceVo> getPerformancesListIsExclusive() {
        String redisKey = KylinRedisConst.PERFORMANCES_LIST_EXCLUSIVE;
        if (!redisUtil.hasKey(redisKey)) {
            // 固定条件
            Query query = getCommonWhere();
            // 不要查询的字段
            query.fields().exclude("details");
            query.fields().exclude("noticeImage");
            query.fields().exclude("ticketTimeList");
            query.fields().exclude("describeElectronic");
            // 推荐
            query.addCriteria(Criteria.where("isExclusive").is(1));
            List<KylinPerformanceVo> exclusiveList = mongoTemplate.find(query, KylinPerformanceVo.class, KylinPerformanceVo.class.getSimpleName());
            redisUtil.set(redisKey, exclusiveList);
        }
        return (List<KylinPerformanceVo>) redisUtil.get(redisKey);
    }

    /**
     * 获取巡演列表redis
     */
    public List<KylinPerformanceVo> getRoadList(String roadShowId) {
        String redisKey = KylinRedisConst.PERFORMANCES_ROADLIST.concat(roadShowId);
        if (!redisUtil.hasKey(redisKey)) {
            List<KylinPerformanceVo> roadList = mongoTemplate.find(
                    Query.query(Criteria.where("roadShowId").is(roadShowId)),
                    KylinPerformanceVo.class,
                    KylinPerformanceVo.class.getSimpleName()
            );
            redisUtil.set(redisKey, roadList);
        }
        return (List<KylinPerformanceVo>) redisUtil.get(redisKey);
    }

    /**
     * 演出搜索固定参数
     */
    public Query getCommonWhere() {
        // 固定条件
        Query query = new Query();
        LocalDateTime nowTime = LocalDateTime.now();
        String nowTimeStr = DateUtil.Formatter.yyyyMMddHHmmss.format(nowTime);
        query.addCriteria(Criteria.where("timeEnd").gte(nowTimeStr));
        query.addCriteria(Criteria.where("appStatus").in(6, 8, 9, 10));
        query.addCriteria(Criteria.where("isShow").is(1));
        return query;
    }

    public KylinOrderRefundsVo getOrderRefundVo(String orderRefundsId){
        if (redisUtil.hasKey(KylinRedisConst.ORDER_REFUND + orderRefundsId)) {
        } else {
            KylinOrderRefundsVo orderRefundsVo = mongoTemplate.findOne(Query.query(Criteria.where("orderRefundsId").is(orderRefundsId)), KylinOrderRefundsVo.class, KylinOrderRefundsVo.class.getSimpleName());
            List<KylinOrderRefundEntitiesVo> kylinOrderTicketEntitiesVoList = mongoTemplate.find(Query.query(Criteria.where("orderRefundsId").is(orderRefundsId)), KylinOrderRefundEntitiesVo.class, KylinOrderRefundEntitiesVo.class.getSimpleName());
            List<KylinOrderRefundPicVo> picVos = mongoTemplate.find(Query.query(Criteria.where("orderRefundsId").is(orderRefundsId)),KylinOrderRefundPicVo.class,KylinOrderRefundPicVo.class.getSimpleName());
            orderRefundsVo.setOrderRefundEntitiesVoList(kylinOrderTicketEntitiesVoList);
            orderRefundsVo.setOrderRefundPicVos(picVos);
            redisUtil.set(KylinRedisConst.ORDER_REFUND + orderRefundsId, orderRefundsVo);
        }
        return (KylinOrderRefundsVo) redisUtil.get(KylinRedisConst.ORDER_REFUND + orderRefundsId);
    }

    public List<KylinOrderRefundsVo> getOrderRefundVoByOrderId(String orderId){
        if (redisUtil.hasKey(KylinRedisConst.ORDER_REFUND_BY_ORDER_ID + orderId)) {
        } else {
            List<KylinOrderRefundsVo> orderRefundsVoList = new ArrayList<>();
            List<KylinOrderRefundsVo> data = mongoTemplate.find(Query.query(Criteria.where("orderTicketsId").is(orderId)), KylinOrderRefundsVo.class, KylinOrderRefundsVo.class.getSimpleName());
            for (KylinOrderRefundsVo item : data) {
                List<KylinOrderRefundEntitiesVo> kylinOrderTicketEntitiesVoList = mongoTemplate.find(Query.query(Criteria.where("orderRefundsId").is(item.getOrderRefundsId())), KylinOrderRefundEntitiesVo.class, KylinOrderRefundEntitiesVo.class.getSimpleName());
                if(kylinOrderTicketEntitiesVoList.size()>1){
                    continue;
                }
                item.setOrderRefundEntitiesVoList(kylinOrderTicketEntitiesVoList);
                List<KylinOrderRefundPicVo> picVos = mongoTemplate.find(Query.query(Criteria.where("orderRefundsId").is(item.getOrderRefundsId())),KylinOrderRefundPicVo.class,KylinOrderRefundPicVo.class.getSimpleName());
                item.setOrderRefundPicVos(picVos);
                orderRefundsVoList.add(item);
            }
            redisUtil.set(KylinRedisConst.ORDER_REFUND_BY_ORDER_ID + orderId, orderRefundsVoList);
        }
        return (List<KylinOrderRefundsVo>) redisUtil.get(KylinRedisConst.ORDER_REFUND_BY_ORDER_ID + orderId);
    }

    public void delOrderRefundVo(String orderRefundsId){
        redisUtil.del(KylinRedisConst.ORDER_REFUND + orderRefundsId);
    }

    public void delOrderRefundVoByOrderId(String orderId){
        redisUtil.del(KylinRedisConst.ORDER_REFUND_BY_ORDER_ID + orderId);
    }

    /**
     * 获取可退款票单价格
     * @param orderTicketVo         订单vo数据
     * @param orderTicketEntitiesId 票单id
     * @return
     */
    public BigDecimal getCanRefundOrderEntitiesPrice(KylinOrderTicketVo orderTicketVo, List<KylinOrderRefundsVo> kylinOrderRefundsVoBaseList, String orderTicketEntitiesId){
        //单票实付价格
        BigDecimal singlePrice = orderTicketVo.getPriceActual().subtract(orderTicketVo.getPriceExpress()).divide(BigDecimal.valueOf(orderTicketVo.getNumber()));
        //单票剩余未退款金额
        BigDecimal canRefundSinglePrice = new BigDecimal("0.00");
        for (KylinOrderTicketEntitiesVo item : orderTicketVo.getEntitiesVoList()) {
            if (item.getOrderTicketEntitiesId().equals(orderTicketEntitiesId)) {
                canRefundSinglePrice = singlePrice.subtract(item.getRefundPrice()==null?BigDecimal.valueOf(0.00):item.getRefundPrice());
            }
        }
        System.out.println("单票剩余未退款金额 : "+canRefundSinglePrice);
        //订单锁定金额
        BigDecimal orderLockPrice = new BigDecimal("0.00");
        for (KylinOrderRefundsVo refundVo : kylinOrderRefundsVoBaseList) {
            int status = refundVo.getStatus();
            if (status == 0 || status == 3 || status == 5 || status == 7) { //退款流程未完成 金额锁定
                List<KylinOrderRefundEntitiesVo> refundEntitiesVos = refundVo.getOrderRefundEntitiesVoList();
                for (KylinOrderRefundEntitiesVo refundEntitiesVo : refundEntitiesVos) {
                    if (refundEntitiesVo.getOrderTicketEntitiesId().equals(orderTicketEntitiesId)) {
                        orderLockPrice = orderLockPrice.add(refundEntitiesVo.getRefundPrice());
                    }
                }
            }
        }
        System.out.println("订单锁定金额 : "+orderLockPrice);
        //订单可退金额
        BigDecimal refundSinglePrice = canRefundSinglePrice.subtract(orderLockPrice);
        System.out.println("订单可退金额 : "+refundSinglePrice);
        //是否包含快递费
        if (orderTicketVo.getPriceActual().subtract(orderTicketVo.getPriceExpress()).subtract(orderTicketVo.getPriceRefund()).compareTo(canRefundSinglePrice) == 0) {
            refundSinglePrice = refundSinglePrice.add(orderTicketVo.getPriceExpress());
        }
        return refundSinglePrice;
    }

    /**
     * 获取演出订单快递状态
     */
    public Integer getOrderExpressInfo(String orderId) {
        String redisKey = KylinRedisConst.ORDER_EXPRESS_INFO.concat(orderId);
        if (!redisUtil.hasKey(redisKey)) {
            return 1;
        } else {
            KylinOrderExpressVo expressInfo  = (KylinOrderExpressVo) redisUtil.get(redisKey);
            return expressInfo.getExpressStatus();
        }
    }

}
