package com.liquidnet.service.adam.controller;

import com.fasterxml.jackson.databind.JsonNode;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.constant.LnsEnum;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.HttpUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.commons.lang.util.ServletUtils;
import com.liquidnet.service.adam.dto.AdamMemberOrderCallbackParam;
import com.liquidnet.service.adam.dto.AdamMemberOrderCodeParam;
import com.liquidnet.service.adam.dto.AdamMemberOrderParam;
import com.liquidnet.service.adam.dto.AdamMemberOrderResult;
import com.liquidnet.service.adam.dto.vo.*;
import com.liquidnet.service.adam.service.AdamRdmService;
import com.liquidnet.service.adam.service.IAdamMemberOrderService;
import com.liquidnet.service.adam.service.IAdamUserService;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.PagedResult;
import com.liquidnet.service.base.ResponseDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.util.List;

@ApiSupport(order = 10031)
@Api(tags = "购买会员")
@Slf4j
@Validated
@RestController
@RequestMapping("member/order")
public class AdamMemberOrderController {
    @Autowired
    Environment env;
    @Autowired
    AdamRdmService adamRdmService;
    @Autowired
    IAdamMemberOrderService adamMemberOrderService;
    @Autowired
    IAdamUserService adamUserService;

    @ApiOperationSupport(order = 0)
    @ApiOperation(value = "购买会员预览")
    @GetMapping("preview/{mno}/{id}")
    public ResponseDto<AdamMemberOrderPreviewVo> preview(@NotBlank(message = "会员类型ID不能为空") @PathVariable String mno,
                                                         @NotBlank(message = "会员价格ID不能为空") @PathVariable String id) {
        AdamMemberVo memberVo = adamRdmService.getMemberVoByMemberId(mno);
        if (null == memberVo) {
            return ResponseDto.failure(ErrorMapping.get("10201"));
        }
        AdamMemberPriceVo memberPriceVo = adamRdmService.getMemberPriceVoByPriceId(mno, id);
        if (null == memberPriceVo) {
            return ResponseDto.failure(ErrorMapping.get("10202"));
        }
        String currentUid = CurrentUtil.getCurrentUid();

        AdamMemberOrderPreviewVo previewVo = AdamMemberOrderPreviewVo.getNew();
        previewVo.setMemberId(mno);
        previewVo.setMemberTitle(memberVo.getTitle());

        previewVo.setRealInfoVo(adamRdmService.getRealInfoVoByUid(currentUid));
        previewVo.setPriceVo(memberPriceVo);

        AdamUserMemberVo userMemberVo = adamRdmService.getUserMemberVoByUid(currentUid);
        if (null == userMemberVo) {
            previewVo.setStageMarker(0);
        } else if (userMemberVo.isActive()) {
            previewVo.setStageMarker(userMemberVo.isOldMember() ? 10 : 11);
        } else {
            previewVo.setStageMarker(2);
        }

        AdamUserInfoVo userInfoVo = adamRdmService.ratingProvince(adamRdmService.getUserInfoVoByUid(currentUid));
        previewVo.setBirthday(userInfoVo.getBirthday());
        previewVo.setArea(userInfoVo.getArea());
        previewVo.setProvince(userInfoVo.getProvince());
        previewVo.setCity(userInfoVo.getCity());
        previewVo.setCounty(userInfoVo.getCounty());

        return ResponseDto.success(previewVo);
    }

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "购买会员或会员码")
    @PostMapping("buy")
    public ResponseDto<AdamMemberOrderResult> buyMemberOrCode(@Valid @RequestBody AdamMemberOrderParam param) {
        log.debug("member/order/buy:param:{}", JsonUtils.toJson(param));
        String currentUid = CurrentUtil.getCurrentUid();

        if (checkMobileOrUid(currentUid)) {
            return ResponseDto.failure(ErrorMapping.get("10213"));
        }

        AdamUserMemberVo userMemberVo = adamRdmService.getUserMemberVoByUid(currentUid);
//        if (null == userMemberVo || !userMemberVo.isActive()) {
//            // 当前用户非会员或会员已过期时，购买会员时会受系统配置[开放/限制购买会员]限制
//            if (1 != adamRdmService.getSwitch(AdamRedisConst.SWITCH_BUY_MEMBER)) {
//                return ResponseDto.failure(ErrorMapping.get("10215"));
//            }
//        }
        if (null == userMemberVo) {// 普通用户购买会员时会受系统配置[是否开售、限购]限制
            AdamMemberVo memberVo = adamRdmService.getMemberVoByMemberId(param.getMemberId());

            if (1 != memberVo.getOnsale()) {// 是否开售:1-开售｜2-停售
                return ResponseDto.failure(ErrorMapping.get("10216"));
            }

//            if (0 == memberVo.calculateLimitMarker(adamRdmService.getMaxMemberLimitation())) {
//                // 购买会员限购[0-名额已满|1-允许购买｜2-限购未开始]
//                return ResponseDto.failure(ErrorMapping.get("10215"));
//            }
        }

        AdamRealInfoVo realInfoVo = adamRdmService.getRealInfoVoByUid(currentUid);
        if (null == realInfoVo) {
            if (StringUtils.isBlank(param.getName()) || StringUtils.isBlank(param.getIdCard())) {
                return ResponseDto.failure(ErrorMapping.get("10101"));
            }
            adamUserService.identity(currentUid, param.getName(), param.getIdCard(), (String) CurrentUtil.getTokenClaims().get(CurrentUtil.TOKEN_MOBILE));
        }

        return adamMemberOrderService.buyMemberOrMemberCode(currentUid, param);
    }

    /**
     * 当订单支付成功后，会立即通知Server一次，若未能响应成功，之后的24小时内会多次通知
     * 通知头：
     * {
     * "token":"eyJpdiI6IlJuTDFpOVBFOVBReUJ6OTd5ankzVVE9PSIsInZhbHVlIjoiMzhxWW52Um1lWVNYOEJ4OVh6OEdzUT09IiwibWFjIjoiM2Q4OWNkNWVkMjJkNzY0OGVjNmE5YTA3ODUwMTI5YWFlN2U5NDgwZmE0YmVmZWE4MTE1NmY1NWZjMGI2YmNiZCJ9"
     * }
     * 通知参数:
     * {
     * "status":"1",
     * "type":"MBEANS",
     * "code":"201807201217135B516249085EC",
     * "order_code":"00001TESTORDER",
     * "price":"0.01",
     * "payment_type":"APPIAP",
     * "payment_at":"2018-07-20 12:17:13",
     * "payment_id":"1032334083425793457349875"
     * }
     * 通知说明：已支付订单status为1，code为交易支付订单号，建议保存code，type为支付种类，payment_type为支付类型，以上参数是必传。
     * 通知响应：'success'，响应成功，将不再通知客户端
     * 通知响应：'fail'，响应失败，将不再通知客户端，建议在订单异常的状态下响应失败
     *
     * @return
     */
    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "会员支付回调")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "status", value = "1-成功｜0-失败", example = "1"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "orderCode", value = "会员订单号", example = "77062761607274496V"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "code", value = "支付订单号", example = "20210616153954199964032352974P"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "type", value = "购买类型", example = "VIP"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "price", value = "支付金额", example = "0.01"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "paymentType", value = "支付类型", example = "APPALIPAY"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "paymentAt", value = "支付时间", example = "2021-06-16 15:40:02"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "paymentId", value = "支付ID", example = "2021061622001468121421203878"),
    })
    @PostMapping("callback")
    public String paymentNotifyCallBack() {
        log.debug("/member/callback.mapParameter:{}, token:{}", JsonUtils.toJson(ServletUtils.getRequest().getParameterMap()), ServletUtils.getRequest().getHeader("token"));
        // TODO: 2021/6/13 验签

        AdamMemberOrderCallbackParam parameter = AdamMemberOrderCallbackParam.getNew().get(ServletUtils.getRequest());
        if (1 != parameter.getStatus() ||
                !parameter.getType().equals("VIP") ||
                null == parameter.getOrderCode() ||
                null == parameter.getCode()) {
            log.error("###购买会员支付回调参数有误:{}", JsonUtils.toJson(parameter));

            return "fail";
        }

        if (!LnsEnum.ENV.dev.name().equals(env.getProperty(CurrentUtil.CK_ENV_ACTIVE))) {
            ResponseDto<Integer> checkOrderResult = this.checkOrderResult(parameter.getOrderCode());
            if (!checkOrderResult.isSuccess() || checkOrderResult.getData() != 1) {
                log.warn("###购买会员支付回调支付中心验证不通过，无法处理[{}]", JsonUtils.toJson(parameter));
                return "fail";
            }
        }

        ResponseDto<Object> dto = adamMemberOrderService.paymentNotifyCallBack(parameter);

        if (!dto.isSuccess()) {
            log.warn("###购买会员支付回调处理失败:dto:{},CallbackParam:{}", JsonUtils.toJson(dto), JsonUtils.toJson(parameter));

            return "fail";
        }
        return "success";
    }

    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "兑换码校验", notes = "响应参数[`code`:0-当前兑换码可用，否则不可用]")
    @GetMapping("check/code")
    public ResponseDto<Integer> checkMemberCode(@NotBlank(message = "兑换码不能为空") @RequestParam String mCode) {
        AdamMemberCodeVo vo = adamRdmService.getMemberCodeVoByCode(mCode);
        if (null == vo) {
            return ResponseDto.failure(ErrorMapping.get("10207"));
        }
        if (vo.getState() != 0) {
            return ResponseDto.failure(ErrorMapping.get(vo.getState() == 1 ? "10208" : "10209"));
        }
        return ResponseDto.success(0);
    }

    /**
     * 购买会员黑名单校验
     *
     * @param uid 用户ID
     * @return true-命中黑名单
     */
    private boolean checkMobileOrUid(String uid) {
        List<String> blacklist = adamRdmService.getBlacklistForMember();
        return !CollectionUtils.isEmpty(blacklist) && blacklist.contains(uid);
    }

    @ApiOperationSupport(order = 4)
    @ApiOperation(value = "使用兑换码")
    @PostMapping("exchange")
    public ResponseDto<AdamMemberOrderResult> exchangeMemberCode(@Valid @RequestBody AdamMemberOrderCodeParam param) {
        log.debug("member/order/exchange:param:{}", JsonUtils.toJson(param));
        String currentUid = CurrentUtil.getCurrentUid();
        // TODO: 2021/6/19 关闭该验证（如果已是会员继续兑换则保留原会员号，兑换码对应会员号记录至会员订单）
//        if (null != adamRdmService.getUserMemberVoByUid(currentUid)) {// 仅限从未购买过会员的用户使用
//            return ResponseDto.failure(ErrorMapping.get("10200"));
//        }

        if (checkMobileOrUid(currentUid)) {
            return ResponseDto.failure(ErrorMapping.get("10213"));
        }

        AdamRealInfoVo realInfoVo = adamRdmService.getRealInfoVoByUid(currentUid);
        if (null == realInfoVo) {
            if (StringUtils.isBlank(param.getName()) || StringUtils.isBlank(param.getIdCard())) {
                return ResponseDto.failure(ErrorMapping.get("10101"));
            }
            AdamRealInfoVo vo = adamUserService.identity(currentUid, param.getName(), param.getIdCard(), (String) CurrentUtil.getTokenClaims().get(CurrentUtil.TOKEN_MOBILE));
            if (null == vo) {
                return ResponseDto.failure(ErrorMapping.get("10000"));
            }
        }

        return adamMemberOrderService.exchangeMemberCode(currentUid, param);
    }

    @ApiOperationSupport(order = 5)
    @ApiOperation(value = "会员订单列表")
    @GetMapping("list")
    public ResponseDto<PagedResult<AdamMemberOrderSimpleVo>> list(@RequestParam(defaultValue = "1", required = false) int pageNo,
                                                                  @RequestParam(defaultValue = "5", required = false) int pageSize) {
        return ResponseDto.success(adamMemberOrderService.queryPage(CurrentUtil.getCurrentUid(), pageNo, pageSize));
    }

    @ApiOperationSupport(order = 6)
    @ApiOperation(value = "会员订单详情")
    @GetMapping("info/{orderNo}")
    public ResponseDto<AdamMemberOrderVo> getMemberOrder(@NotBlank(message = "订单号不能为空") @PathVariable String orderNo) {
        AdamMemberOrderVo mOrderVo = adamRdmService.getShotMemberOrderVoByOrderNo(orderNo);

        if (null == mOrderVo) {
            List<AdamMemberOrderVo> memberOrderVos = adamRdmService.getMemberOrderVosByUid(CurrentUtil.getCurrentUid());

            mOrderVo = adamRdmService.getMemberOrderVoByOrderNo(memberOrderVos, orderNo);
        }

        if (null != mOrderVo) {
            AdamMemberVo memberVo = adamRdmService.getMemberVoByMemberId(mOrderVo.getMemberId());
            mOrderVo.setMemberNotes(memberVo.getNotes());
            if (mOrderVo.getMode() == 1) {
                AdamMemberCodeVo memberCodeVo = adamRdmService.getMemberCodeVoByBuyOrderNo(mOrderVo.getOrderNo());

                mOrderVo.setMemberCodeVo(memberCodeVo);
            }
            return ResponseDto.success(mOrderVo);
        }
        return ResponseDto.failure(ErrorMapping.get("10211"));
    }

    @ApiOperationSupport(order = 7)
    @ApiOperation(value = "会员订单状态", notes = "响应参数[`data`:0-未支付｜1-已支付]")
    @GetMapping("check")
    public ResponseDto<Integer> checkOrderResult(@NotBlank(message = "订单号不能为空") @RequestParam String orderNo) {
        AdamMemberOrderVo memberOrderInfo = adamRdmService.getShotMemberOrderVoByOrderNo(orderNo);

        if (null == memberOrderInfo) {
            List<AdamMemberOrderVo> memberOrderVos = adamRdmService.getMemberOrderVosByUid(CurrentUtil.getCurrentUid());

            memberOrderInfo = adamRdmService.getMemberOrderVoByOrderNo(memberOrderVos, orderNo);

            if (null != memberOrderInfo && memberOrderInfo.getState() == 1) {
                return ResponseDto.success(1);
            }
        }

        if (null == memberOrderInfo) {
            return ResponseDto.failure(ErrorMapping.get("10211"));
        }

        if (memberOrderInfo.getMode() <= 1) {
            try {
                // 支付中心返回说明:已支付订单status==1，未支付订单status==0
                String respStr = HttpUtil.get(env.getProperty("liquidnet.url-pay.check") + "?code=" + memberOrderInfo.getPayNo(), null);

                JsonNode respJNode = JsonUtils.fromJson(respStr, JsonNode.class);

                if (null == respJNode || !"0".equals(respJNode.get("code").asText())) {
                    log.warn("购买会员:订单状态查询失败[orderNo:{},respStr:{}]", orderNo, respStr);
                    return ResponseDto.failure(ErrorMapping.get("10212"));
                }

                int status = respJNode.get("data").get("status").asInt();

                return ResponseDto.success(status == 1 ? status : 0);
            } catch (Exception e) {
                log.error("购买会员:订单状态查询失败:{}", orderNo, e);
                return ResponseDto.failure(ErrorMapping.get("10212"));
            }
        }
        return ResponseDto.success(memberOrderInfo.getState());
    }
}
