package com.liquidnet.service.consumer.base.receiver.sweet;

import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.base.constant.MQConst;
import com.liquidnet.service.consumer.base.receiver.AbstractSqlRedisReceiver;
import com.liquidnet.service.feign.stone.api.FeignStoneIntegralClient;
import com.liquidnet.service.sweet.param.SweetStoneIntegralParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.connection.stream.MapRecord;
import org.springframework.data.redis.connection.stream.StreamRecords;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import java.util.HashMap;

@Slf4j
@Component
public class ConsumerSweetStoneIntegralReceiver extends AbstractSqlRedisReceiver {

    @Autowired
    StringRedisTemplate stringRedisTemplate;

    @Autowired
    private FeignStoneIntegralClient feignStoneIntegralClient;

    @Override
    protected String getRedisStreamKey() {
        return MQConst.SweetQueue.SWEET_STONE_INTEGRAL.getKey();
    }

    @Override
    protected String getRedisStreamGroup() {
        return MQConst.SweetQueue.SWEET_STONE_INTEGRAL.getGroup();
    }

    @Override
    public void onMessage(MapRecord<String, String, String> message) {
        String redisStreamKey = this.getRedisStreamKey();
        log.debug("CONSUMER MSG[streamKey:{},messageId:{},stream:{},body:{}]", redisStreamKey, message.getId(), message.getStream(), message.getValue());
        boolean result = this.consumerMessageHandler(message.getValue().get("message"));
        log.info("CONSUMER MSG RESULT:{} ==> [{}]MESSAGE_ID:{}", result, redisStreamKey, message.getId());

        try {
            stringRedisTemplate.opsForStream().acknowledge(getRedisStreamGroup(), message);
        } catch (Exception e) {
            log.error("#CONSUMER MSG EX_ACK ==> [{}]RESULT:{},MESSAGE:{}", redisStreamKey, result, message.getValue(), e);
        }
        try {
            stringRedisTemplate.opsForStream().delete(redisStreamKey, message.getId());
        } catch (Exception e) {
            log.error("#CONSUMER MSG EX_DEL ==> [{}]RESULT:{},MESSAGE:{}", redisStreamKey, result, message.getValue(), e);
        }
    }

    private boolean consumerMessageHandler(String msg) {
        boolean aBoolean = false;
        try {
            SweetStoneIntegralParam param = JsonUtils.fromJson(msg, SweetStoneIntegralParam.class);
            if (param == null) {
                aBoolean = true;
            } else {
                ResponseDto<String> res = null;
                if (null == param.getType()) {
                    aBoolean = true;
                    return true;
                }
                if (param.getType() == 1) {
                    res = feignStoneIntegralClient.in2112(
                            param.getUid(),
                            param.getScore(),
                            param.getContent()
                    );
                } else if (param.getType() == 2) {
                    res = feignStoneIntegralClient.de2111(
                            param.getUid(),
                            param.getScore(),
                            param.getContent()
                    );
                }
                if (null != res && res.getCode().equals("0")) {
                    aBoolean = true;
                }
            }
        } catch (Exception e) {
            log.error("CONSUMER MSG EX_HANDLE ==> [{}]:{}", this.getRedisStreamKey(), msg, e);
        } finally {
            if (!aBoolean) {
                HashMap<String, String> map = CollectionUtil.mapStringString();
                map.put("message", msg);
                stringRedisTemplate.opsForStream().add(StreamRecords.mapBacked(map).withStreamKey(this.getRedisStreamKey()));
            }
        }
        return aBoolean;
    }
}
