package com.liquidnet.service.kylin.controller;


import com.github.pagehelper.PageInfo;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.kylin.dto.param.PayAgainParam;
import com.liquidnet.service.kylin.dto.param.PayOrderParam;
import com.liquidnet.service.kylin.dto.param.SyncOrderParam;
import com.liquidnet.service.kylin.dto.vo.returns.KylinOrderListVo;
import com.liquidnet.service.kylin.dto.vo.returns.OrderDetailsVo;
import com.liquidnet.service.kylin.dto.vo.returns.OrderRefundVo;
import com.liquidnet.service.kylin.dto.vo.returns.PayResultVo;
import com.liquidnet.service.kylin.service.IKylinOrderTicketsService;
import com.liquidnet.service.kylin.service.impl.KylinOrderTicketsServiceImpl;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import javax.validation.constraints.NotNull;
import java.util.HashMap;
import java.util.List;

/**
 * <p>
 * 订单 前端控制器
 * </p>
 *
 * @author liquidnet
 * @since 2021-05-20
 */
@Api(tags = "前端-订单相关")
@RestController
@RequestMapping("order")
@Validated
public class KylinOrderTicketsController {

    @Autowired
    IKylinOrderTicketsService orderTicketsService;

    @PostMapping("pre")
    @ApiOperation("下单")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<PayResultVo> checkCanOrder(@RequestBody @Valid PayOrderParam payOrderParam) {
        return orderTicketsService.checkCanOrder(payOrderParam);
    }

    @PostMapping("payAgain")
    @ApiOperation("再次支付")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<PayResultVo> payAgain(@RequestBody @Valid PayAgainParam payAgainParam) {
        PayResultVo vo = orderTicketsService.payAgain(payAgainParam).getData();
        if (null == vo) {
            return ResponseDto.failure("订单已失效");
        } else {
            return ResponseDto.success(vo);
        }
    }

    @PostMapping("syncOrder")
    @ApiOperation("支付回调")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public String syncOrder(@ModelAttribute @Valid SyncOrderParam syncOrderParam) {
        return orderTicketsService.syncOrder(syncOrderParam);
    }

    @GetMapping("list")
    @ApiOperation("订单列表")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<PageInfo<List<KylinOrderListVo>>> orderList() {
        return ResponseDto.success(orderTicketsService.orderList());
    }

    @GetMapping("details")
    @ApiOperation("订单详情")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<OrderDetailsVo> orderDetails(@RequestParam(value = "orderId", required = false) @NotNull(message = "订单id不能为空") String orderId) {
        OrderDetailsVo vo = orderTicketsService.orderDetails(orderId);
        if (null == vo) {
            return ResponseDto.failure(ErrorMapping.get("20003"));
        } else {
            return ResponseDto.success(vo);
        }
    }

    @GetMapping("toOrderRefundDetails")
    @ApiOperation("想要退款订单详情")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<OrderDetailsVo> toOrderRefundDetails(@RequestParam(value = "orderId", required = false) @NotNull(message = "订单id不能为空") String orderId) {
        OrderDetailsVo vo = orderTicketsService.toOrderRefundDetails(orderId);
        if (null == vo) {
            return ResponseDto.failure(ErrorMapping.get("20003"));
        } else {
            return ResponseDto.success(vo);
        }
    }

    @PostMapping("sendOrderRefund")
    @ApiOperation("发起退款申请")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<Boolean> sendOrderRefund(@RequestParam(value = "orderId", required = true) @NotNull(message = "订单id不能为空") String orderId,
                                                @RequestParam(value = "orderTicketEntitiesId", required = true) @NotNull(message = "票单id不能为空") String orderTicketEntitiesId,
                                                @RequestParam(value = "reason", required = true) @NotNull(message = "退款原因不能为空") String reason,
                                                @RequestParam(value = "picList", required = false) String picList) {
        return orderTicketsService.sendOrderRefund(orderId, orderTicketEntitiesId, reason, picList);
    }

    @GetMapping("orderRefundDetails")
    @ApiOperation("发起退款申请")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<OrderRefundVo> orderRefundDetails(@RequestParam(value = "orderId", required = true) @NotNull(message = "订单id不能为空") String orderId,
                                                         @RequestParam(value = "orderRefundId", required = true) @NotNull(message = "退款订单id不能为空") String orderRefundId) {
        return ResponseDto.success(orderTicketsService.orderRefundDetails(orderId, orderRefundId));
    }

    @PostMapping("sendOrderRefund")
    @ApiOperation("发起退款申请")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<Boolean> orderRefundWithdraw(@RequestParam(value = "orderRefundId", required = true) @NotNull(message = "退款订单id不能为空") String orderRefundId) {
        return orderTicketsService.orderRefundWithdraw(orderRefundId) == null ? ResponseDto.failure() : ResponseDto.success();
    }


    @GetMapping("checkPayment")
    @ApiOperation("订单状态")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<HashMap> checkOrderResult(@RequestParam("orderId") @NotNull(message = "订单id不能为空") String orderId) {
        Integer status = orderTicketsService.checkOrderResult(orderId).getData();
        if (null == status) {
            return ResponseDto.failure(ErrorMapping.get("20003"));
        } else {
            HashMap<String, Integer> map = new HashMap<>();
            map.put("status", status);
            return ResponseDto.success(map);
        }
    }

    @GetMapping("orderUnPayCount")
    @ApiOperation("待支付演出订单数量")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<HashMap> orderUnPayCount() {
        Integer unPayCount = orderTicketsService.orderUnPayCount().getData();
        HashMap<String, Integer> map = new HashMap<>();
        map.put("unPayCount", unPayCount);
        return ResponseDto.success(map);
    }


    @PostMapping("checkOrderTime")
    @ApiOperation("订单过期脚本")
    @ApiResponse(code = 200, message = "接口返回对象参数")
    public ResponseDto<Boolean> checkOrderTime() {
        return ResponseDto.success(orderTicketsService.checkOrderTime(""));
    }
}
