package com.liquidnet.service.goblin.controller.manage;

import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.liquidnet.commons.lang.util.CollectionUtil;
import com.liquidnet.commons.lang.util.CurrentUtil;
import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.base.ErrorMapping;
import com.liquidnet.service.base.PagedResult;
import com.liquidnet.service.base.ResponseDto;
import com.liquidnet.service.goblin.dto.GoblinGoodsSpecDto;
import com.liquidnet.service.goblin.dto.manage.GoblinStoreMgtGoodsActionParam;
import com.liquidnet.service.goblin.dto.manage.GoblinStoreMgtGoodsAddParam;
import com.liquidnet.service.goblin.dto.manage.GoblinStoreMgtGoodsEditSkuParam;
import com.liquidnet.service.goblin.dto.manage.GoblinStoreMgtGoodsFilterParam;
import com.liquidnet.service.goblin.dto.manage.vo.GoblinStoreMgtGoodsInfoVo;
import com.liquidnet.service.goblin.dto.manage.vo.GoblinStoreMgtGoodsListVo;
import com.liquidnet.service.goblin.dto.vo.*;
import com.liquidnet.service.goblin.service.manage.IGoblinstoreMgtGoodsService;
import com.liquidnet.service.goblin.util.GoblinMongoUtils;
import com.liquidnet.service.goblin.util.GoblinRedisUtils;
import com.liquidnet.service.goblin.util.ObjectUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

@ApiSupport(order = 149004)
@Api(tags = "商品管理")
@Slf4j
@Validated
@RestController
@RequestMapping("store/mgt/goods")
public class GoblinStoreMgtGoodsController {
    @Autowired
    IGoblinstoreMgtGoodsService goblinstoreMgtGoodsService;
    @Autowired
    GoblinRedisUtils goblinRedisUtils;
    @Autowired
    GoblinMongoUtils goblinMongoUtils;

    @ApiOperationSupport(order = 1)
    @ApiOperation(value = "SPU列表")
    @PostMapping("list")
    public ResponseDto<PagedResult<GoblinStoreMgtGoodsListVo>> list(@Valid @RequestBody GoblinStoreMgtGoodsFilterParam storeMgtGoodsFilterParam) {
        String currentUid = CurrentUtil.getCurrentUid();
        if (!goblinRedisUtils.hasStoreId(currentUid, storeMgtGoodsFilterParam.getStoreId())) {
            return ResponseDto.success();
        }
        if (log.isDebugEnabled()) {
            log.debug("商品管理:SPU列表:[GoblinStoreMgtGoodsFilterParam={}]", JsonUtils.toJson(storeMgtGoodsFilterParam));
        }
        return ResponseDto.success(goblinstoreMgtGoodsService.goodsList(storeMgtGoodsFilterParam));
    }

    @ApiOperationSupport(order = 2)
    @ApiOperation(value = "SPU导出")
    @PostMapping("export")
    public void export(@Valid @RequestBody GoblinStoreMgtGoodsFilterParam storeMgtGoodsFilterParam) {
        List<String> storeIds = goblinRedisUtils.getStoreIds(CurrentUtil.getCurrentUid());
        boolean storeIdIsBlank = StringUtils.isBlank(storeMgtGoodsFilterParam.getStoreId());
        if (CollectionUtils.isEmpty(storeIds) || (
                !storeIdIsBlank && !goblinRedisUtils.hasStoreId(CurrentUtil.getCurrentUid(), storeMgtGoodsFilterParam.getStoreId())
        )) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("商品管理:SPU导出:[GoblinStoreMgtGoodsFilterParam={}]", JsonUtils.toJson(storeMgtGoodsFilterParam));
        }
        if (storeIdIsBlank) {
            storeMgtGoodsFilterParam.setStoreId(storeIds.get(0));
        }

        log.info("商品管理:SPU导出...");
        // TODO: 2021/12/28 zhanggb
    }

    @ApiOperationSupport(order = 3)
    @ApiOperation(value = "SPU管理")
    @PostMapping("operate")
    public ResponseDto<Object> action(@Valid @RequestBody GoblinStoreMgtGoodsActionParam storeMgtGoodsActionParam) {
        String currentUid = CurrentUtil.getCurrentUid();
        if (!goblinRedisUtils.hasStoreId(currentUid, storeMgtGoodsActionParam.getStoreId())) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        if (log.isDebugEnabled()) {
            log.debug("商品管理:SPU管理:[GoblinStoreMgtGoodsActionParam={}]", JsonUtils.toJson(storeMgtGoodsActionParam));
        }
        // TODO: 2021/12/28 zhanggb 逻辑校验

        switch (storeMgtGoodsActionParam.getAction()) {
            case "ONSHELVES":
                goblinstoreMgtGoodsService.goodsShelvesProcessing(storeMgtGoodsActionParam, currentUid, true);
                break;
            case "UNSHELVE":
                goblinstoreMgtGoodsService.goodsShelvesProcessing(storeMgtGoodsActionParam, currentUid, false);
                break;
            case "REMOVE":
                goblinstoreMgtGoodsService.goodsRemove(storeMgtGoodsActionParam, currentUid);
                break;
            default:
                log.warn("Invalid operation[action={}]", storeMgtGoodsActionParam.getAction());
                return ResponseDto.failure(ErrorMapping.get("149001"));
        }
        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 4)
    @ApiOperation(value = "SPU添加")
    @PutMapping("add")
    public ResponseDto<Object> add(@Valid @RequestBody GoblinStoreMgtGoodsAddParam storeMgtGoodsAddParam) {
        String currentUid = CurrentUtil.getCurrentUid();
        if (!goblinRedisUtils.hasStoreId(currentUid, storeMgtGoodsAddParam.getStoreId())) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        if (log.isDebugEnabled()) {
            log.debug("商品管理:SPU添加:[GoblinStoreMgtGoodsAddParam={}]", JsonUtils.toJson(storeMgtGoodsAddParam));
        }

        LocalDateTime now = LocalDateTime.now();
        GoblinGoodsInfoVo goodsInfoVo = storeMgtGoodsAddParam.initGoodsInfoVo();
        goodsInfoVo.setCreatedBy(currentUid);
        goodsInfoVo.setCreatedAt(now);

        ArrayList<GoblinGoodsSpecVo> goodsSpecVoList = ObjectUtil.getGoblinGoodsSpecVoArrayList();
        List<GoblinGoodsSkuInfoVo> goodsSkuInfoVoList = ObjectUtil.getGoblinGoodsSkuInfoVoArrayList();

        storeMgtGoodsAddParam.initGoodsSkuInfoVo(goodsInfoVo, goodsSkuInfoVoList, goodsSpecVoList);

        {// 标签处理
            List<String> goodsTagIdList = storeMgtGoodsAddParam.getTagList(), goodsExtagIdList = storeMgtGoodsAddParam.getExtagList();
            List<String> tagIdList = CollectionUtil.arrayListString();
            tagIdList.addAll(goodsTagIdList);
            tagIdList.addAll(goodsExtagIdList);
            List<GoblinSelfTagVo> selfTagVoList;
            if (!CollectionUtils.isEmpty(tagIdList) && !CollectionUtils.isEmpty(selfTagVoList = goblinMongoUtils.getMgtSelfTagVos(tagIdList))) {
                if (!CollectionUtils.isEmpty(goodsTagIdList)) {
                    List<GoblinGoodsTagVo> goodsTagVoList = ObjectUtil.getGoblinGoodsTagVoArrayList();
                    for (int i = 0; i < goodsTagIdList.size(); i++)
                        for (GoblinSelfTagVo r : selfTagVoList)
                            if (r.getTagId().equals(goodsTagIdList.get(i))) goodsTagVoList.add(GoblinGoodsTagVo.getNew().copy(r).setSort(i));
                    goodsInfoVo.setTagVoList(goodsTagVoList);
                }
                if (!CollectionUtils.isEmpty(goodsExtagIdList)) {
                    List<GoblinGoodsExtagVo> goodsExtagVoList = ObjectUtil.getGoblinGoodsExtagVoArrayList();
                    for (int i = 0; i < goodsExtagIdList.size(); i++)
                        for (GoblinSelfTagVo r : selfTagVoList)
                            if (r.getTagId().equals(goodsExtagIdList.get(i))) goodsExtagVoList.add(GoblinGoodsExtagVo.getNew().copy(r).setSort(i));
                    goodsInfoVo.setExtagVoList(goodsExtagVoList);
                }
            }
        }

        // 服务保障处理
        goodsInfoVo.setServiceSupportVoList(goblinMongoUtils.getServiceSupportVos(storeMgtGoodsAddParam.getSsidList()));

        goblinstoreMgtGoodsService.goodsAdd(goodsInfoVo, goodsSkuInfoVoList);
        return ResponseDto.success(goodsInfoVo.getSpuId());
    }

    @ApiOperationSupport(order = 5)
    @ApiOperation(value = "SPU详情")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "storeId", value = "店铺ID"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "spuId", value = "商品ID"),
    })
    @GetMapping("info")
    public ResponseDto<GoblinStoreMgtGoodsInfoVo> info(@NotBlank(message = "店铺ID不能为空") @RequestParam String storeId,
                                                       @NotBlank(message = "商品ID不能为空") @RequestParam String spuId) {
        if (!goblinRedisUtils.hasStoreId(CurrentUtil.getCurrentUid(), storeId)) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        if (log.isDebugEnabled()) {
            log.debug("商品管理:SPU详情:[spuId={}]", spuId);
        }
        return ResponseDto.success(goblinstoreMgtGoodsService.goodsInfo(storeId, spuId));
    }

    @ApiOperationSupport(order = 6)
    @ApiOperation(value = "商品编辑:SPU编辑", notes = "只修改商品信息，不包含规格相关信息")
    @PostMapping("edit_spu")
    public ResponseDto<Object> editSpu(@Valid @RequestBody GoblinStoreMgtGoodsAddParam storeMgtGoodsAddParam) {
        if (!goblinRedisUtils.hasStoreId(CurrentUtil.getCurrentUid(), storeMgtGoodsAddParam.getStoreId())) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        if (StringUtils.isBlank(storeMgtGoodsAddParam.getSpuId()) || null == goblinRedisUtils.getMgtGoodsInfoVo(storeMgtGoodsAddParam.getSpuId())) {
            return ResponseDto.failure(ErrorMapping.get("149010"));
        }
        if (log.isDebugEnabled()) {
            log.debug("商品管理:商品编辑:SPU编辑:[GoblinStoreMgtGoodsAddParam={}]", JsonUtils.toJson(storeMgtGoodsAddParam));
        }
        return ResponseDto.success(goblinstoreMgtGoodsService.goodsEditSpu(storeMgtGoodsAddParam));
    }

    @ApiOperationSupport(order = 7)
    @ApiOperation(value = "商品编辑:SKU编辑", notes = "只修改单品信息，不包含商品信息")
    @PostMapping("edit_sku")
    public ResponseDto<Object> editSku(@Valid @RequestBody GoblinStoreMgtGoodsEditSkuParam storeMgtGoodsEditSkuParam) {
        if (!goblinRedisUtils.hasStoreId(CurrentUtil.getCurrentUid(), storeMgtGoodsEditSkuParam.getStoreId())) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        GoblinGoodsSkuInfoVo mgtGoodsSkuInfoVo = goblinRedisUtils.getMgtGoodsSkuInfoVo(storeMgtGoodsEditSkuParam.getSkuId());
        if (StringUtils.isBlank(storeMgtGoodsEditSkuParam.getSkuId())
                || null == mgtGoodsSkuInfoVo || !mgtGoodsSkuInfoVo.getStoreId().equals(storeMgtGoodsEditSkuParam.getStoreId())) {
            return ResponseDto.failure(ErrorMapping.get("149011"));
        }
        GoblinGoodsInfoVo goodsInfoVo = goblinRedisUtils.getMgtGoodsInfoVo(mgtGoodsSkuInfoVo.getSpuId());
        // TODO: 2022/1/5 zhanggb +分类-ISBN校验

        if (log.isDebugEnabled()) {
            log.debug("商品管理:商品编辑:SKU编辑:[GoblinStoreMgtGoodsEditSkuParam={}]", JsonUtils.toJson(storeMgtGoodsEditSkuParam));
        }
        storeMgtGoodsEditSkuParam.setStock(mgtGoodsSkuInfoVo.getStock());
        storeMgtGoodsEditSkuParam.setSkuStock(mgtGoodsSkuInfoVo.getSkuStock());
        String paramSkuId = storeMgtGoodsEditSkuParam.getSkuId();
        Integer operStock = storeMgtGoodsEditSkuParam.getOperStock();
        if (null != operStock && operStock != 0) {// 处理库存
            int operStockVal = Math.abs(operStock);
            Integer stock = mgtGoodsSkuInfoVo.getSkuStock();// 总库存
            Integer skuStock = mgtGoodsSkuInfoVo.getSkuStock();// SKU总库存

            int rtSkuStock = goblinRedisUtils.getSkuStock(null, paramSkuId);// 当前剩余库存
            if (operStock < 0 && rtSkuStock >= operStockVal) {// SKU实时库存充足时允许扣减
                if (goblinRedisUtils.decrSkuStock(null, paramSkuId, operStockVal) < 0) {// 过程中库存不足时，回滚库存，编辑无效
                    goblinRedisUtils.incrSkuStock(null, paramSkuId, operStockVal);
                    return ResponseDto.failure(ErrorMapping.get("149012"));
                } else {
                    storeMgtGoodsEditSkuParam.setSkuStock(skuStock - operStockVal);
                    storeMgtGoodsEditSkuParam.setStock(stock - operStockVal);
                }
            } else if (operStock > 0) {// 增加
                goblinRedisUtils.incrSkuStock(null, paramSkuId, operStockVal);
                storeMgtGoodsEditSkuParam.setSkuStock(skuStock + operStockVal);
                storeMgtGoodsEditSkuParam.setStock(stock + operStockVal);
            }
        }
        List<String> skuIdList = goodsInfoVo.getSkuIdList();
        for (String skuId : skuIdList) {// 比对所有SKU规格信息
            GoblinGoodsSkuInfoVo existMgtGoodsSkuInfoVo = goblinRedisUtils.getMgtGoodsSkuInfoVo(skuId);
            Map<String, String> editBefSkuSpecMap = existMgtGoodsSkuInfoVo.getSkuSpecList().stream()
                    .collect(Collectors.toMap(GoblinGoodsSpecDto::getSpecName, GoblinGoodsSpecDto::getSpecVname, (k1, k2) -> k2));
            boolean diffSpecFlg = false;
            List<GoblinGoodsSpecDto> paramSkuSpecList = storeMgtGoodsEditSkuParam.getSkuSpecList();
            for (GoblinGoodsSpecDto paramGoodsSpecDto : paramSkuSpecList) {
                if (!editBefSkuSpecMap.get(paramGoodsSpecDto.getSpecName()).equals(paramGoodsSpecDto.getSpecVname())) {
                    diffSpecFlg = true;// 多个规格中只要存在一个不一致，即可跳过，标记为允许添加
                    break;
                }
            }
            if (skuId.equals(paramSkuId)) {
                if (diffSpecFlg) {// 更改了规格，则同步修改SPU里的规格信息
                    Map<String, String> paramSkuSpecMap = paramSkuSpecList.stream()
                            .collect(Collectors.toMap(GoblinGoodsSpecDto::getSpecName, GoblinGoodsSpecDto::getSpecVname, (k1, k2) -> k2));
                    List<GoblinGoodsSpecVo> specVoList = goodsInfoVo.getSpecVoList();
                    for (Map.Entry<String, String> entry : editBefSkuSpecMap.entrySet()) {
                        for (GoblinGoodsSpecVo specVo : specVoList) {
                            if (specVo.getSpecName().equals(entry.getKey())) {
                                List<GoblinGoodsSpecValueVo> specValues = specVo.getSpecValues();
                                int idx = IntStream.range(0, specValues.size())
                                        .filter(i -> specValues.get(i).getSpecVname().equals(entry.getValue()))
                                        .findFirst().orElse(-1);
                                if (idx == -1) {
                                    log.warn("商品管理:商品编辑:SKU编辑:规格信息不一致[specVoList={},editBefSkuSpecMap={}]", JsonUtils.toJson(specVoList), editBefSkuSpecMap);
                                    return ResponseDto.failure(ErrorMapping.get("149013"));
                                } else {
                                    specValues.get(idx).setSpecVname(paramSkuSpecMap.get(specVo.getSpecName()));
                                }
                            }
                        }
                    }
                } else {// 未更改规格，则标记为NULL（用于判断MONGO是否更新SPU规格信息）
                    storeMgtGoodsEditSkuParam.setSkuSpecList(null);
                }
                continue;
            }
            if (!diffSpecFlg) {
                log.warn("商品管理:商品编辑:SKU编辑:重复的规格信息[editBefSkuSpecMap={},paramSkuSpecMap={}]", editBefSkuSpecMap, JsonUtils.toJson(paramSkuSpecList));
                return ResponseDto.failure(ErrorMapping.get("149014"));
            }
        }
        if (goblinstoreMgtGoodsService.goodsEditSku(storeMgtGoodsEditSkuParam, goodsInfoVo)) {
            return ResponseDto.success(goblinRedisUtils.getMgtGoodsSkuInfoVo(paramSkuId));
        } else {
            if (null != operStock && operStock != 0) {// 处理库存:回滚
                int operStockVal = Math.abs(operStock);
                if (operStock > 0) {// 增加的减回去
                    goblinRedisUtils.decrSkuStock(null, paramSkuId, operStockVal);
                } else {// 扣减的加回去
                    goblinRedisUtils.incrSkuStock(null, paramSkuId, operStockVal);
                }
            }
            return ResponseDto.failure(ErrorMapping.get("149013"));
        }
    }

    @ApiOperationSupport(order = 8)
    @ApiOperation(value = "商品编辑:SKU添加")
    @PutMapping("edit_sku/add")
    public ResponseDto<Object> editSkuAdd(@Valid @RequestBody GoblinStoreMgtGoodsEditSkuParam storeMgtGoodsEditSkuParam) {
        if (!goblinRedisUtils.hasStoreId(CurrentUtil.getCurrentUid(), storeMgtGoodsEditSkuParam.getStoreId())) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        GoblinGoodsInfoVo goodsInfoVo = goblinRedisUtils.getMgtGoodsInfoVo(storeMgtGoodsEditSkuParam.getSpuId());
        if (null == goodsInfoVo || !goodsInfoVo.getStoreId().equals(storeMgtGoodsEditSkuParam.getStoreId())) {
            return ResponseDto.failure(ErrorMapping.get("149010"));
        }
        List<String> skuIdList = goodsInfoVo.getSkuIdList();
        List<GoblinGoodsSpecDto> skuSpecList = storeMgtGoodsEditSkuParam.getSkuSpecList();
        for (String skuId : skuIdList) {// 比对所有SKU规格信息
            GoblinGoodsSkuInfoVo mgtGoodsSkuInfoVo = goblinRedisUtils.getMgtGoodsSkuInfoVo(skuId);
            Map<String, String> skuSpecMap = mgtGoodsSkuInfoVo.getSkuSpecList().stream()
                    .collect(Collectors.toMap(GoblinGoodsSpecDto::getSpecName, GoblinGoodsSpecDto::getSpecVname, (k1, k2) -> k2));
            boolean diffSpecFlg = false;
            for (GoblinGoodsSpecDto goodsSpecDto : skuSpecList) {
                if (!skuSpecMap.get(goodsSpecDto.getSpecName()).equals(goodsSpecDto.getSpecVname())) {
                    diffSpecFlg = true;// 多个规格中只要存在一个不一致，即可跳过，标记为允许添加
                    break;
                }
            }
            if (!diffSpecFlg) {
                log.warn("商品管理:商品编辑:SKU添加:重复的规格信息[skuSpecMap={},paramSkuSpecMap={}]", skuSpecMap, JsonUtils.toJson(skuSpecList));
                return ResponseDto.failure(ErrorMapping.get("149014"));
            }
        }
        List<GoblinGoodsSpecVo> specVoList = goodsInfoVo.getSpecVoList();
        if (CollectionUtils.isEmpty(specVoList)) {
            if (null == specVoList) specVoList = ObjectUtil.getGoblinGoodsSpecVoArrayList();
            int size = skuSpecList.size();
            for (int i = 0; i < size; i++) {
                GoblinGoodsSpecDto specDto = skuSpecList.get(i);
                ArrayList<GoblinGoodsSpecValueVo> specValueVoList = ObjectUtil.getGoblinGoodsSpecValueVoArrayList();
                specValueVoList.add(GoblinGoodsSpecValueVo.getNew().setSpecVname(specDto.getSpecVname()).setSpecVsort(i));
                specVoList.add(GoblinGoodsSpecVo.getNew().setSpecName(specDto.getSpecName()).setSpecSort(i).setSpecValues(specValueVoList));
            }
        } else {
            Map<String, String> paramSkuSpecMap = skuSpecList.stream()
                    .collect(Collectors.toMap(GoblinGoodsSpecDto::getSpecName, GoblinGoodsSpecDto::getSpecVname, (k1, k2) -> k2));
            for (GoblinGoodsSpecVo specVo : specVoList) {// 同步添加SPU规格信息
                List<GoblinGoodsSpecValueVo> specValues = specVo.getSpecValues();
                specValues.add(GoblinGoodsSpecValueVo.getNew().setSpecVname(paramSkuSpecMap.get(specVo.getSpecName())).setSpecVsort(specValues.size()));
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("商品管理:商品编辑:SKU添加:[GoblinStoreMgtGoodsEditSkuParam={}]", JsonUtils.toJson(storeMgtGoodsEditSkuParam));
        }
        goblinstoreMgtGoodsService.goodsEditSkuAdd(storeMgtGoodsEditSkuParam, goodsInfoVo);
        return ResponseDto.success();
    }

    @ApiOperationSupport(order = 9)
    @ApiOperation(value = "商品编辑:SKU删除")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "storeId", value = "店铺ID"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "skuId", value = "单品ID"),
    })
    @PostMapping("edit_sku/del")
    public ResponseDto<Object> editSkuDel(@NotBlank(message = "店铺ID不能为空") String storeId,
                                          @NotBlank(message = "单品ID不能为空") String skuId) {
        if (!goblinRedisUtils.hasStoreId(CurrentUtil.getCurrentUid(), storeId)) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        GoblinGoodsSkuInfoVo mgtGoodsSkuInfoVo = goblinRedisUtils.getMgtGoodsSkuInfoVo(skuId);
        if (null == mgtGoodsSkuInfoVo || !mgtGoodsSkuInfoVo.getStoreId().equals(storeId)) {
            return ResponseDto.failure(ErrorMapping.get("149011"));
        }
        return ResponseDto.success(goblinstoreMgtGoodsService.goodsEditSkuDel(mgtGoodsSkuInfoVo));
    }

    @ApiOperationSupport(order = 10)
    @ApiOperation(value = "商品编辑:SKU批改")
    @ApiImplicitParams({
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "storeId", value = "店铺ID"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "spuId", value = "商品ID"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "batField", value = "批量修改属性"),
            @ApiImplicitParam(type = "form", required = true, dataType = "String", name = "batFieldVal", value = "批量修改属性值"),
    })
    @PostMapping("edit_sku/bat")
    public ResponseDto<Object> editSkuBat(@NotBlank(message = "店铺ID不能为空") String storeId,
                                          @NotBlank(message = "商品ID不能为空") String spuId,
                                          @NotBlank(message = "批量修改属性不能为空") String batField,
                                          @NotBlank(message = "批量修改属性值不能为空") String batFieldVal) {
        if (!goblinRedisUtils.hasStoreId(CurrentUtil.getCurrentUid(), storeId)) {
            return ResponseDto.failure(ErrorMapping.get("149002"));
        }
        return ResponseDto.success(goblinstoreMgtGoodsService.goodsEditSkuBat(spuId, batField, batFieldVal));
    }
}
