package com.liquidnet.service.account.wallet.service.processor;

import com.liquidnet.commons.lang.util.JsonUtils;
import com.liquidnet.service.account.common.ErrorConstants;
import com.liquidnet.service.account.common.FinConstants;
import com.liquidnet.service.account.wallet.dto.WalletDashboardResult;
import com.liquidnet.service.account.wallet.dto.base.WalletContextParam;
import com.liquidnet.service.account.wallet.entity.FinWallet;
import com.liquidnet.service.account.wallet.service.FinBizTransService;
import com.liquidnet.service.account.wallet.service.FinWalletService;
import com.liquidnet.service.account.wallet.service.WalletProcessorService;
import com.liquidnet.service.account.wallet.vo.WalletDashboardInfoVo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * WalletDashboardProcessor.class
 *
 * @author zhanggb
 * Created by IntelliJ IDEA at 2020/11/9
 */
@Service
public class WalletDashboardProcessor extends WalletProcessorService {
    private static final Logger log = LoggerFactory.getLogger(WalletDashboardProcessor.class);
    @Autowired
    private FinWalletService walletService;
    @Autowired
    private FinBizTransService bizTransService;

    @Override
    public WalletDashboardResult checkInputParams(WalletContextParam context) {
        if (StringUtils.isEmpty(context.getWalletNo())) {
            return new WalletDashboardResult(ErrorConstants.PARAMS_VALUE_ERROR_CODE, "Invalid walletNo");
        }
        if (null == context.getDashboardPeriod()) {
            return new WalletDashboardResult(ErrorConstants.PARAMS_VALUE_ERROR_CODE, "Invalid dashboardPeriod");
        }
        if (null == context.getDashboardChart()) {
            return new WalletDashboardResult(ErrorConstants.PARAMS_VALUE_ERROR_CODE, "Invalid dashboardChart");
        }
        return new WalletDashboardResult();
    }

    @Override
    public WalletDashboardResult checkBussinessLogic(WalletContextParam context) {
        return new WalletDashboardResult();
    }

    @Override
    public WalletDashboardResult doBussiness(WalletContextParam context) {
        FinWallet wallet = walletService.query(context.getWalletNo());
        if (null == wallet) {
            return new WalletDashboardResult(ErrorConstants.PARAMS_VALUE_ERROR_CODE, "Not found wallet");
        }
        FinConstants.Dashboard.Chart dashboardChart = context.getDashboardChart();
        FinConstants.Dashboard.Period dashboardPeriod = context.getDashboardPeriod();

        LocalDateTime nowTime = LocalDateTime.now(), beginTime = null, endTime = null;
        endTime = nowTime.plusDays(1).with(LocalTime.MIN);
        beginTime = endTime.plusHours(-dashboardPeriod.getVal());

        log.info("begin:{}", beginTime);
        log.info("end:{}", endTime);

        // query transactions within a specified time
        Map<String, Object> parameter = new HashMap<>();
        parameter.put("groupingDateFormat", dashboardPeriod.getFmt());
        parameter.put("walletNoEqualTo", context.getWalletNo());
        parameter.put("tracingTimeGreaterThanOrEqualTo", beginTime);
        parameter.put("tracingTimeLessThan", endTime);
//        parameter.put("transStatusEqualTo", FinConstants.TxStatus.SUCCESS.getVal());
        log.info("Wallet dashboard transaction balance query parameter:{}", JsonUtils.toJson(parameter));

        Map<String, List<WalletDashboardInfoVo>> voMap = new HashMap<>();
        switch (dashboardChart) {
            case All:
                voMap.put(FinConstants.Dashboard.Chart.Wallet_Balance.name(), bizTransService.queryTransBalanceStats(parameter));
                break;
            default :
                voMap.put(dashboardChart.name(), bizTransService.queryTransBalanceStats(parameter));
                break;
        }

        return new WalletDashboardResult(voMap);
    }
}
